from typing import Union

import numpy as np
import tensorflow as tf

from hailo_model_optimization.acceleras.atomic_ops.activation_op import ActivationOp
from hailo_model_optimization.acceleras.atomic_ops.bias_add_op import AddBiasOp
from hailo_model_optimization.acceleras.atomic_ops.element_wise_mult_on_mac_op import ElementwiseMultOnMacOp
from hailo_model_optimization.acceleras.atomic_ops.passthru_op import PassthruOp
from hailo_model_optimization.acceleras.atomic_ops.reduce_sum_wo_kernel_op import ReduceSumWOKernelOp
from hailo_model_optimization.acceleras.encoding.encoding_data import EncodingType
from hailo_model_optimization.acceleras.hailo_layers.base_hailo_layer import BaseHailoLayer
from hailo_model_optimization.acceleras.hailo_layers.layer_flow import LayerFlow
from hailo_model_optimization.acceleras.model_optimization_config.mo_config_layer import (
    LayerPrecisionConfig,
    LayerWeightsClippingConfig,
)
from hailo_model_optimization.acceleras.utils.acceleras_definitions import (
    ActivationType,
    BiasMode,
    DataPath,
    EquivClassification,
    EWMultType,
    LayerHandlerType,
    OptimizationTarget,
    PrecisionMode,
)
from hailo_model_optimization.acceleras.utils.acceleras_exceptions import (
    AccelerasNumerizationError,
    InvalidInputShape,
)
from hailo_model_optimization.acceleras.utils.hn_npz_utils import LayerParams
from hailo_model_optimization.acceleras.utils.opt_utils import (
    get_decomposition_count_by_bias_mode,
    get_kernel_bits_and_sign_by_precision_mode,
)


class HailoElementwiseMultOnMac(BaseHailoLayer):
    SUPPORTED_PRECISION_MODE = {
        PrecisionMode.a16_w16,
        PrecisionMode.a16_w16_a16,
    }
    SUPPORTED_BIAS_MODE = {
        BiasMode.double_scale_initialization,  # maybe we should remove it
    }
    SUPPORTED_QUANTIZATION_GROUPS = False

    def __init__(
        self,
        name: str,
        activation: Union[str, callable, ActivationType] = ActivationType.LINEAR,
        input_repeats=None,
        reduce_sum_groups=None,
        logger=None,
        **kwargs,
    ):
        self.ew_mult_op = ElementwiseMultOnMacOp(f"{name}/elementwise_mult_op", input_repeats, logger=logger)
        self.bias_add_op = AddBiasOp.get_passthru_bias(f"{name}/bias_add_op", logger=logger)
        if reduce_sum_groups is None:
            self.optional_reduce_sum_op = PassthruOp(f"{name}/optional_reduce_sum_op", logger=logger)
        else:
            self.optional_reduce_sum_op = ReduceSumWOKernelOp(
                f"{name}/optional_reduce_sum_op", groups=reduce_sum_groups, logger=logger
            )
        self.act_op = ActivationOp(f"{name}/act_op", activation, logger=logger)
        self.output_op = PassthruOp(f"{name}/passthru_op", logger=logger)
        super().__init__(name=name, logger=logger, **kwargs)
        self.output_scale_scalar_dof = 1
        self.encoding_const = False
        self.input_repeats = input_repeats if input_repeats else [[1, 1, 1], [1, 1, 1]]
        self._forced_output_scale_scalar_dof = None  # degree of freedom
        self._shift_delta = 0

    @property
    def pre_acc_shift(self):
        return self.ew_mult_op._multiplier_shift

    @property
    def forced_output_scale_scalar_dof(self):
        return self._forced_output_scale_scalar_dof

    @forced_output_scale_scalar_dof.setter
    def forced_output_scale_scalar_dof(self, forced_output_scale_scalar_dof):
        self._forced_output_scale_scalar_dof = forced_output_scale_scalar_dof

    def _build_flow(self) -> LayerFlow:
        layer_flow = LayerFlow()
        in1 = layer_flow.add_input()
        in2 = layer_flow.add_input()
        out1 = layer_flow.add_output()

        layer_flow.add_node(self.ew_mult_op)
        layer_flow.add_node(self.bias_add_op)
        layer_flow.add_node(self.optional_reduce_sum_op)
        layer_flow.add_node(self.act_op)
        layer_flow.add_node(self.output_op)

        layer_flow.add_edge(in1, self.ew_mult_op, DataPath.LAYER_IN, input_index=0)
        # TODO: maybe rename signed_layer_in to data_layer_in?
        layer_flow.add_edge(in2, self.ew_mult_op, DataPath.LAYER_IN_WEIGHTS, input_index=1)
        layer_flow.add_edge(self.ew_mult_op, self.optional_reduce_sum_op, DataPath.ACCUMULATOR)
        layer_flow.add_edge(self.optional_reduce_sum_op, self.bias_add_op, DataPath.ACCUMULATOR)
        layer_flow.add_edge(self.bias_add_op, self.act_op, DataPath.ACCUMULATOR)
        layer_flow.add_edge(self.act_op, self.output_op, DataPath.LAYER_OUT)
        layer_flow.add_edge(self.output_op, out1, DataPath.LAYER_OUT)

        return layer_flow

    def enforce_internal_encoding(self, training=False, **kwargs):
        """
        Calls infer_encodings for underlying atomic ops.
        """
        # TODO: I don't like that the scales and zp are external properties, and I have to set them explicitly.
        #       Which affects the infer_encoding implicitly
        self._enforce_output_encoding()
        self.ew_mult_op.enforce_encoding()
        self.optional_reduce_sum_op.input_zero_point = self.ew_mult_op.output_zero_point
        self.optional_reduce_sum_op.input_scale = self.ew_mult_op.output_scale
        self.optional_reduce_sum_op.enforce_encoding()

        self.bias_add_op.input_scale = self.optional_reduce_sum_op.output_scale
        self.bias_add_op.input_zero_point = self.optional_reduce_sum_op.output_zero_point
        self.bias_add_op.output_scale = self.optional_reduce_sum_op.output_scale
        self.bias_add_op.enforce_encoding()

        self.act_op.input_zero_points = [self.bias_add_op.output_zero_point]
        self.act_op.input_scales = [self.bias_add_op.output_scale]
        self.act_op.enforce_encoding(training=training)

    def fast_enforce_internal_encoding(self, **kwargs):
        pass

    def _enforce_output_encoding(self):
        self.output_op.backward_encoding()
        self.act_op.output_scale = self.output_op.input_scales[0]
        self.act_op.output_zero_point = self.output_op.input_zero_points[0]

    def import_weights(self, layer_params: LayerParams):
        """
        Args:
            layer_params: layer's params from the npz

        """
        # TODO: Do we want to load kernel and bias values? (instead of the auto-generated values)
        self.act_op.import_weights(layer_params)

    def _export_weights(self):
        weights = dict()
        weights["bias"] = self.bias_add_op.export_weights()
        weights.update(self.act_op.export_weights())
        return weights

    def _export_layer_metadata(self):
        export_vals = super()._export_layer_metadata()
        if self.forced_output_scale_scalar_dof is not None:
            export_vals["forced_output_scale_scalar_dof"] = self.forced_output_scale_scalar_dof
        return export_vals

    def _import_layer_metadata(self, npz):
        self.forced_output_scale_scalar_dof = npz.get("forced_output_scale_scalar_dof", None)
        return super()._import_layer_metadata(npz)

    @classmethod
    def get_default_params(cls):
        # TODO: this is temporary solution until we have pydantic scheme
        defaults = {
            "activation": "linear",
            "input_repeats": [[1, 1, 1], [1, 1, 1]],  # Do we need the input repeats here?
        }
        return dict(defaults)

    @classmethod
    def from_hn(cls, lname, hn_element, logger=None):
        params = cls.get_default_params()
        params.update(hn_element.get("params", dict()))
        reduce_sum_groups = params.get("reduce_sum_groups", None)
        input_channels = max(shape[-1] for shape in hn_element.get("input_shapes"))

        if reduce_sum_groups == input_channels:
            reduce_sum_groups = None

        if reduce_sum_groups is not None and input_channels // reduce_sum_groups != input_channels / reduce_sum_groups:
            raise ValueError(
                f"Number of groups ({reduce_sum_groups}) doesn't devide evenly the number of channels ({input_channels})"
            )
        layer = cls(
            name=lname,
            activation=params["activation"],
            input_repeats=params["input_repeats"],
            reduce_sum_groups=reduce_sum_groups,
            logger=logger,
        )
        layer.finalize_from_hn(hn_element)
        return layer

    def to_hn(self, out_degree=None):
        hn = super().to_hn(out_degree)
        hn.setdefault("params", {})

        hn["params"].update(
            {
                "activation": self.act_op.act_name.value,
                "ew_mult_type": EWMultType.on_mac.value,
                "is_softmax_mask": False,
                "input_repeats": self.input_repeats,
            }
        )
        return hn

    def create_hw_params(
        self, weights_clipping: LayerWeightsClippingConfig, optimization_target: OptimizationTarget, hw_shifts=None
    ):
        self._enforce_output_encoding()
        # pre_acc_shift = hw_shifts[0] if hw_shifts is not None else hw_shifts
        # TODO: propagate the pre_acc_shift to the ew_mult_op
        self.ew_mult_op.create_hw_params(force_shift=0)
        self.enforce_internal_encoding()

        self.bias_add_op.pre_acc_shift = self.ew_mult_op._multiplier_shift
        self.bias_add_op.create_hw_params()
        self.enforce_internal_encoding()

        ratio = self.act_op.get_input_scale_candidate() / self.optional_reduce_sum_op.output_scale
        desired_ratio = 0.9

        if np.any(ratio > desired_ratio):
            shift_delta = np.log2(np.max(ratio / desired_ratio))
        else:
            shift_delta = 0
        self._shift_delta = shift_delta

        self.act_op.create_hw_params(self.optional_reduce_sum_op.output_scale, optimization_target, nudging=False)
        self.output_op.create_hw_params()

    @property
    def shift_delta(self):
        return self._shift_delta

    def create_quant_element_custom_behavior(
        self,
        precision_config: LayerPrecisionConfig,
        optimization_target: OptimizationTarget,
    ):
        kernel_bits, signed = get_kernel_bits_and_sign_by_precision_mode(precision_config.precision_mode)
        num_decomposition = get_decomposition_count_by_bias_mode(precision_config.bias_mode)
        self.bias_add_op.create_weight_quant_element(kernel_bits, signed, num_decomposition)

        # TODO: Do we need signed layer in data_path?
        self.act_op.create_weight_quant_element(optimization_target)

    def get_equalization_handler_type(self, predecessor_index=None):
        if self.has_reduce_sum():
            handler_type = LayerHandlerType.unsupported
        else:
            handler_type = LayerHandlerType.transparent
        return EquivClassification(handler_type, is_source=False)

    def _force_output_scale(self):
        # set output scale to be the product of the (self.input_scales[0] * self.input_scales[1])*forced_output_scale_scalar_dof
        if self.forced_output_scale_scalar_dof is not None and self.output_scale.shape != 0:
            self.set_output_scale(self.input_scales[0] * self.input_scales[1] * self.forced_output_scale_scalar_dof, 0)

    def has_reduce_sum(self):
        #! TODO: remove it once the op is removed
        return isinstance(self.optional_reduce_sum_op, ReduceSumWOKernelOp)

    def _get_post_mult_scale(self):
        """
        utility for dof scale calculation which will be overriden in subclasses
        """
        return tf.math.multiply(*self._get_repeated_input_scales())

    def _create_out_in_scale_ratio(self):
        """
        create the output_scale_scalar_dof
        """
        self._enforce_output_encoding()
        output_scale = self._get_output_scale_for_scalar_dof()
        if self.has_reduce_sum():
            output_scale_avg = np.mean(output_scale)
            input_scale_tmp = self._get_post_mult_scale()
            input_scale = np.mean(input_scale_tmp)
            if not np.allclose(input_scale, input_scale_tmp, atol=0) or not np.allclose(
                output_scale, output_scale_avg, atol=0
            ):
                raise AccelerasNumerizationError(f"unexpected scale values in {self.full_name}")
            output_scale = output_scale_avg
        else:
            input_scale = self._get_post_mult_scale()

        _out_in_scale_ratio = output_scale / input_scale
        eps = 1e-6
        if _out_in_scale_ratio.shape != ():
            if eps < np.max(np.abs(_out_in_scale_ratio - _out_in_scale_ratio[0]) / _out_in_scale_ratio[0]):
                # Possible fail case: coming from concat, so input scale is scalar while output is vector..
                raise AccelerasNumerizationError(
                    f"output_scale - input_scale ratio of {self.full_name} should be a scalar"
                )
            # create attribute to be used in scales-training context should it come
            self.output_scale_scalar_dof = _out_in_scale_ratio[0]
        else:
            self.output_scale_scalar_dof = _out_in_scale_ratio

    def _get_output_scale_for_scalar_dof(self):
        return self.output_scale

    def enforce_io_encoding(self, training=False, **kwargs):
        #! wip
        if self.has_reduce_sum():
            self.output_op.output_scale = (
                tf.reduce_mean(tf.math.multiply(*self._get_repeated_input_scales()))
                * self.output_scale_scalar_dof
                * tf.ones_like(self.output_op.output_scale)
            )
        else:
            self.output_op.output_scale = (
                tf.math.multiply(*self._get_repeated_input_scales()) * self.output_scale_scalar_dof
            )

    def _get_repeated_input_scales(self):
        # modulo logic to handle both feature multipler and ew_mult
        index_a = 0 % self.num_inputs
        index_b = 1 % self.num_inputs
        input_scale_a = tf.repeat(self.input_scales[index_a], self.input_repeats[index_a][-1], axis=-1)
        input_scale_b = tf.repeat(self.input_scales[index_b], self.input_repeats[index_b][-1], axis=-1)
        return input_scale_a, input_scale_b

    def update_scale_scalar_dof(self, shift):
        output_factor = 2**shift
        self.output_scale_scalar_dof /= output_factor

    def define_encodings(self, flow):
        super().define_encodings(flow)
        flow.add_encoding(f"{self.full_name}/output_scale_scalar_dof:0", EncodingType.Scale, scalar=False, shape=())
        flow.nodes[f"{self.act_op.full_name}/output_factor_by_group:0"]["encoding"].scalar = True

    def define_constraints(self, enc):
        # TODO: Not sure if it is correct
        super().define_constraints(enc)

        if self.ew_mult_op.encoding_const and self.output_op.encoding_const:
            enc.identity(f"{self.full_name}/output_scale_scalar_dof:0", self.output_scale_scalar_dof)
        else:
            enc.mul(
                enc.dummy(0), f"{self.ew_mult_op.full_name}/input_scale:0", f"{self.ew_mult_op.full_name}/input_scale:1"
            )
            enc.div(enc.dummy(1), f"{self.output_op.full_name}/output_scale:0", enc.dummy(0))
            enc.callback(
                f"{self.full_name}/output_scale_scalar_dof:0",
                enc.dummy(1),
                lambda x: x[0] if x.shape != () else x,
                outs_scalar=True,
                outs_shape=(),
            )

    def update_encoding(self, encodings):
        super().update_encoding(encodings)
        self.output_scale_scalar_dof = encodings[f"{self.full_name}/output_scale_scalar_dof:0"]

    def verify_layer_inputs_shape(self, input_shapes):
        if len(input_shapes) == 2:
            # factorizes the input shapes according to the input repeats
            factored_shapes = []
            for input_shape, input_repeat in zip(input_shapes, self.input_repeats):
                factored_shapes += [[dim * ratio for dim, ratio in zip(input_shape[1:], input_repeat)]]
            if not all(x == y for x, y in zip(*factored_shapes)):
                raise InvalidInputShape(
                    f"Input shapes {input_shapes} doesn't match each other in {self.full_name}", self.full_name
                )

    def _is_precision_config_supported(self, precision_mode, bias_mode, arch):
        return True

    @classmethod
    def get_default_precision_mode(cls):
        return PrecisionMode.a16_w16

    @classmethod
    def get_default_bias_mode(cls):
        return BiasMode.double_scale_initialization
