import json
from abc import ABC, abstractmethod
from enum import Enum, EnumMeta
from typing import Dict, NamedTuple

from pydantic.v1 import BaseModel, Field

SKIP_DEFAULT = True


class CommandMeta(NamedTuple):
    line: int
    command: str
    is_glob: bool


def _value_to_str(values):
    """
    Converts the data to printable 'flat' data.
    Currently, the only purpose is to convert list from `['a', 'b', 'c']` to `[a, b, c]` to match the alls expectation
    Args:
        values: values to convert

    Returns
        given value as str (for the alls str)

    """
    # TODO: consider replacing the raw dict normal to dict and convert the enum here.
    if isinstance(values, (list, tuple)):
        as_str = "[" + ", ".join(f"{v}" for v in values) + "]"
    elif isinstance(values, EnumMeta):
        if hasattr(values, "internals"):
            as_str = "{" + ", ".join(f"{v.value}" for v in values if v not in values.internals()) + "}"
        else:
            as_str = "{" + ", ".join(f"{v.value}" for v in values) + "}"
    elif isinstance(values, Enum):
        as_str = values.value
    else:
        as_str = str(values)
    return as_str


class BaseConfigBaseModel(BaseModel, ABC):
    meta: Dict[str, CommandMeta] = Field(None)

    class Config:
        extra = "forbid"

    @classmethod
    @abstractmethod
    def get_default(cls):
        """
        Default configuration of the feature
        Returns:
            initialized class with default values
        """

    @classmethod
    @abstractmethod
    def get_command(cls):
        """
        Alls command name, for creating alls command
        Returns:

        """

    @classmethod
    @abstractmethod
    def get_feature(cls):
        """
        Alls feature name, for creating alls command, may be None if command has no feature
        Returns:

        """

    @classmethod
    def keys(cls):
        """
        Keys for the current config's internal-config
        Returns:
            set of keys
        """
        return cls.__fields__.keys() - {"meta"}

    @classmethod
    def internal_keys(cls) -> set:
        """
        Internal keys of configuration.
        internal may be internal config or internal data (e.g. `meta`)
        These keys won't be printed when creating raw dict

        Returns
            set of internal keys of config

        """
        return {"meta"} | cls._internal_keys()

    @classmethod
    @abstractmethod
    def _internal_keys(cls) -> set:
        """
        More internal config of feature, read BaseConfigBaseModel's internal_keys for more info
        Returns:
            set with internal_keys keys
        """

    def raw_dict(self, exclude_defaults=False, exclude_internal=True):
        """
        raw dict is creating by exporting and loading to json. it converts the data from enum to the enum values.
        This way, the data can be easily converted to string command

        Args:
            exclude_defaults: whether the defaults should be excluded for the dict or not
            exclude_internal: whether the internal keys should be excluded for the dict or not

        Returns:

        """
        exclude = self.internal_keys() if exclude_internal else {"meta"}
        as_dict = json.loads(self.json(exclude_defaults=exclude_defaults, exclude=exclude, exclude_none=True))

        # Always include internal_keys if they are none-default values
        include = self.internal_keys() - {"meta"}
        internal_keys_as_dict = json.loads(self.json(exclude_defaults=True, include=include, exclude_none=True))
        as_dict.update(internal_keys_as_dict)

        # Remove empty lists from dict, because the alls format doesn't support empty list
        def is_empty_list(v):
            return isinstance(v, (list, tuple)) and len(v) == 0

        def as_hashable(v):
            # When exporting layer configuration, layers are grouped based on config, so the dict need to be hashable
            # The tuple is converted back to str in _value_to_str when we export the command
            if isinstance(v, list):
                v = tuple(v)
            return v

        return {k: as_hashable(v) for k, v in as_dict.items() if not is_empty_list(v)}

    def has_meta_info(self, key):
        """
        Check if config has meta info for given key.
        """
        return self.meta is not None and key in self.meta and self.meta[key].command

    @classmethod
    def is_internal_feature(cls):
        return False
