class AccelerasException(Exception):
    pass


class AccelerasResourceError(AccelerasException):
    """For resource errors (GPU)"""


class AcclerasDataError(AccelerasException):
    """For dataset errors"""


class AccelerasInferenceError(AccelerasException):
    """For runtime exceptions"""


class AccelerasImplementationError(AccelerasException):
    """For implementation exceptions"""


class AccelerasEncodingError(AccelerasException):
    """For implementation exceptions"""


class AccelerasUnsupportedError(AccelerasException):
    """For unsupported exceptions"""


class InferenceError(AccelerasException):
    """For inference exceptions"""


class AccelerasInitializationError(AccelerasException):
    """For operation build and initialization errors"""


class AccelerasNumerizationError(AccelerasException):
    """
    Errors that derive from the numerization process should trigger this exception
    """


class InconsistentEncodingError(AccelerasException):
    pass


class ActivationClippingError(AccelerasException):
    pass


class AccelerasDecompositionError(AccelerasNumerizationError):
    def __init__(self, target_value, max_a, max_b) -> None:
        self.target_value = target_value
        self.max_a = max_a
        self.max_b = max_b
        super().__init__(f"Target value {target_value} cannot be decomposed with max_a: {max_a}, max_b: {max_b}")


class AccelerasElementwiseDecompositionError(AccelerasNumerizationError):
    def __init__(self, lname, decomp_error: AccelerasDecompositionError):
        new_feed_repeat = max(decomp_error.target_value // decomp_error.max_a)
        msg = (
            f"{lname}: Decomposition error when decomposting value {decomp_error.target_value}, "
            f"max_int={decomp_error.max_a}, feed_repeat={decomp_error.max_b}. "
            f"Consider increasing the max_elementwise_feed_repeat of the current layer to {new_feed_repeat}"
        )
        super().__init__(msg)


class AccelerasBiaseDecompositionError(AccelerasNumerizationError):
    def __init__(self, lname, decomp_error: AccelerasDecompositionError):
        new_feed_repeat = max(decomp_error.target_value // decomp_error.max_a)
        msg = (
            f"{lname}: Decomposition error when decomposting value {decomp_error.target_value}, "
            f"max_int={decomp_error.max_a}, feed_repeat={decomp_error.max_b}. "
            f"Consider increasing the max_bias_feed_repeat of the current layer to {new_feed_repeat}"
        )
        super().__init__(msg)


class AccelerasConfigurationError(AccelerasException):
    """
    Errors that derive from the numerization process should trigger this exception
    """


class AccelerasImportParamConfigMismatch(AccelerasConfigurationError):
    def __init__(self, field, current_value, import_value, op_name):
        msg = (
            f"{field} in imported params doesn't match config in op. \n"
            f"op: {op_name}, current {field}: {current_value}, "
            f"imported {field}: {import_value}"
        )
        super().__init__(msg)


class AccelerasPrematureQuantOperation(AccelerasException):
    """
    Errors that derive from the numerization process should trigger this exception
    """

    def __init__(self, func, op_name):
        msg = f"Tried calling {func} before loading appropriate config. op: {op_name}"
        super().__init__(msg)


class AccelerasValueError(AccelerasException, ValueError):
    pass


class BiasCorrectionError(AccelerasException):
    pass


class BadInputsShape(AccelerasInferenceError):
    def __init__(self, layer_name, input_shape, data_shape):
        super(BadInputsShape, self).__init__(
            f"Data shape {data_shape} for layer {layer_name} doesn't match network's input shape {input_shape}",
        )


class AccelerasNegativeSlopesError(AccelerasNumerizationError):
    def __init__(self, exponet, supported_range, lname):
        self.exponenet = exponet
        self.supported_range = supported_range
        super().__init__(f"the exponent {exponet} is not in range {supported_range} for layer {lname}.")


class NegativeSlopeExponentNonFixable(AccelerasNumerizationError):
    def __init__(self, fix_shift, output_bits, lname):
        super().__init__(
            f"Quantization failed in layer {lname} due to unsupported required slope. "
            f"Desired shift is {fix_shift}, but op has only {output_bits} data bits. "
            f"This error raises when the data or weight range are not balanced. Mostly happens when using random calibration-set/weights, "
            f"the calibration-set is not normalized properly or batch-normalization was not used during training.",
        )


class NegativeSlopeExponentReapeat(AccelerasNumerizationError):
    def __init__(self, shifts, lname):
        super().__init__(f"Repeating negative slope exponent in layer {lname}. shifts: {shifts}")


class ConfigurationError(AccelerasException):
    def __init__(self, msg, loc):
        super().__init__(msg)
        self.loc = loc


class LayerEquivError(AccelerasException):
    pass


class MissingPandocException(AccelerasException):
    pass


class AdaRoundError(AccelerasException):
    """Generic adaround related exception"""


class EqualizationError(AccelerasException):
    """Generic equalization related exception"""


class EqualizationFactorsError(EqualizationError):
    pass


class MatchingAlgoError(AccelerasException):
    """Generic MatchingAlgo related exception"""


class AccelerasImportError(AccelerasException):
    """Import error for params in acceleras"""


class DatasetException(Exception):
    pass


class SubprocessTracebackFailure(Exception):
    def __init__(self, exception_dict):
        self.inner_error = exception_dict["exception"]
        self.traceback = exception_dict["traceback"]

    def __str__(self):
        return (
            f"Subprocess failed with exception: {str(self.inner_error)}\n\n Exception traceback: \n\n {self.traceback}"
        )


class SubprocessUnexpectedFailure(Exception):
    pass


class SubprocessFailure(Exception):
    pass


class ResolutionReductionError(AccelerasException):
    pass


class InvalidInputShape(AccelerasException):
    def __init__(self, msg, lname) -> None:
        super().__init__(msg)
        self.lname = lname


class EWAddFusingError(AccelerasException):
    pass


class AlgoErrorHint(AccelerasException):
    """
    This class inherits from the AccelerasException class and is used to handle exceptions
    related a generic algorithm processing.
    It provides hints about what might have caused the error and potential solutions.

    Attributes:
        general_info (str): string that gives Description of the problem.
        solutions (list): a list of strings that gives potential solutions.
        explenations (list): a list of strings that gives explenations for the solutions.

    """

    def __init__(self, general_info, solutions, explenations) -> None:
        self.general_info = general_info
        self.solutions = solutions
        self.explenations = explenations
        msg = [self.general_info]
        for i in range(len(self.solutions)):
            msg.append(f"{i + 1}. Use the following {self.solutions[i]} so that {self.explenations[i]}")

        super().__init__("\n".join(msg))


class SplitEWMultByBitSignificanceError(AccelerasException):
    pass
