__all__ = [
    "default_logger",
    "set_default_logger_level",
    "create_custom_logger",
    "SDK_CLIENT_LOG_EN_ENV_VAR",
    "LOG_LEVEL_ENV_VAR",
    "DEFAULT_LEVEL",
    "DEFAULT_LOG_DIR",
    "DFC_FOLDER_PATH",
    "SDK_CLIENT_LOG_DIR_ENV_VAR",
    "has_append_permission",
]

import logging
import os
import sys

import verboselogs

SDK_CLIENT_LOG_EN_ENV_VAR = "HAILO_CLIENT_LOGS_ENABLED"
SDK_CLIENT_LOG_DIR_ENV_VAR = "HAILO_SDK_LOG_DIR"
LOG_LEVEL_ENV_VAR = "LOGLEVEL"
ACCELERAS_LOG_NAME = "acceleras"
DEFAULT_LOG_DIR = "."
DFC_FOLDER = "etc/hailo/dfc/"
DFC_FOLDER_PATH = os.path.join(os.environ.get("VIRTUAL_ENV", f"{os.environ['HOME']}/.hailo"), DFC_FOLDER)
ACCELERAS_LOG_FILE_PATH = os.path.join(DFC_FOLDER_PATH, f"{ACCELERAS_LOG_NAME}.log")
DEFAULT_LEVEL = "INFO"


class ConsoleFormatter(logging.Formatter):
    """
    Logging Formatter for the console.
    """

    COLOR_PREFIX = "\x1b["
    COLOR_SUFFIX = "\x1b[0m"
    COLORS = {
        logging.DEBUG: "36m",  # blue
        verboselogs.VERBOSE: "36m",  # blue
        logging.INFO: "32m",  # green
        logging.WARNING: "33;1m",  # bold yellow
        logging.ERROR: "31;1m",  # bold red
        logging.CRITICAL: "41;1m",  # bold white on red
    }

    def format(self, record):
        level_name = record.levelname
        level_no = record.levelno
        message = "%(message)s"
        if sys.stdout.isatty():
            level_fmt = f"[{self.COLOR_PREFIX}{self.COLORS[level_no]}{level_name.lower()}{self.COLOR_SUFFIX}]"
        else:
            level_fmt = f"[{level_name.lower()}]"
        formatter = logging.Formatter(f"{level_fmt} {message}")
        return formatter.format(record)


def has_append_permission(path):
    if os.path.exists(path):
        return os.access(path, os.W_OK)

    return os.access(os.path.dirname(os.path.abspath(path)), os.W_OK)


def _create_logger(level=None, logger_path=None, fmt=None, console=True):
    """
    Creates a logging object and returns it
    """
    logger_path = logger_path if logger_path else os.path.join(DEFAULT_LOG_DIR, ACCELERAS_LOG_NAME + ".log")
    logger_name, _ = os.path.splitext(os.path.basename(logger_path))
    logger = verboselogs.VerboseLogger(logger_name)
    logger.propagate = False

    console_level = DEFAULT_LEVEL
    if level is not None:
        file_level = level
    else:
        file_level = DEFAULT_LEVEL

    logger.setLevel(file_level)

    if fmt is None:
        fmt = "%(asctime)s - %(levelname)s - %(filename)s:%(lineno)s - %(message)s"
    formatter = logging.Formatter(fmt)

    if console:
        console_fh = logging.StreamHandler(sys.stdout)
        console_fh.setFormatter(ConsoleFormatter())
        console_fh.set_name("stream_handler")
        console_fh.setLevel(console_level)
        logger.addHandler(console_fh)

    enable_log_file = os.environ.get(SDK_CLIENT_LOG_EN_ENV_VAR)

    if not enable_log_file or enable_log_file.lower() != "false":
        # creates directories (if needed) for the log file that will be stored in hailo/dfc folder
        os.makedirs(DFC_FOLDER_PATH, exist_ok=True)
        logger_folder = os.path.dirname(logger_path)
        if logger_folder:
            os.makedirs(logger_folder, exist_ok=True)
        # create the logging file handlers
        for logger_file_path in [logger_path, ACCELERAS_LOG_FILE_PATH]:
            if has_append_permission(logger_file_path):
                fh = logging.FileHandler(logger_file_path, delay=True)
                fh.setFormatter(formatter)
                logger.addHandler(fh)
            elif logger_file_path != ACCELERAS_LOG_FILE_PATH:
                logger.warning(f"Unable to create log file in {logger_file_path} (Permission denied)")

    return logger


_g_logger = None


def default_logger():
    global _g_logger
    if _g_logger is None:
        sdk_client_log_dir = os.environ.get(SDK_CLIENT_LOG_DIR_ENV_VAR)
        logger_path = os.path.join(sdk_client_log_dir, ACCELERAS_LOG_NAME + ".log") if sdk_client_log_dir else None
        log_level = os.environ.get(LOG_LEVEL_ENV_VAR, DEFAULT_LEVEL).upper()
        _g_logger = _create_logger(level=log_level, logger_path=logger_path)
    return _g_logger


def create_custom_logger(log_path, fmt=None, console=False):
    """Returns a logger to the specified path, creating it if needed."""
    return _create_logger(logger_path=log_path, fmt=fmt, console=console)


def set_default_logger_level(level):
    """
    Set log level of the default log.
    Should only be called from initialization code to avoid log level collisions.
    """
    if level == "ERROR":
        _g_logger.verbose(f"Logger verbosity has been decreased to {level.lower()}s only ")
    else:
        _g_logger.verbose(f"Logger verbosity is: {level}")
    _g_logger.setLevel(level)
