import subprocess

from hailo_model_optimization.acceleras.utils.logger import default_logger

logger = default_logger()


def select_least_used_gpu(max_memory_utilization=0.05):
    """
    Select the first GPU with memory usage below `max_memory_utilization` ratio.
    Default is 5% usage (safe to assume idle).

    Returns:
        str: GPU index as string, e.g., '0' or '2' (for CUDA_VISIBLE_DEVICES) or None if none found.
    """
    try:
        # Run nvidia-smi to query memory usage and total memory
        result = subprocess.run(
            ["nvidia-smi", "--query-gpu=index,memory.used,memory.total", "--format=csv,noheader,nounits"],
            stdout=subprocess.PIPE,
            text=True,
            check=True,
        )

        gpu_list = result.stdout.strip().split("\n")
        candidates = []

        for line in gpu_list:
            index, used_mem, total_mem = map(int, line.split(","))
            usage_ratio = used_mem / total_mem

            if usage_ratio <= max_memory_utilization:
                candidates.append((index, usage_ratio))

        if candidates:
            selected_gpu = str(sorted(candidates, key=lambda x: x[1])[0][0])
            gpu_selection_msg = f"No GPU chosen, Selected GPU {selected_gpu}"
            logger.debug(gpu_selection_msg)
            logger.info(gpu_selection_msg)
            return selected_gpu

        logger.debug("No GPUs found below usage threshold.")
        return None

    except subprocess.CalledProcessError as e:
        logger.debug(f"nvidia-smi command failed: {e}")
        return None
    except Exception as e:
        logger.debug(f"Unexpected error during GPU selection: {e}")
        return None
