import numpy as np
from scipy.optimize import minimize


def minimize_scale_dof(
    max_vector: np.ndarray,
    min_vector: np.ndarray,
    acc_vector: np.ndarray,
    bins: int,
    start_zp: float = None,
    start_dof: float = 1,
):
    """
    Minimizes alpha subject to given constraints.

    Parameters:
    max_vector : numpy.ndarray
        The 'max' vector in the constraints.
    min_vector : numpy.ndarray
        The 'min' vector in the constraints.
    acc_vector : numpy.ndarray
        The 'acc' acceleration vector.
    b : float
        A constant used in the constraints.

    Returns:
    dict
        A dictionary containing the optimal values for scale_dof and zp,
        the minimum value of scale_dof, and a success message.
    """

    # Objective function, focusing on minimizing scale_dof
    def objective(x):
        scale_dof, _ = x
        return scale_dof**2  # We are minimizing scale_dof directly

    # Constraints
    def constraint1(x):
        scale_dof, zp = x
        return min_vector + scale_dof * zp * acc_vector  # -1 * scale_dof * zp * acc <= min

    def constraint2(x):
        scale_dof, zp = x
        return scale_dof * (bins - zp) * acc_vector - max_vector  # scale_dof * (b - zp) * acc >= max

    def constraint3(x):
        _, zp = x
        return zp

    def constraint4(x):
        _, zp = x
        return bins - zp

    def constraint5(x):
        scale_dof, _ = x
        return scale_dof

    # Initial guess
    if start_zp is None:
        start_zp = bins // 2
    x0 = [start_dof, start_zp]

    # Define constraints in scipy format
    cons = [
        {"type": "ineq", "fun": constraint1},
        {"type": "ineq", "fun": constraint2},
        {"type": "ineq", "fun": constraint3},
        {"type": "ineq", "fun": constraint4},
        {"type": "ineq", "fun": constraint5},
    ]

    # Call minimize
    result = minimize(objective, x0, method="SLSQP", constraints=cons)

    # Return results in a structured format
    if result.success:
        scale_dof, zp = result.x
    else:
        scale_dof, zp = start_dof, start_zp

    return scale_dof, np.round(zp)


def minimize_scale_dof_and_zp(
    max_vector: np.ndarray,
    min_vector: np.ndarray,
    acc_vector: np.ndarray,
    bins: int,
    start_zp: float = None,
    start_dof: float = 1,
):
    """
    SEarch for the smallest dof and best zp values to aling the scales from the
    accumulator to the output limvals.

    Parameters:
    max_vector : numpy.ndarray
        The 'max' vector in the constraints.
    min_vector : numpy.ndarray
        The 'min' vector in the constraints.
    acc_vector : numpy.ndarray
        The 'acc' acceleration vector.
    b : float
        A constant used in the constraints.

    Returns:
    dict
        A dictionary containing the optimal values for scale_dof and zp,
        the minimum value of scale_dof, and a success message.
    """

    # Objective function, focusing on minimizing scale_dof
    def objective(x):
        scale_dof = x[0]
        return scale_dof**2  # We are minimizing scale_dof directly

    # Constraints
    def constraint1(x):
        scale_dof = x[0]
        zp = x[1:]
        return min_vector + scale_dof * zp * acc_vector  # -1 * scale_dof * zp * acc <= min

    def constraint2(x):
        scale_dof = x[0]
        zp = x[1:]
        return scale_dof * (bins - zp) * acc_vector - max_vector  # scale_dof * (b - zp) * acc >= max

    def constraint3(x):
        zp = x[1:]
        return zp

    def constraint4(x):
        zp = x[1:]
        return bins - zp

    def constraint5(x):
        scale_dof = x[0]
        return scale_dof

    # Initial guess
    if start_zp is None:
        start_zp = bins // 2
    x0 = [start_dof, *start_zp]

    # Define constraints in scipy format
    cons = [
        {"type": "ineq", "fun": constraint1},
        {"type": "ineq", "fun": constraint2},
        {"type": "ineq", "fun": constraint3},
        {"type": "ineq", "fun": constraint4},
        {"type": "ineq", "fun": constraint5},
    ]

    # Call minimize
    result = minimize(objective, x0, method="SLSQP", constraints=cons)

    # Return results in a structured format
    if result.success:
        # scale_dof, zp = result.x
        scale_dof = result.x[0]
        zp = result.x[1:]
    else:
        scale_dof, zp = start_dof, start_zp

    return scale_dof, np.round(zp)


def find_the_best_zp(
    xmin: np.array,
    xmax: np.array,
    bins: int,
    start_scales: np.ndarray = None,
    start_zp: float = None,
):
    """
    Minimizes the norm of vector S subject to given constraints.

    Parameters:
    xmin : numpy.ndarray
        The 'xmin' vector in the constraints.
    xmax : numpy.ndarray
        The 'xmax' vector in the constraints.
    b : float
        A constant used in the constraints.

    Returns:
    dict
        A dictionary containing the optimal values for scales and zp,
        the minimum norm of scales, and a success message.
    """

    # Number of elements in scales
    n = len(xmin)

    # Objective function, focusing on minimizing the norm of scales
    def objective(x):
        scales = x[:-1]  # All but last element are scales
        zp = x[-1]
        return np.linalg.norm(scales) + 1e-6 * np.abs(zp)

    # Constraints
    def constraint1(x):
        zp = x[-1]  # Last element is zp
        scales = x[:-1]
        return 1 * zp * scales + xmin  # -zp * scales <= xmin -> zp * scales >= -xmin

    def constraint2(x):
        zp = x[-1]
        scales = x[:-1]
        return scales * bins - zp * scales - xmax  # scales * b - zp * scales >= xmax

    def constraint3(x):
        zp = x[-1]
        return zp

    def constraint4(x):
        zp = x[-1]
        return bins - zp

    # Initial guess
    if start_scales is None:
        rng = np.random.RandomState(seed=42)
        start_scales = rng.uniform(0.9, 1.1, n)
    if start_zp is None:
        start_zp = bins // 2
    x0 = np.append(start_scales, start_zp)

    # Define constraints in scipy format
    cons = [
        {"type": "ineq", "fun": constraint1},
        {"type": "ineq", "fun": constraint2},
        {"type": "ineq", "fun": constraint3},
        {"type": "ineq", "fun": constraint4},
    ]

    # Call minimize
    result = minimize(objective, x0.astype(np.float64), method="SLSQP", constraints=cons)
    if result.success:
        zp = result.x[-1]  # Last element is zp
        scales = result.x[:-1]

    else:
        scales, zp = start_scales, start_zp

    return scales, np.round(zp)
