from typing import Any, Dict, Generic, Iterable, TypeVar

ClassName = TypeVar("ClassName", bound=type)


class Register(Generic[ClassName]):
    def __init__(self, base: ClassName):
        self.base = base
        self._registry: Dict[Any, ClassName] = {}

    def __call__(self, target_class: ClassName) -> ClassName:
        return self.register(target_class)

    def register(self, target_class: ClassName) -> ClassName:
        def add_item(key, value: ClassName) -> ClassName:
            if not callable(value):
                raise Exception(f"Error:{value} must be callable!")
            if key in self._registry:
                raise Exception(f"{key} already exists.")
            self[key] = value
            return value

        return lambda x: add_item(target_class, x)

    def __setitem__(self, key, value: ClassName):
        if not issubclass(value, self.base):
            raise TypeError(f"Class {value.__name__} is not a subclass of {self.base.__name__}")
        self._registry[key] = value

    def __getitem__(self, key) -> ClassName:
        return self._registry[key]

    def __contains__(self, key):
        return key in self._registry

    def __str__(self):
        return str(self._registry)

    def keys(self):
        return self._registry.keys()

    def values(self) -> Iterable[ClassName]:
        return self._registry.values()

    def items(self):
        return self._registry.items()

    def __iter__(self):
        return iter(self._registry)


class ClassRegister(Register):
    def register(self, target: ClassName) -> ClassName:
        name = target.__name__
        if name in self._registry:
            raise Exception(f"{name} already exists.")
        self._registry[target.__name__] = target
        return target
