from copy import deepcopy
from pathlib import PosixPath
from shutil import copy2
from shutil import move as shutil_move
from typing import Callable, Type, TypeVar

from pydantic.v1 import BaseModel, Field

MementoChild = TypeVar("MementoChild", bound="BaseMemento")


class BaseMemento(BaseModel):
    base_path: PosixPath = Field(default_factory=PosixPath)
    alive: bool = True

    class Config:
        extra = "forbid"
        arbitrary_types_allowed = False
        validate_assignment = True
        use_enum_values = True

    def delete(self, *, recursive: bool = False):
        """Delete the memento, optionally deleting nested memento objects."""
        delete_memento(self, recursive=recursive)

    def copy(self, path: PosixPath, *, recursive: bool = False) -> MementoChild:
        """
        Copy the memento to a new location, optionally copying nested memento objects.

        Args:
            path (PosixPath): The target directory path where the memento should be copied.
            recursive (bool): Whether to recursively copy nested memento objects.

        Returns:
            BaseMemento: A new instance of BaseMemento at the new location.
        """
        path = PosixPath(path)
        return copy_memento(self, path, recursive=recursive)

    def move(self, path: PosixPath, *, recursive: bool = False) -> MementoChild:
        """
        Move the memento to a new location, optionally moving nested memento objects.

        Args:
            path (PosixPath): The target directory path where the memento should be moved.
            recursive (bool): Whether to recursively move nested memento objects.

        Returns:
            BaseMemento: The memento instance now located at the new path.
        """
        path = PosixPath(path)
        return move_memento(self, path, recursive=recursive)

    def replace_base_path(self, path: PosixPath, *, recursive=False):
        """
        Replace the base path of all PosixPath attributes in the memento, optionally handling nested structures.
        Args:
            path (PosixPath): The new base path to be set.
            recursive (bool): Whether to recursively apply the new base path to nested memento objects.
        """
        path = PosixPath(path)
        replace_base_path(self, path, recursive=recursive)

    def save(self, path: PosixPath) -> PosixPath:
        """
        Save the memento to a specified directory, including a JSON file with the memento's data.

        Args:
            path (PosixPath): The target directory path where the memento should be saved.
        """
        path = PosixPath(path)
        return save_memento(self, path)

    def load(self, path: PosixPath) -> MementoChild:
        """
        Load a memento object from a specified directory or JSON file.

        Args:
            path (PosixPath): The directory path or JSON file path containing the memento data.

        Returns:
            MementoChild: The loaded memento object.
        """
        path = PosixPath(path)
        return load_memento(self.__class__, path)


def replace_base_path(memento: MementoChild, path: PosixPath, *, recursive: bool = False):
    for field_name in memento.__fields__.keys():
        original_value = getattr(memento, field_name)

        if "base_path" == field_name:
            memento.base_path = path
            continue

        elif isinstance(original_value, BaseMemento) and recursive:
            replace_base_path(original_value, path, recursive=recursive)

        elif isinstance(original_value, (list, tuple, set, dict, BaseModel)) and recursive:
            walk_and_apply(original_value, memento.base_path, replace_base_path, recursive=True)

        elif isinstance(original_value, PosixPath):
            if original_value.name != "":
                new_path = path / original_value.name
                if not new_path.is_file():
                    continue
                    # TODO SDK SDK-48985 -> Need to check that only files that exist are created
                    raise FileExistsError(
                        "Memento attr points to a "
                        f"file {memento.__class__.__name__}.{field_name}->{new_path} This is not a File"
                    )
                setattr(memento, field_name, new_path)


def delete_memento(memento: MementoChild, *, recursive: bool = False):
    for field_name in memento.__fields__.keys():
        original_value = getattr(memento, field_name)

        if "base_path" == field_name:
            continue
        elif isinstance(original_value, BaseMemento) and recursive:
            delete_memento(original_value, recursive=recursive)

        elif isinstance(original_value, (list, tuple, set, dict, BaseModel)) and recursive:
            walk_and_apply(original_value, memento.base_path, delete_memento, recursive=True)

        elif isinstance(original_value, PosixPath) and original_value.is_file():
            original_value.unlink()

    if memento.base_path.is_dir() and not any(memento.base_path.iterdir()):
        memento.base_path.rmdir()

    memento.alive = False


def copy_memento(memento: MementoChild, path: PosixPath, *, recursive: bool = False) -> MementoChild:
    if path.is_file():
        raise ValueError(f"Path needs to be a directory: \n {path} <- is a file")
    path.mkdir(parents=True, exist_ok=True)

    attribute_values = {}
    for field_name in memento.__fields__.keys():
        original_value = getattr(memento, field_name)

        if "base_path" == field_name:
            attribute_values["base_path"] = path

        elif isinstance(original_value, BaseMemento) and recursive:
            copied_instance = copy_memento(original_value, path, recursive=True)
            attribute_values[field_name] = copied_instance

        elif isinstance(original_value, (list, tuple, set, dict, BaseModel)) and recursive:
            new_val = walk_and_apply(original_value, path, copy_memento, recursive=True)
            attribute_values[field_name] = new_val

        elif isinstance(original_value, PosixPath) and original_value.is_file():
            new_field_path = path / original_value.name
            new_field_path.parent.mkdir(parents=True, exist_ok=True)
            copy2(original_value, new_field_path)
            attribute_values[field_name] = new_field_path
        else:
            attribute_values[field_name] = deepcopy(original_value)

    new_instance = memento.__class__(**attribute_values)
    return new_instance


def move_memento(memento: MementoChild, path: PosixPath, *, recursive: bool = False) -> MementoChild:
    if not path.is_dir():
        raise ValueError(f"Target path must be a directory: {path}")

    for field_name in memento.__fields__.keys():
        original_value = getattr(memento, field_name)

        if field_name == "base_path":
            # Update the base_path field directly
            memento.base_path = path
        elif isinstance(original_value, BaseMemento) and recursive:
            # Recursively move nested BaseMemento instances
            move_memento(original_value, path, recursive=True)

        elif isinstance(original_value, (list, tuple, set, dict, BaseModel)) and recursive:
            # Recursively move items in containers
            setattr(memento, field_name, walk_and_apply(original_value, path, move_memento, recursive=True))

        elif isinstance(original_value, PosixPath) and original_value.is_file():
            # Calculate new path under the specified directory
            new_field_path = path / original_value.name
            new_field_path.parent.mkdir(parents=True, exist_ok=True)
            # Move the file
            shutil_move(original_value, new_field_path)
            # Update the attribute to reflect the new path
            setattr(memento, field_name, new_field_path)

        else:
            # Non-file and non-BaseMemento attributes need no specific action
            continue

    return memento


def walk_and_apply(obj, path: PosixPath, operation: Callable, *args, **kwargs):
    # First check if the object is an instance of BaseMemento
    if isinstance(obj, BaseMemento):
        return operation(obj, path, *args, **kwargs)

    # If not BaseMemento, check if the object is a list, tuple, or set
    elif isinstance(obj, (list, tuple, set)):
        return type(obj)(walk_and_apply(item, path, operation, *args, **kwargs) for item in obj)

    # Check if the object is a dictionary
    elif isinstance(obj, dict):
        return {key: walk_and_apply(value, path, operation, *args, **kwargs) for key, value in obj.items()}

    # Then check if the object is an instance of BaseModel
    elif isinstance(obj, BaseModel):
        # Iterate over its fields
        temp = obj.copy()
        for field_name in temp.__fields__.keys():
            field_value = getattr(obj, field_name)

            setattr(obj, field_name, walk_and_apply(field_value, path, operation, *args, **kwargs))
        return temp

    return obj


def save_memento(memento: BaseMemento, path: PosixPath) -> PosixPath:
    path = PosixPath(path)
    path.parent.mkdir(parents=True, exist_ok=True)
    copied_memento = copy_memento(memento, path, recursive=True)
    json_path = path.joinpath("memento.json")
    with json_path.open("w") as fd:
        fd.write(copied_memento.json(indent=4))
    return json_path


def load_memento(memento_class: Type[MementoChild], path: PosixPath) -> MementoChild:
    if path.is_dir():
        path = path.joinpath("memento.json")

    with path.open("r") as fd:
        memento = memento_class.parse_raw(fd.read())

    return memento
