from dataclasses import dataclass
from typing import Any, Dict, List, Optional, Union

import pyparsing as pp


@dataclass
class CommandInfo:
    """
    Meta data and content of command lines in model script
    """

    loc: int
    length: int
    command: str
    args: Optional[List[Any]]
    kwargs: Optional[Dict[str, Any]]
    command_object: Optional[str] = None
    return_val: Optional[Union[List[str], str, Dict[str, List[str]]]] = None


@dataclass
class CommentInfo:
    """
    Meta data and content of comment line in model script
    """

    loc: int
    length: int
    comment: str


def get_command_grammar():
    """
    Returns abstract grammar description of model script command using PyParsing
    """
    pp.ParserElement.setDefaultWhitespaceChars(" \t\r\n")
    left_par, right_par = pp.Literal("(").suppress(), pp.Literal(")").suppress()
    left_brck, right_brck = pp.Literal("[").suppress(), pp.Literal("]").suppress()
    left_cbrck, right_cbrck = pp.Literal("{"), pp.Literal("}")

    eq = pp.Literal("=").suppress()

    identifier = pp.Word(pp.alphanums, pp.alphanums + "_" + "-" + "/")
    identifier_val = pp.Word(pp.alphanums + "+-" + "_", pp.alphanums + "_" + "-" + "/" + ".")

    basic_variable_val = pp.quotedString | identifier_val
    glob_syntax_variable_val = pp.Combine(
        left_cbrck + pp.Word(pp.alphanums + "*", pp.alphanums + "-/_*?![]") + right_cbrck,
    )

    single_variable_val = glob_syntax_variable_val | basic_variable_val + pp.NotAny("=")
    list_of_variables_val = pp.Group(left_brck + pp.delimitedList(single_variable_val, ",") + right_brck)
    set_of_variables_val = pp.Combine(left_cbrck + pp.Word(pp.alphanums + "*") + right_cbrck)

    variable_val = set_of_variables_val | list_of_variables_val | single_variable_val

    args = pp.Group(pp.delimitedList(variable_val, ","))("args")
    keyval = pp.dictOf(identifier + eq, variable_val)
    kwargs = pp.Group(pp.delimitedList(keyval, ","))("kwargs")
    fxn_args = (args + "," + kwargs) | (pp.Optional(args, default="") + pp.Optional(kwargs, default=""))
    command = identifier("command") + left_par + fxn_args + right_par

    # Define elements for key-value pairs with variable names
    LBRACE, RBRACE, LBRACK, RBRACK, COMMA, COLON = map(pp.Literal, "{}[],:")
    value = pp.delimitedList(identifier_val, delim=COMMA)
    key_value_pairs = pp.Group(
        pp.delimitedList(
            pp.dictOf(identifier_val + COLON.suppress(), LBRACK.suppress() + value + RBRACK.suppress()), delim=COMMA
        )
    )

    # Define the main expression
    dict_return_vals = LBRACE.suppress() + pp.Optional(key_value_pairs, {}) + RBRACE.suppress()

    return_val = (pp.delimitedList(identifier, ",") | dict_return_vals)("return_val") + eq

    command_object = identifier("command_object") + pp.Literal(".").suppress()

    complete_command = pp.LineStart() + pp.Optional(return_val) + pp.Optional(command_object) + command + pp.LineEnd()

    return complete_command


def get_comment_grammar():
    """
    Returns abstract grammar description of model script comment using PyParsing
    """
    comment = pp.LineStart() + pp.Literal("#").suppress() + pp.Optional(pp.restOfLine)("comment") + pp.LineEnd()
    return comment


def get_blank_grammar():
    """
    Returns abstract grammar description of empty line in model script
    """
    new_line = pp.LineEnd().suppress()
    return new_line


def handle_command(loc, tokens) -> CommandInfo:
    """
    Tokens handler for model script command
    """
    command_info = tokens.asDict()
    info = CommandInfo(**command_info, loc=loc, length=0)
    return info


def handle_comment(loc, tokens) -> CommentInfo:
    """
    Tokens handler for model script comment
    """
    comment_info = tokens.asDict()
    info = CommentInfo(**comment_info, loc=loc, length=0)
    return info


def parse_model_script(model_script: str) -> List[Union[CommandInfo, CommentInfo]]:
    """
    Parse a model script into simple dataclass obects using Pyparsing.
    """
    if model_script is None:
        model_script = ""
    command = get_command_grammar()
    comment = get_comment_grammar()
    blank = get_blank_grammar()

    command.addParseAction(handle_command)
    comment.addParseAction(handle_comment)

    full_grammar = pp.ZeroOrMore(blank | command | comment)
    parsed = full_grammar.parseString(model_script, parseAll=True)
    for i in range(len(parsed) - 1):
        parsed[i].length = parsed[i + 1].loc - parsed[i].loc
    if len(parsed) > 0:
        parsed[-1].length = len(model_script) - parsed[-1].loc
    return parsed
