#!/usr/bin/env python

from enum import Enum

from hailo_sdk_common.hailo_nn.tools_params import (
    AutoInt,
    ToolsParams,
    allowed_str_for_bool,
    allowed_str_for_enumerate,
    convert_str_to_bool,
    convert_str_to_enumerate,
    convert_time_to_auto_int,
    convert_time_to_int,
    convert_to_auto_int,
    convert_to_float,
    convert_to_int,
)
from hailo_sdk_common.paths_manager.config import get_parsed_config_from_path


class AllocatorAgent(Enum):
    LAZY = "lazy"
    MELG = "melg"
    MIGDALODON = "migdalodon"
    AUTOMATIC = "automatic"

    def __str__(self):
        return self.value


class AllocatorStrategy(Enum):
    LAZY = "lazy"
    BLIND_SEARCH = "blind_search"
    POSITIVE_SEARCH = "positive_search"


class BuilderExitPoint(Enum):
    POST_WINDOWS = "post-windows"
    POST_RACEHORSE = "post-racehorse"
    POST_DEFUSE = "post-defuse"
    POST_EXPANSION = "post-expansion"
    POST_SPLIT = "post-split"
    POST_PARTITION = "post-partition"
    POST_ALLOCATION = "post-allocation"
    POST_COMPILATION = "post-compilation"
    POST_CAT = "post-cat"
    POST_FORMULA = "post-formula"

    def __str__(self):
        return self.value


class ResolverRevertStrategy(Enum):
    PATIENT = "patient"
    ASAP = "asap"


class InputFeatureSplitterDefuse(Enum):
    ALLOWED = "allowed"
    ENABLED = "enabled"
    DISABLED = "disabled"


InputFeatureSplitterDefusePolicies = {x.value: x for x in InputFeatureSplitterDefuse}


class WidthSplitterDefuse(Enum):
    ALLOWED = "allowed"
    ENABLED = "enabled"
    DISABLED = "disabled"


WidthSplitterDefusePolicies = {x.value: x for x in WidthSplitterDefuse}


class AutomaticReshapesPolicy(Enum):
    ALLOWED = "allowed"
    ENABLED = "enabled"
    DISABLED = "disabled"


AutomaticReshapesPolicies = {x.value: x for x in AutomaticReshapesPolicy}


class PartialRowBuffersType(Enum):
    PARTIAL_ROW_BUFFERS_CONSTRAINED = "constrained"
    PARTIAL_ROW_BUFFERS_DISABLED = "disabled"
    PARTIAL_ROW_BUFFERS_ENABLED = "enabled"


class EnableFixerPolicy(Enum):
    DISABLED = "disabled"
    ENABLED = "enabled"
    MAX_ADJACENTS = "max_adjcents"

    @classmethod
    def _missing_(cls, value):
        if isinstance(value, str):
            if value.lower() == "true":
                return cls.ENABLED
            elif value.lower() == "false":
                return cls.DISABLED
            else:
                return super()._missing_(value)
        elif isinstance(value, bool):
            return cls.ENABLED if value else cls.DISABLED
        else:
            return super()._missing_(value)


class AllocatorParams(ToolsParams):
    DEFAULT_PARAMS = {
        "timeout": AutoInt("automatic"),
        "cluster_timeout": 60 * 5,
        "splitter_timeout": 60 * 10,
        "validate": True,
        "agent": AllocatorAgent.AUTOMATIC,
        "builder_exit_point": BuilderExitPoint.POST_COMPILATION,
        "strategy": AllocatorStrategy.BLIND_SEARCH,
        "enable_macros": True,
        "dannox": False,
        "max_cluster_util": 1.0,
        "seed": 0,
        "max_auto_merge_depth": 2,
        "merge_min_layer_utilization": 0.95,
        "merge_max_layer_utilization": 1.2,
        "merge_max_memory_utilization": 1.0,
        "defuse_by_validator": True,
        "enable_fixer": EnableFixerPolicy.ENABLED,
        "enable_unikorn": True,
        "enable_dual_shmifo": True,
        "max_auto_defuse": AutoInt("automatic"),
        "enable_swapper": True,
        "automatic_l4_portals": True,
        "automatic_l4_portals_l3_cuts_threshold": 15,
        "enable_lanes": True,
        # TODO: change to True once collapser feature is fully implemented (SDK-16377)
        "enable_collapser": False,
        "dump_statistics": False,
        "automatic_ddr": True,
        "automatic_ddr_l3_cuts_threshold": AutoInt("automatic"),
        "enable_loose_mode": False,
        "enable_axis_upsize": True,
        "enable_defuse_with_slack": True,
        "resolver_revert_strategy": ResolverRevertStrategy.ASAP,
        "resolver_revert_strategy_asap_delay": 15,
        "microcoder_without_halts": False,
        "prepost_haltless": False,
        "optimize_buffers": True,
        "split_aware_optimize_buffers": True,
        "print_minimal_buffers": False,
        "buffer_calc_fps": 0,
        "save_latency_timeline": True,
        "optimize_one_to_many_shmifos": True,
        "use_minimal_buffers": False,
        "minimal_buffers_at_fifos": False,
        "enable_partial_row_buffers": PartialRowBuffersType.PARTIAL_ROW_BUFFERS_CONSTRAINED,
        "enable_muxer": True,
        "enable_muxer_metric": True,
        "enable_input_phases": True,
        "muxer_slack": 0,
        "enable_swapper_router": True,
        "enable_shiftman": True,
        "input_feature_splitter_defuse": InputFeatureSplitterDefuse.ALLOWED,
        "compilation_num_threads": AutoInt("automatic"),
        "automatic_reshapes": AutomaticReshapesPolicy.ALLOWED,
        "offload_argmax": False,
        "success_asap": True,
        "num_of_workers": 4,
        "enable_hw_padding": True,
        "enable_barakbak_scripts": False,
        "enable_l3_balloon_optimization": True,
        "disable_row_per_cut": False,
        "max_clusters_forcing": 0,
        "ppu_reshape_slack": AutoInt("automatic"),
        "number_of_active_clusters": AutoInt("automatic"),
        "enable_mjitc": False,
        "enable_auto_spatial_reshapes": True,
        "enable_auto_spatial_collapse": True,
        "enable_post_split_average_buffers": True,
        "enable_periph": True,
        "enable_network_groups": False,
        "automatic_resize_reshapes": AutomaticReshapesPolicy.ALLOWED,
        "calculate_power": True,
        "auto_split_const_inputs": True,
        "enable_conv_2_apus_in_racehorse_pyramids": True,
        "width_splitter_defuse": WidthSplitterDefuse.ALLOWED,
        "enable_automatic_offdevice_outputs": True,
        "enable_lcu_ecc": True,
        "assert_mux_succeeded": True,
        "input_layer_max_successors": AutoInt("automatic"),
        "optimize_pipeline_stoppage": False,
        "force_map_on_mirror": False,
    }
    BACKWARD_COMP_PARAMS = {"max_lcu_utilization": "merge_min_layer_utilization"}
    ALLOWED_VALUES = {
        "validate": allowed_str_for_bool(),
        "agent": allowed_str_for_enumerate(AllocatorAgent),
        "builder_exit_point": allowed_str_for_enumerate(BuilderExitPoint),
        "strategy": allowed_str_for_enumerate(AllocatorStrategy),
        "enable_macros": allowed_str_for_bool(),
        "dannox": allowed_str_for_bool(),
        "defuse_by_validator": allowed_str_for_bool(),
        "enable_fixer": allowed_str_for_bool() + allowed_str_for_enumerate(EnableFixerPolicy),
        "enable_unikorn": allowed_str_for_bool(),
        "enable_dual_shmifo": allowed_str_for_bool(),
        "enable_swapper": allowed_str_for_bool(),
        "automatic_l4_portals": allowed_str_for_bool(),
        "enable_lanes": allowed_str_for_bool(),
        "enable_collapser": allowed_str_for_bool(),
        "dump_statistics": allowed_str_for_bool(),
        "automatic_ddr": allowed_str_for_bool(),
        "enable_loose_mode": allowed_str_for_bool(),
        "enable_axis_upsize": allowed_str_for_bool(),
        "enable_defuse_with_slack": allowed_str_for_bool(),
        "resolver_revert_strategy": allowed_str_for_enumerate(ResolverRevertStrategy),
        "microcoder_without_halts": allowed_str_for_bool(),
        "prepost_haltless": allowed_str_for_bool(),
        "optimize_buffers": allowed_str_for_bool(),
        "split_aware_optimize_buffers": allowed_str_for_bool(),
        "print_minimal_buffers": allowed_str_for_bool(),
        "optimize_one_to_many_shmifos": allowed_str_for_bool(),
        "use_minimal_buffers": allowed_str_for_bool(),
        "minimal_buffers_at_fifos": allowed_str_for_bool(),
        "enable_partial_row_buffers": allowed_str_for_enumerate(PartialRowBuffersType),
        "enable_muxer": allowed_str_for_bool(),
        "enable_muxer_metric": allowed_str_for_bool(),
        "enable_input_phases": allowed_str_for_bool(),
        "enable_swapper_router": allowed_str_for_bool(),
        "enable_shiftman": allowed_str_for_bool(),
        "input_feature_splitter_defuse": allowed_str_for_enumerate(InputFeatureSplitterDefuse),
        "automatic_reshapes": allowed_str_for_enumerate(AutomaticReshapesPolicy),
        "offload_argmax": allowed_str_for_bool(),
        "success_asap": allowed_str_for_bool(),
        "enable_hw_padding": allowed_str_for_bool(),
        "enable_barakbak_scripts": allowed_str_for_bool(),
        "enable_l3_balloon_optimization": allowed_str_for_bool(),
        "disable_row_per_cut": allowed_str_for_bool(),
        "enable_mjitc": allowed_str_for_bool(),
        "enable_auto_spatial_reshapes": allowed_str_for_bool(),
        "enable_auto_spatial_collapse": allowed_str_for_bool(),
        "enable_post_split_average_buffers": allowed_str_for_bool(),
        "enable_periph": allowed_str_for_bool(),
        "enable_network_groups": allowed_str_for_bool(),
        "automatic_resize_reshapes": allowed_str_for_enumerate(AutomaticReshapesPolicy),
        "calculate_power": allowed_str_for_bool(),
        "auto_split_const_inputs": allowed_str_for_bool(),
        "enable_conv_2_apus_in_racehorse_pyramids": allowed_str_for_bool(),
        "width_splitter_defuse": allowed_str_for_enumerate(WidthSplitterDefuse),
        "enable_automatic_offdevice_outputs": allowed_str_for_bool(),
        "enable_lcu_ecc": allowed_str_for_bool(),
        "optimize_pipeline_stoppage": allowed_str_for_bool(),
        "force_map_on_mirror": allowed_str_for_bool(),
    }
    VALUE_CONVERTERS = {
        "timeout": convert_time_to_auto_int,
        "cluster_timeout": convert_time_to_int,
        "splitter_timeout": convert_time_to_int,
        "validate": convert_str_to_bool,
        "agent": convert_str_to_enumerate(AllocatorAgent),
        "builder_exit_point": convert_str_to_enumerate(BuilderExitPoint),
        "strategy": convert_str_to_enumerate(AllocatorStrategy),
        "enable_macros": convert_str_to_bool,
        "dannox": convert_str_to_bool,
        "max_cluster_util": convert_to_float,
        "seed": convert_to_int,
        "max_auto_merge_depth": convert_to_int,
        "merge_min_layer_utilization": convert_to_float,
        "merge_max_layer_utilization": convert_to_float,
        "merge_max_memory_utilization": convert_to_float,
        "defuse_by_validator": convert_str_to_bool,
        "enable_fixer": convert_str_to_enumerate(EnableFixerPolicy),
        "enable_unikorn": convert_str_to_bool,
        "enable_dual_shmifo": convert_str_to_bool,
        "max_auto_defuse": convert_to_auto_int,
        "enable_swapper": convert_str_to_bool,
        "automatic_l4_portals": convert_str_to_bool,
        "automatic_l4_portals_l3_cuts_threshold": convert_to_int,
        "enable_lanes": convert_str_to_bool,
        "enable_collapser": convert_str_to_bool,
        "dump_statistics": convert_str_to_bool,
        "automatic_ddr": convert_str_to_bool,
        "automatic_ddr_l3_cuts_threshold": convert_to_auto_int,
        "enable_loose_mode": convert_str_to_bool,
        "enable_axis_upsize": convert_str_to_bool,
        "enable_defuse_with_slack": convert_str_to_bool,
        "resolver_revert_strategy": convert_str_to_enumerate(ResolverRevertStrategy),
        "resolver_revert_strategy_asap_delay": convert_to_int,
        "microcoder_without_halts": convert_str_to_bool,
        "prepost_haltless": convert_str_to_bool,
        "optimize_buffers": convert_str_to_bool,
        "split_aware_optimize_buffers": convert_str_to_bool,
        "print_minimal_buffers": convert_str_to_bool,
        "buffer_calc_fps": convert_to_float,
        "save_latency_timeline": convert_str_to_bool,
        "optimize_one_to_many_shmifos": convert_str_to_bool,
        "use_minimal_buffers": convert_str_to_bool,
        "minimal_buffers_at_fifos": convert_str_to_bool,
        "enable_partial_row_buffers": convert_str_to_enumerate(PartialRowBuffersType),
        "enable_muxer": convert_str_to_bool,
        "enable_muxer_metric": convert_str_to_bool,
        "enable_input_phases": convert_str_to_bool,
        "muxer_slack": convert_to_int,
        "enable_swapper_router": convert_str_to_bool,
        "enable_shiftman": convert_str_to_bool,
        "input_feature_splitter_defuse": convert_str_to_enumerate(InputFeatureSplitterDefuse),
        "compilation_num_threads": convert_to_auto_int,
        "automatic_reshapes": convert_str_to_enumerate(AutomaticReshapesPolicy),
        "offload_argmax": convert_str_to_bool,
        "success_asap": convert_str_to_bool,
        "num_of_workers": convert_to_int,
        "enable_hw_padding": convert_str_to_bool,
        "enable_barakbak_scripts": convert_str_to_bool,
        "enable_l3_balloon_optimization": convert_str_to_bool,
        "disable_row_per_cut": convert_str_to_bool,
        "max_clusters_forcing": convert_to_int,
        "ppu_reshape_slack": convert_to_auto_int,
        "number_of_active_clusters": convert_to_auto_int,
        "enable_mjitc": convert_str_to_bool,
        "enable_auto_spatial_reshapes": convert_str_to_bool,
        "enable_auto_spatial_collapse": convert_str_to_bool,
        "enable_post_split_average_buffers": convert_str_to_bool,
        "enable_periph": convert_str_to_bool,
        "enable_network_groups": convert_str_to_bool,
        "automatic_resize_reshapes": convert_str_to_enumerate(AutomaticReshapesPolicy),
        "calculate_power": convert_str_to_bool,
        "auto_split_const_inputs": convert_str_to_bool,
        "enable_conv_2_apus_in_racehorse_pyramids": convert_str_to_bool,
        "width_splitter_defuse": convert_str_to_enumerate(WidthSplitterDefuse),
        "enable_automatic_offdevice_outputs": convert_str_to_bool,
        "enable_lcu_ecc": convert_str_to_bool,
        "assert_mux_succeeded": convert_str_to_bool,
        "input_layer_max_successors": convert_to_auto_int,
        "optimize_pipeline_stoppage": convert_str_to_bool,
        "force_map_on_mirror": convert_str_to_bool,
    }
    DEPRECATED_PARAMS = {
        "network_fps",
        "clk_freq",
        "optimize_one_to_many_shmifos",
        "max_lcu_usage",
    }
    POLICY_TYPES = {
        "input_feature_splitter_defuse": InputFeatureSplitterDefusePolicies,
        "automatic_reshapes": AutomaticReshapesPolicies,
        "width_splitter_defuse": WidthSplitterDefusePolicies,
    }

    def __init__(
        self,
        timeout=None,
        agent=None,
        strategy=None,
        max_cluster_util=None,
        builder_exit_point=None,
        enable_fixer=True,
        enable_swapper=True,
        enable_axis_upsize=True,
        enable_defuse_with_slack=True,
        dannox=False,
        enable_lanes=True,
        enable_collapser=False,
        dump_statistics=False,
        offload_argmax=False,
    ):
        super().__init__()
        self.set_default_params()
        self.override_params_from_kwargs(
            timeout=timeout,
            agent=agent,
            strategy=strategy,
            max_cluster_util=max_cluster_util,
            builder_exit_point=builder_exit_point,
            enable_fixer=EnableFixerPolicy(enable_fixer),
            enable_swapper=enable_swapper,
            enable_axis_upsize=enable_axis_upsize,
            enable_defuse_with_slack=enable_defuse_with_slack,
            dannox=dannox,
            enable_lanes=enable_lanes,
            enable_collapser=enable_collapser,
            dump_statistics=dump_statistics,
            offload_argmax=offload_argmax,
        )

    def to_pb(self, pb, pb_wrapper):
        self.set_pb_field(pb, "agent", pb_wrapper.ALLOCATOR_AGENT_TYPE_TO_PB)
        self.set_pb_field(pb, "strategy", pb_wrapper.ALLOCATOR_STRATEGY_TYPE_TO_PB)
        self.set_auto_pb_field(pb, pb_wrapper, "timeout")
        self.set_pb_field(pb, "cluster_timeout")
        self.set_pb_field(pb, "splitter_timeout")
        self.set_pb_field(pb, "validate")
        self.set_pb_field(pb, "enable_macros")
        self.set_pb_field(pb, "dannox")
        if self._params.get("dannox"):
            config = get_parsed_config_from_path()
            dannox_gui = config["dannox"]["gui"] if ("dannox" in config and "gui" in config["dannox"]) else False
            pb.dannox_gui = pb_wrapper.STR_TO_BOOL_TO_PB[dannox_gui]
            pb.dannox_read_port = int(config["dannox"]["read_port"])
            pb.dannox_write_port = int(config["dannox"]["write_port"])
            pb.dannox_game_port = int(config["dannox"]["game_port"])
        self.set_pb_field(pb, "max_cluster_util")
        self.set_pb_field(pb, "seed")
        self.set_pb_field(pb, "max_auto_merge_depth")
        self.set_pb_field(pb, "merge_min_layer_utilization")
        self.set_pb_field(pb, "merge_max_layer_utilization")
        self.set_pb_field(pb, "merge_max_memory_utilization")
        self.set_pb_field(pb, "defuse_by_validator")
        self.set_pb_field(pb, "builder_exit_point", pb_wrapper.BUILDER_EXIT_POINT_TYPE_TO_PB)
        self.set_pb_field(pb, "enable_fixer", pb_wrapper.ENABLE_FIXER_POLICY_TYPE_TO_PB)
        self.set_pb_field(pb, "enable_unikorn")
        self.set_pb_field(pb, "enable_dual_shmifo")
        self.set_pb_field(pb, "enable_swapper")
        self.set_pb_field(pb, "automatic_l4_portals")
        self.set_pb_field(pb, "automatic_l4_portals_l3_cuts_threshold")
        self.set_pb_field(pb, "enable_lanes")
        self.set_pb_field(pb, "enable_collapser")
        self.set_pb_field(pb, "dump_statistics")
        self.set_pb_field(pb, "automatic_ddr")
        self.set_auto_pb_field(pb, pb_wrapper, "automatic_ddr_l3_cuts_threshold")
        self.set_auto_pb_field(pb, pb_wrapper, "compilation_num_threads")
        self.set_auto_pb_field(pb, pb_wrapper, "ppu_reshape_slack")
        self.set_auto_pb_field(pb, pb_wrapper, "number_of_active_clusters")
        self.set_pb_field(pb, "enable_loose_mode")
        self.set_pb_field(pb, "enable_axis_upsize")
        self.set_pb_field(pb, "enable_defuse_with_slack")
        self.set_pb_field(pb, "resolver_revert_strategy", pb_wrapper.RESOLVER_REVERT_STRATEGY_TYPE_TO_PB)
        self.set_pb_field(pb, "resolver_revert_strategy_asap_delay")
        self.set_pb_field(pb, "microcoder_without_halts")
        self.set_pb_field(pb, "prepost_haltless")
        self.set_pb_field(pb, "optimize_buffers")
        self.set_pb_field(pb, "split_aware_optimize_buffers")
        self.set_pb_field(pb, "print_minimal_buffers")
        self.set_pb_field(pb, "buffer_calc_fps")
        self.set_pb_field(pb, "save_latency_timeline")
        self.set_pb_field(pb, "optimize_one_to_many_shmifos")
        self.set_auto_pb_field(pb, pb_wrapper, "max_auto_defuse")
        self.set_pb_field(pb, "use_minimal_buffers")
        self.set_pb_field(pb, "minimal_buffers_at_fifos")
        self.set_pb_field(pb, "enable_partial_row_buffers", pb_wrapper.PARTIAL_ROW_BUFFERS_TYPE_TO_PB)
        self.set_pb_field(pb, "enable_muxer")
        self.set_pb_field(pb, "enable_muxer_metric")
        self.set_pb_field(pb, "enable_input_phases")
        self.set_pb_field(pb, "muxer_slack")
        self.set_pb_field(pb, "enable_swapper_router")
        self.set_pb_field(pb, "enable_shiftman")
        self.set_pb_field(pb, "input_feature_splitter_defuse", pb_wrapper.INPUT_FEATURE_SPLITTER_DEFUSE_TYPE_TO_PB)
        self.set_pb_field(pb, "automatic_reshapes", pb_wrapper.AUTOMATIC_RESHAPES_TYPE_TO_PB)
        self.set_pb_field(pb, "offload_argmax")
        self.set_pb_field(pb, "num_of_workers")
        self.set_pb_field(pb, "enable_hw_padding")
        self.set_pb_field(pb, "enable_barakbak_scripts")
        self.set_pb_field(pb, "enable_l3_balloon_optimization")
        self.set_pb_field(pb, "disable_row_per_cut")
        self.set_pb_field(pb, "max_clusters_forcing")
        self.set_pb_field(pb, "enable_mjitc")
        self.set_pb_field(pb, "enable_auto_spatial_reshapes")
        self.set_pb_field(pb, "enable_auto_spatial_collapse")
        self.set_pb_field(pb, "enable_post_split_average_buffers")
        self.set_pb_field(pb, "enable_periph")
        self.set_pb_field(pb, "enable_network_groups")
        self.set_pb_field(pb, "automatic_resize_reshapes", pb_wrapper.AUTOMATIC_RESHAPES_TYPE_TO_PB)
        self.set_pb_field(pb, "calculate_power")
        self.set_pb_field(pb, "auto_split_const_inputs")
        self.set_pb_field(pb, "enable_conv_2_apus_in_racehorse_pyramids")
        self.set_pb_field(pb, "width_splitter_defuse", pb_wrapper.WIDTH_SPLITTER_DEFUSE_TYPE_TO_PB)
        self.set_pb_field(pb, "enable_automatic_offdevice_outputs")
        self.set_pb_field(pb, "enable_lcu_ecc")
        self.set_pb_field(pb, "assert_mux_succeeded")
        self.set_auto_pb_field(pb, pb_wrapper, "input_layer_max_successors")
        self.set_pb_field(pb, "optimize_pipeline_stoppage")
        self.set_pb_field(pb, "force_map_on_mirror")

    def from_pb(self, pb, pb_wrapper):
        self.get_pb_field(pb, "agent", pb_wrapper.ALLOCATOR_AGENT_PB_TO_TYPE)
        self.get_pb_field(pb, "strategy", pb_wrapper.ALLOCATOR_STRATEGY_PB_TO_TYPE)
        self.get_auto_pb_field(pb, "timeout", AutoInt)
        self.get_pb_field(pb, "validate", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_pb_field(pb, "cluster_timeout")
        self.get_pb_field(pb, "splitter_timeout")
        self.get_pb_field(pb, "enable_macros", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_pb_field(pb, "max_cluster_util")
        self.get_pb_field(pb, "seed")
        self.get_pb_field(pb, "merge_min_layer_utilization")
        self.get_pb_field(pb, "merge_max_layer_utilization")
        self.get_pb_field(pb, "merge_max_memory_utilization")
        self.get_pb_field(pb, "max_auto_merge_depth")
        self.get_pb_field(pb, "defuse_by_validator")
        self.get_pb_field(pb, "builder_exit_point", pb_wrapper.BUILDER_EXIT_POINT_PB_TO_TYPE)
        self.get_pb_field(pb, "enable_fixer", pb_wrapper.ENABLE_FIXER_PB_TO_TYPE)
        self.get_pb_field(pb, "enable_unikorn")
        self.get_pb_field(pb, "enable_dual_shmifo")
        self.get_pb_field(pb, "enable_swapper")
        self.get_pb_field(pb, "automatic_l4_portals")
        self.get_pb_field(pb, "automatic_l4_portals_l3_cuts_threshold")
        self.get_pb_field(pb, "enable_lanes")
        self.get_pb_field(pb, "enable_collapser")
        self.get_pb_field(pb, "dump_statistics")
        self.get_pb_field(pb, "automatic_ddr")
        self.get_auto_pb_field(pb, "automatic_ddr_l3_cuts_threshold", AutoInt)
        self.get_auto_pb_field(pb, "compilation_num_threads", AutoInt)
        self.get_auto_pb_field(pb, "ppu_reshape_slack", AutoInt)
        self.get_auto_pb_field(pb, "number_of_active_clusters", AutoInt)
        self.get_pb_field(pb, "enable_loose_mode")
        self.get_pb_field(pb, "enable_axis_upsize")
        self.get_pb_field(pb, "enable_defuse_with_slack")
        self.get_pb_field(pb, "resolver_revert_strategy", pb_wrapper.RESOLVER_REVERT_STRATEGY_PB_TO_TYPE)
        self.get_pb_field(pb, "resolver_revert_strategy_asap_delay")
        self.get_pb_field(pb, "microcoder_without_halts", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_pb_field(pb, "prepost_haltless", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_pb_field(pb, "optimize_buffers", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_pb_field(pb, "split_aware_optimize_buffers", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_pb_field(pb, "print_minimal_buffers", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_pb_field(pb, "buffer_calc_fps")
        self.get_pb_field(pb, "save_latency_timeline")
        self.get_pb_field(pb, "optimize_one_to_many_shmifos", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_auto_pb_field(pb, "max_auto_defuse", AutoInt)
        self.get_pb_field(pb, "use_minimal_buffers")
        self.get_pb_field(pb, "minimal_buffers_at_fifos")
        self.get_pb_field(pb, "enable_partial_row_buffers", pb_wrapper.PARTIAL_ROW_BUFFERS_PB_TO_TYPE)
        self.get_pb_field(pb, "enable_muxer", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_pb_field(pb, "enable_muxer_metric", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_pb_field(pb, "enable_input_phases", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_pb_field(pb, "muxer_slack")
        self.get_pb_field(pb, "enable_swapper_router", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_pb_field(pb, "enable_shiftman", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_pb_field(pb, "input_feature_splitter_defuse", pb_wrapper.INPUT_FEATURE_SPLITTER_DEFUSE_PB_TO_TYPE)
        self.get_pb_field(pb, "automatic_reshapes", pb_wrapper.AUTOMATIC_RESHAPES_PB_TO_TYPE)
        self.get_pb_field(pb, "offload_argmax", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_pb_field(pb, "num_of_workers")
        self.get_pb_field(pb, "enable_hw_padding", pb_wrapper.STR_TO_BOOL_TO_PB)
        self.get_pb_field(pb, "enable_l3_balloon_optimization")
        self.get_pb_field(pb, "disable_row_per_cut")
        self.get_pb_field(pb, "max_clusters_forcing")
        self.get_pb_field(pb, "enable_mjitc")
        self.get_pb_field(pb, "enable_auto_spatial_reshapes")
        self.get_pb_field(pb, "enable_auto_spatial_collapse")
        self.get_pb_field(pb, "enable_post_split_average_buffers")
        self.get_pb_field(pb, "enable_periph")
        self.get_pb_field(pb, "enable_network_groups")
        self.get_pb_field(pb, "automatic_resize_reshapes", pb_wrapper.AUTOMATIC_RESHAPES_PB_TO_TYPE)
        self.get_pb_field(pb, "calculate_power")
        self.get_pb_field(pb, "auto_split_const_inputs")
        self.get_pb_field(pb, "enable_conv_2_apus_in_racehorse_pyramids")
        self.get_pb_field(pb, "width_splitter_defuse", pb_wrapper.WIDTH_SPLITTER_DEFUSE_PB_TO_TYPE)
        self.get_pb_field(pb, "enable_automatic_offdevice_outputs")
        self.get_pb_field(pb, "enable_lcu_ecc")
        self.get_pb_field(pb, "assert_mux_succeeded")
        self.get_auto_pb_field(pb, "input_layer_max_successors", AutoInt)
        self.get_pb_field(pb, "optimize_pipeline_stoppage")
        self.get_pb_field(pb, "force_map_on_mirror")
