#!/usr/bin/env python

import sys

from hailo_model_optimization.acceleras.utils.acceleras_definitions import (
    BiasMode,
    ConcatAxis,
    IOType,
    OutputMinMaxStrategy,
    PostprocessTarget,
    PostprocessType,
    PrecisionMode,
)
from hailo_sdk_client.allocator import integrated_hw_graph_base_pb2, integrated_hw_graph_pb2
from hailo_sdk_client.allocator.allocator_params import (
    AllocatorAgent,
    AllocatorStrategy,
    AutomaticReshapesPolicy,
    BuilderExitPoint,
    EnableFixerPolicy,
    InputFeatureSplitterDefuse,
    PartialRowBuffersType,
    ResolverRevertStrategy,
    WidthSplitterDefuse,
)
from hailo_sdk_client.allocator.context_switch_params import ContextSwitchMode, Partitioner, ToposortMode
from hailo_sdk_client.allocator.hef_params import (
    EnableLcuFromSequencerPolicy,
    ParamsLoadTimeCompressionPolicy,
    UseSequencerPolicy,
)
from hailo_sdk_client.allocator.logger_params import LoggerVerbosity
from hailo_sdk_client.allocator.platform_params import PlatformHints, PlatformTargets
from hailo_sdk_client.allocator.resources_params import ResourcesObjective, ResourcesStrategy
from hailo_sdk_common.hailo_nn.exceptions import ProtobufExportError, UnsupportedModelError
from hailo_sdk_common.hailo_nn.hailo_nn import HailoNN, HailoNNExporter, HailoNNImporter, NetParams
from hailo_sdk_common.hailo_nn.hn_definitions import (
    ActivationType,
    BalanceOutputMultisplitPolicy,
    CompressL3WeightsPolicy,
    DefuseType,
    DepthToSpaceType,
    EnableEwAddOptimizationPolicy,
    FeatureMultiplierType,
    FormatConversionType,
    HnStage,
    HnVersion,
    HWLayerType,
    KoPolicy,
    LayerBuffersFormat,
    LayerType,
    NMSMetaArchitectures,
    NormalizationType,
    PaddingType,
    ParallelActivationPolicy,
    PrecisionSplitMode,
    ResizeBilinearPixelsMode,
    ResizeBilinearStreamingPolicy,
    ResizeMethod,
    ResourcesAllocationStrategy,
    ShapeSplitterType,
    ShouldAlignCcwsSection,
    SpaceToDepthType,
    Subclusters16x4Policy,
    SubclustersNoContextsPolicy,
    TwoLineBufferModePolicy,
    UseL2WeightsPolicy,
)
from hailo_sdk_common.hailo_nn.hn_layers import (
    ActivationLayer,
    ArgmaxLayer,
    BatchNormLayer,
    BboxDecoderLayer,
    BiasAddLayer,
    ConcatLayer,
    ConstInputLayer,
    Conv2DLayer,
    DemuxLayer,
    DenseLayer,
    DepthToSpaceLayer,
    EWAddLayer,
    EWMultLayer,
    EWSubLayer,
    ExternalInputLayer,
    ExternalOutputLayer,
    ExternalPadLayer,
    FeatureInterleaveLayer,
    FeatureMultiplierLayer,
    FeatureShuffleLayer,
    FeatureSplitterLayer,
    FormatConversionLayer,
    FusedBatchNormLayer,
    FusedBboxDecoderLayer,
    FusedConv2DLayer,
    FusedDenseLayer,
    FusedEWAddLayer,
    FusedEWSubLayer,
    FusedSliceLayer,
    FusedStandaloneActivationLayer,
    FusedStandaloneEWAddLayer,
    FusedStandaloneEWSubLayer,
    GlobalAvgPoolingLayer,
    InputLayer,
    LayerNormalizationLayer,
    LoopbackLayer,
    MatmulLayer,
    MergedLayer,
    NMSLayer,
    NormalizationLayer,
    OutputLayer,
    OutputMuxLayer,
    PoolingLayer,
    PortalLayer,
    PostprocessLayer,
    PPInputLayer,
    PPOutputLayer,
    PrecisionSplitterLayer,
    ProposalGeneratorLayer,
    ReduceMaxLayer,
    ReduceMeanLayer,
    ReduceSumLayer,
    ResizeLayer,
    RowSplitterLayer,
    ShortcutLayer,
    SliceLayer,
    SoftmaxLayer,
    SpaceToDepthLayer,
    WidthSplitterLayer,
)
from hailo_sdk_common.hailo_nn.tools_params import AutoVariablePolicy
from hailo_sdk_common.logger.logger import default_logger


class ProtobufExporter(HailoNNExporter):
    def __init__(self, hailo_nn, allocator_script_filename=None):
        super().__init__(hailo_nn)
        self._layers_to_pb_indices = {}
        self._pb_indices_to_layers = {}
        self._graph = None
        self._integrated_pb = None
        self._allocator_script_filename = allocator_script_filename

    def to_pb(self, graph=None):
        self._hailo_nn.set_names_and_indices()
        self._hailo_nn.update_output_indices()
        # backwards compatibility for activations as weights
        self._hailo_nn.update_signed_output()
        if graph is None:
            graph = integrated_hw_graph_base_pb2.ProtoNetworkGraph()
        self._graph = graph
        self._layers_to_pb_indices = {}
        self._pb_indices_to_layers = {}
        self._add_nodes()
        self._add_edges()
        self._add_net_params()
        return self._graph

    def to_integrated_pb(self):
        self._integrated_pb = integrated_hw_graph_pb2.ProtoIntegratedHWGraph()
        self.to_pb(self._integrated_pb.network_graph)
        return self._integrated_pb

    def _get_performance_pb(self, network_fps, clk_freq, ppu_clk_freq, shmifo_clk_freq, optimization_level=None):
        pb = integrated_hw_graph_base_pb2.ProtoPerformance()
        if optimization_level is not None:
            pb.optimization_level = optimization_level.value
        if network_fps is not None:
            pb.network_fps = network_fps
            pb.is_fps_set = True
        else:
            pb.is_fps_set = False
        pb.clk_freq = clk_freq
        if ppu_clk_freq is not None:
            pb.ppu_clk_freq = ppu_clk_freq
        else:
            default_logger().debug("ppu_clk_freq not provided, using default clk_freq")
            pb.ppu_clk_freq = clk_freq
        if shmifo_clk_freq is not None:
            pb.shmifo_clk_freq = shmifo_clk_freq
        else:
            default_logger().debug("shmifo_clk_freq not provided, using default clk_freq")
            pb.shmifo_clk_freq = clk_freq
        return pb

    def save_integrated_pb(
        self,
        pb_path,
        network_fps,
        clk_freq,
        allocator_params=None,
        allocator_script_pb=None,
        nms_metadata_pb=None,
        ppu_clk_freq=None,
        shmifo_clk_freq=None,
        dumps=False,
    ):
        self.to_integrated_pb()
        if allocator_script_pb is not None:
            self._integrated_pb.allocator_script.CopyFrom(allocator_script_pb)
        if allocator_params is not None:
            self._integrated_pb.allocator_params.CopyFrom(allocator_params)
        if nms_metadata_pb is not None:
            # attaches nms_metadata_pb to postprocess node
            nms_postprocess_types = [
                integrated_hw_graph_base_pb2.PROTO_NMS_TYPE,
                integrated_hw_graph_base_pb2.PROTO_BBOX_DECODER_TYPE,
            ]
            nms_postprocess_node = next(
                node
                for node in self._integrated_pb.network_graph.nodes
                if node.postprocess_type in nms_postprocess_types
                and node.type == integrated_hw_graph_base_pb2.PROTO_NETWORK_POSTPROCESS
            )
            nms_postprocess_node.nms_metadata.CopyFrom(nms_metadata_pb)
        self._integrated_pb.performance.CopyFrom(
            self._get_performance_pb(network_fps, clk_freq, ppu_clk_freq, shmifo_clk_freq),
        )
        self._integrated_pb.run_command = " ".join(sys.argv)
        if not dumps:
            with open(pb_path, "wb") as f:
                f.write(self._integrated_pb.SerializeToString())
        return self._integrated_pb

    def save_pb(self, pb_path):
        self.to_pb()
        with open(pb_path, "wb") as f:
            f.write(self._graph.SerializeToString())
        return self._graph

    def _add_nodes(self):
        for pb_index, hn_layer in enumerate(self._hailo_nn.stable_toposort()):
            self._pb_indices_to_layers[pb_index] = hn_layer
            self._layers_to_pb_indices[hn_layer] = pb_index
            pb_wrapper = PbWrapper()
            layer_pb = hn_layer.to_pb(pb_wrapper, self._hailo_nn.is_multi_scope)
            if hn_layer.ew_add_enabled:
                if len(hn_layer.ew_add_connections) != 1:
                    raise ProtobufExportError(
                        "Layer %s has %d elementwise add connections, only 1 is supported"
                        % (str(hn_layer), len(hn_layer.ew_add_connections)),
                    )
                layer_pb.ew_add_pb_index = self._layers_to_pb_indices[hn_layer.ew_add_connections[0]]
            self._graph.nodes.extend([layer_pb])

    def _add_clusters_to_skip(self):
        if self._hailo_nn.net_params.clusters_to_skip:
            for cluster_to_skip in self._hailo_nn.net_params.clusters_to_skip:
                pb_cluster_to_skip = integrated_hw_graph_base_pb2.ProtoCluster()
                if isinstance(cluster_to_skip, str):
                    # "cluster_0", "cluster_1", ..., "cluster_8", "prepost_0"
                    # OR "0", "1", ..., "8"
                    # Pattern validated by JSON Schema
                    if "prepost" in cluster_to_skip:
                        pb_cluster_to_skip.cluster_type = integrated_hw_graph_base_pb2.ProtoClusterType.PROTO_PREPOST
                    else:
                        pb_cluster_to_skip.cluster_type = integrated_hw_graph_base_pb2.ProtoClusterType.PROTO_CLUSTER
                    pb_cluster_to_skip.cluster_index = int(cluster_to_skip[-1])
                else:
                    # 0, 1, 2, ...
                    pb_cluster_to_skip.cluster_type = integrated_hw_graph_base_pb2.ProtoClusterType.PROTO_CLUSTER
                    pb_cluster_to_skip.cluster_index = cluster_to_skip
                self._graph.net_params.clusters_to_skip.append(pb_cluster_to_skip)

    def _add_net_params(self):
        net_params = self._graph.net_params
        for cluster_index, layers in enumerate(self._hailo_nn.net_params.clusters_placement):
            for layer_name in layers:
                placement = net_params.clusters_placement.add()
                # TODO: Add exception if the layer name not in the hn (SDK-30580)
                placement.layer_index = self._layers_to_pb_indices[self._hailo_nn.get_layer_by_name(layer_name)]
                placement.cluster_index = cluster_index
        if self._hailo_nn.net_params.stage:
            self._graph.net_params.stage = PbWrapper.STAGE_TYPE_TO_PB[self._hailo_nn.net_params.stage]
        if self._hailo_nn.net_params.version:
            self._graph.net_params.version = PbWrapper.VERSION_TYPE_TO_PB[self._hailo_nn.net_params.version]
        for out_layer in self._hailo_nn.net_params.output_layers_order:
            self._graph.net_params.output_layers_order.append(self._layer_name_for_pb(out_layer))
        self._graph.net_params.transposed_net = self._hailo_nn.is_transposed()
        self._graph.net_params.output_scale_per_channel = self._hailo_nn.is_output_scale_per_channel()
        for scope in self._hailo_nn.net_params.net_scopes:
            self._graph.net_params.net_scopes.append(scope)
        self._add_clusters_to_skip()

    def _add_edges(self):
        for hn_layer in self._hailo_nn.stable_toposort():
            for output_layer in self._hailo_nn.successors(hn_layer):
                output_index = self._layers_to_pb_indices[output_layer]
                graph_edge = self._graph.edges.add()
                graph_edge.first_index = self._layers_to_pb_indices[hn_layer]
                graph_edge.second_index = output_index

    def _layer_name_for_pb(self, layer_name):
        if self._hailo_nn.is_single_scope:
            return layer_name.split("/")[-1]
        return layer_name


class ProtobufImporter(HailoNNImporter):
    def __init__(self):
        super().__init__()
        self._pb_graph = None
        self._layers = {}
        self._hailo_nn = HailoNN()

    def parse_pb_clusters_placement(self, pb_clusters_placement):
        max_cluster_index = max([node.cluster_index for node in pb_clusters_placement])
        cluster_placements = [[] for _ in range(max_cluster_index + 1)]
        for node in pb_clusters_placement:
            if node.layer_index in self._hailo_nn.layers_by_index:
                cluster_placements[node.cluster_index].append(self._hailo_nn.get_layer_name_by_index(node.layer_index))
        return cluster_placements

    def get_net_params(self):
        net_params_dict = {}
        for field, value in self._pb_graph.net_params.ListFields():
            if field.name == "stage":
                value = PbWrapper.STAGE_PB_TO_TYPE[value].value
            if field.name == "version":
                value = PbWrapper.VERSION_PB_TO_TYPE[value].value
            if field.name == "clusters_placement":
                value = self.parse_pb_clusters_placement(value)
            if field.name == "net_scopes":
                value = list(value)
            net_params_dict[field.name] = value
        return NetParams(net_params_dict)

    def _from_pb(self):
        self._hailo_nn.net_params = self.get_net_params()
        self._layers = {}
        self._add_layers()
        self._add_connections()
        self._handle_ew_add_connections()
        self._handle_concat_connections()
        self._handle_fused_output_indices()
        try:
            self._hailo_nn.calculate_shapes()
            self._hailo_nn.validate_shapes()
        except UnsupportedModelError as e:
            if "Unexpected split shapes" in str(e):
                default_logger().debug(f"{e}")
            else:
                default_logger().warning(f"{e}")
        except Exception as e:
            default_logger().warning(f"{e}")

        if self._hailo_nn.is_single_scope:
            for component in self._hailo_nn.components:
                self._hailo_nn.add_scope_name_to_component(component, self._hailo_nn.net_params.net_scopes[0])

        return self._hailo_nn

    def from_integrated_pb_file(self, pb_path):
        with open(pb_path, "rb") as f:
            return self.from_integrated_pb_data(f.read())

    def from_integrated_pb_data(self, pb_data):
        integrated_hw_graph = integrated_hw_graph_pb2.ProtoIntegratedHWGraph()
        integrated_hw_graph.ParseFromString(pb_data)
        self._pb_graph = integrated_hw_graph.network_graph
        return self._from_pb()

    def from_pb_graph(self, pb_graph):
        self._pb_graph = pb_graph
        return self._from_pb()

    def from_pb(self, pb_path):
        pb_graph = integrated_hw_graph_base_pb2.ProtoNetworkGraph()
        with open(pb_path, "rb") as f:
            pb_graph.ParseFromString(f.read())
        return self.from_pb_graph(pb_graph)

    def from_pb_string(self, pb, hailo_nn_before_fusing=None):
        pb_graph = integrated_hw_graph_base_pb2.ProtoNetworkGraph()
        pb_graph.ParseFromString(pb)
        return self.from_pb_graph(pb_graph, hailo_nn_before_fusing)

    def _add_layers(self):
        pb_wrapper = PbWrapper()
        for layer_pb in self._pb_graph.nodes:
            layer_class = pb_wrapper.PB_TYPE_TO_CLASS[layer_pb.type]
            # some layers (e.g. output layer) exist only in the PB representation
            if layer_class is not None:
                layer_parsed = layer_class.from_pb(layer_pb, pb_wrapper)
                self._layers[layer_pb.index] = layer_parsed
                self._hailo_nn.add_node(layer_parsed)

    def _add_connections(self):
        for layer in self._layers.values():
            for input_index in layer.input_indices:
                input_layer = self._layers[input_index]
                layer.inputs.append(input_layer.name)
                input_layer.outputs.append(layer.name)
                self._hailo_nn.add_edge(input_layer, layer)

    def _handle_ew_add_connections(self):
        for layer_pb in self._pb_graph.nodes:
            layer = self._layers[layer_pb.index]
            if layer_pb.ew_add:
                ew_add_layer = self._layers[layer_pb.ew_add_pb_index]
                conv_ew_add_layer = layer
                conv_ew_add_layer.add_ew_connection(ew_add_layer)
                assert len(conv_ew_add_layer.inputs) == 2
                if conv_ew_add_layer.inputs[1] != ew_add_layer.name:
                    conv_ew_add_layer.inputs = conv_ew_add_layer.inputs[::-1]
            if layer.op in [LayerType.ew_add, LayerType.ew_sub, LayerType.ew_mult, LayerType.ew_div]:
                for index in layer.input_indices:
                    layer.append_to_input_list(self._layers[index])

    def _handle_concat_connections(self):
        for layer_pb in self._pb_graph.nodes:
            layer = self._layers.get(layer_pb.index, None)
            if layer is None:
                continue
            elif layer.op in [
                LayerType.concat,
                LayerType.output_mux,
                LayerType.bbox_decoder,
                LayerType.fused_bbox_decoder,
                LayerType.proposal_generator,
            ]:
                for index in layer.input_indices:
                    layer.append_to_input_list(self._layers[index])

    def _handle_fused_output_indices(self):
        for layer in self._layers.values():
            if (
                layer.op in [LayerType.row_splitter, LayerType.width_splitter, LayerType.feature_splitter]
                and layer.output_indices
            ):
                sorted_outputs = []
                for index in layer.output_indices:
                    sorted_outputs.append(self._layers[index].name)
                layer.outputs = sorted_outputs


class PbWrapper:
    PB_TYPE_TO_CLASS = {
        integrated_hw_graph_base_pb2.PROTO_NETWORK_CONV: FusedConv2DLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_BASE_DECONV: Conv2DLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DECONV: FusedConv2DLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_FEATURE_INTERLEAVE: FeatureInterleaveLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DW: FusedConv2DLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_MAXPOOL: PoolingLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_AVGPOOL: PoolingLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DENSE: FusedDenseLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_LOOPBACK: LoopbackLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_INPUT: InputLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_OUTPUT: OutputLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION: FusedStandaloneActivationLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_BASE_ACTIVATION: ActivationLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_BATCH_NORM: FusedBatchNormLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_BASE_BATCH_NORM: BatchNormLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_BASE_CONV: Conv2DLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_BASE_DW: Conv2DLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_BASE_DENSE: DenseLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_BASE_EW_ADD: EWAddLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_EW_ADD: FusedEWAddLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_STANDALONE_EW_ADD: FusedStandaloneEWAddLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_GLOBAL_AVG_POOL: GlobalAvgPoolingLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_CONCAT: ConcatLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_OUTPUT_MUX: OutputMuxLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_SHORTCUT: ShortcutLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_RESIZE: ResizeLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_PORTAL: PortalLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_FORMAT_CONVERSION: FormatConversionLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_EXTERNAL_INPUT: ExternalInputLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_EXTERNAL_OUTPUT: ExternalOutputLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_PP_INPUT: PPInputLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_PP_OUTPUT: PPOutputLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEPTH_TO_SPACE: DepthToSpaceLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_NORMALIZATION: NormalizationLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ARGMAX: ArgmaxLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_FEATURE_SHUFFLE: FeatureShuffleLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_SOFTMAX: SoftmaxLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_LAYERNORM: LayerNormalizationLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_NMS: NMSLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_BBOX_DECODER: BboxDecoderLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_PROPOSAL_GENERATOR: ProposalGeneratorLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_FEATURE_SPLITTER: FeatureSplitterLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_SLICE: FusedSliceLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_MERGE: MergedLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_BIAS_ADD: BiasAddLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ROW_SPLITTER: RowSplitterLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_WIDTH_SPLITTER: WidthSplitterLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_EW_MULT: EWMultLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_BASE_SLICE: SliceLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_REDUCE_MAX: ReduceMaxLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_SPACE_TO_DEPTH: SpaceToDepthLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEMUX: DemuxLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_EXTERNAL_PAD: ExternalPadLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_MATMUL: MatmulLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_REDUCE_SUM: ReduceSumLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_BASE_EW_SUB: EWSubLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_EW_SUB: FusedEWSubLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_STANDALONE_EW_SUB: FusedStandaloneEWSubLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_FEATURE_MULTIPLIER: FeatureMultiplierLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_CONST_INPUT: ConstInputLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_POSTPROCESS: PostprocessLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_REDUCE_MEAN: ReduceMeanLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_PRECISION_SPLITTER: PrecisionSplitterLayer,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_FUSED_BBOX_DECODER: FusedBboxDecoderLayer,
    }

    STAGE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_STAGE_HN: HnStage.HN,
        integrated_hw_graph_base_pb2.PROTO_STAGE_DEFUSED: HnStage.DEFUSED,
        integrated_hw_graph_base_pb2.PROTO_STAGE_PRE_FUSED: HnStage.PRE_FUSED,
        integrated_hw_graph_base_pb2.PROTO_STAGE_EXPANDED: HnStage.EXPANDED,
        integrated_hw_graph_base_pb2.PROTO_STAGE_SPLIT: HnStage.SPLIT,
    }

    STAGE_TYPE_TO_PB = {v: k for k, v in STAGE_PB_TO_TYPE.items()}

    VERSION_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_VERSION_HN_V1_0: HnVersion.V1_0,
    }

    VERSION_TYPE_TO_PB = {v: k for k, v in VERSION_PB_TO_TYPE.items()}

    ACTIVATION_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_LINEAR: ActivationType.linear,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_RELU: ActivationType.relu,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_RELU6: ActivationType.relu6,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_ELU: ActivationType.elu,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_SIGMOID: ActivationType.sigmoid,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_LEAKY: ActivationType.leaky,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_EXP: ActivationType.exp,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_TANH: ActivationType.tanh,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_THRESHOLD: ActivationType.threshold,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_BIASED_DELTA: ActivationType.biased_delta,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_PRELU: ActivationType.prelu,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_SOFTPLUS: ActivationType.softplus,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_SILU: ActivationType.silu,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_GELU: ActivationType.gelu,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_MISH: ActivationType.mish,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_INV_POS: ActivationType.inv_pos,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_HARDSWISH: ActivationType.hardswish,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_SWISH: ActivationType.swish,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_SQRT: ActivationType.sqrt,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_RELU1: ActivationType.relu1,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_LESS: ActivationType.less,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_LOG: ActivationType.log,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_HARDSIGMOID: ActivationType.hardsigmoid,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_CLIP: ActivationType.clip,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_INV_SQRT: ActivationType.inv_sqrt,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_SOFTSIGN: ActivationType.softsign,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_DELTA: ActivationType.delta,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_GREATER: ActivationType.greater,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_POW: ActivationType.pow,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_HDR_COMPRESSION: ActivationType.hdr_compression,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_RELU_POSITIVE_SQUARE: ActivationType.relu_positive_square,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_PWL: ActivationType.pwl,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_EXP_DECOMPOSE: ActivationType.exp_decompose,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_ACTIVATION_SHIFT: ActivationType.shift,
    }

    ACTIVATION_TYPE_TO_PB = {v: k for k, v in ACTIVATION_PB_TO_TYPE.items()}

    PADDING_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_NETWORK_PADDING_SAME: PaddingType.same,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_PADDING_VALID: PaddingType.valid,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_PADDING_SAME_TENSORFLOW: PaddingType.same_tensorflow,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_PADDING_DECONV: PaddingType.deconv,
    }
    PADDING_TYPE_TO_PB = {v: k for k, v in PADDING_PB_TO_TYPE.items()}

    DEFUSE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_NONE: DefuseType.none,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_NORMAL: DefuseType.normal,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_DECONV: DefuseType.deconv,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_MAXPOOL: DefuseType.maxpool,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_SPATIAL_W: DefuseType.spatial_w,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_COMPUTE_LANES: DefuseType.compute_lanes,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_PROPOSAL_GENERATOR_ANCHORS: DefuseType.proposal_generator,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_NMS: DefuseType.nms,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_PORTAL_L4: DefuseType.portal_l4,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_PORTAL_DDR: DefuseType.portal_ddr,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_RESIZE: DefuseType.resize,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_INPUT_FEATURES: DefuseType.input_features,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_SUPER_CONV: DefuseType.super_conv,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_ARCH_REQUIRED: DefuseType.arch_required,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_DEPTH_TO_SPACE: DefuseType.depth_to_space,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_SPACE_TO_DEPTH: DefuseType.space_to_depth,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_RESIZE_TRANSPOSE: DefuseType.resize_transpose,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_SUPER_DW: DefuseType.super_dw,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_CONST_INPUT: DefuseType.const_input,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_GLOBAL_AVGPOOL_TRANSPOSED_INPUT: DefuseType.global_avgpool_transposed_input,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_SPATIAL_RESHAPE: DefuseType.spatial_reshape,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_EW_MULT_ON_MAC: DefuseType.ew_mult_on_mac,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_DOUBLE_PRECISION_CONV: DefuseType.double_precision_conv,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_NV: DefuseType.nv,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_I420: DefuseType.i420,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_DYNAMIC_WEIGHTS: DefuseType.dynamic_weights,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_L3_PORTAL: DefuseType.l3_portal,
        integrated_hw_graph_base_pb2.PROTO_NETWORK_DEFUSE_CONV_WEIGHT_GROUPS: DefuseType.conv_weight_groups,
    }
    DEFUSE_TYPE_TO_PB = {v: k for k, v in DEFUSE_PB_TO_TYPE.items()}

    RESOURCES_ALLOCATION_STRATEGY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_AUTOMATIC_SCS_SELECTION: ResourcesAllocationStrategy.automatic_scs_selection,
        integrated_hw_graph_base_pb2.PROTO_MANUAL_SCS_SELECTION: ResourcesAllocationStrategy.manual_scs_selection,
    }
    RESOURCES_ALLOCATION_STRATEGY_TYPE_TO_PB = {
        ResourcesAllocationStrategy.automatic_scs_selection: integrated_hw_graph_base_pb2.PROTO_AUTOMATIC_SCS_SELECTION,
        ResourcesAllocationStrategy.manual_scs_selection: integrated_hw_graph_base_pb2.PROTO_MANUAL_SCS_SELECTION,
        ResourcesAllocationStrategy.min_scs_add_mem: integrated_hw_graph_base_pb2.PROTO_AUTOMATIC_SCS_SELECTION,
        ResourcesAllocationStrategy.min_scs_match_fps: integrated_hw_graph_base_pb2.PROTO_AUTOMATIC_SCS_SELECTION,
        ResourcesAllocationStrategy.min_l3_mem_match_fps: integrated_hw_graph_base_pb2.PROTO_AUTOMATIC_SCS_SELECTION,
    }

    OUTPUT_MIN_MAX_STRATEGY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_OUTPUT_MIN_MAX_SIGMOID: OutputMinMaxStrategy.sigmoid,
        integrated_hw_graph_base_pb2.PROTO_OUTPUT_MIN_MAX_SOFTMAX: OutputMinMaxStrategy.softmax,
        integrated_hw_graph_base_pb2.PROTO_OUTPUT_MIN_MAX_DEFAULT: OutputMinMaxStrategy.default,
    }
    OUTPUT_MIN_MAX_STRATEGY_TYPE_TO_PB = {v: k for k, v in OUTPUT_MIN_MAX_STRATEGY_PB_TO_TYPE.items()}

    RESIZE_METHOD_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_RESIZE_BILINEAR: ResizeMethod.bilinear,
        integrated_hw_graph_base_pb2.PROTO_RESIZE_NEAREST_NEIGHBOR: ResizeMethod.nearest_neighbor,
    }
    RESIZE_METHOD_TYPE_TO_PB = {v: k for k, v in RESIZE_METHOD_PB_TO_TYPE.items()}

    SPACE_TO_DEPTH_TYPE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_CLASSIC_DCR_TYPE: SpaceToDepthType.classic_dcr,
        integrated_hw_graph_base_pb2.PROTO_CLASSIC_CRD_TYPE: SpaceToDepthType.classic_crd,
        integrated_hw_graph_base_pb2.PROTO_FOCUS_TYPE: SpaceToDepthType.focus,
        integrated_hw_graph_base_pb2.PROTO_SERIAL_TYPE: SpaceToDepthType.serial,
    }

    SPACE_TO_DEPTH_TYPE_TYPE_TO_PB = {v: k for k, v in SPACE_TO_DEPTH_TYPE_PB_TO_TYPE.items()}

    DEPTH_TO_SPACE_TYPE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_DCR_TYPE: DepthToSpaceType.dcr,
        integrated_hw_graph_base_pb2.PROTO_CRD_TYPE: DepthToSpaceType.crd,
    }

    DEPTH_TO_SPACE_TYPE_TYPE_TO_PB = {v: k for k, v in DEPTH_TO_SPACE_TYPE_PB_TO_TYPE.items()}

    LOGITS_TYPE_TO_PB = {
        LayerType.argmax: integrated_hw_graph_base_pb2.PROTO_ARGMAX_TYPE,
        LayerType.softmax: integrated_hw_graph_base_pb2.PROTO_SOFTMAX_TYPE,
    }

    META_ARCH_TYPE_TO_PB = {
        NMSMetaArchitectures.YOLOV5: integrated_hw_graph_base_pb2.PROTO_META_ARCH_YOLOV5,
        NMSMetaArchitectures.YOLOX: integrated_hw_graph_base_pb2.PROTO_META_ARCH_YOLOX,
        NMSMetaArchitectures.SSD: integrated_hw_graph_base_pb2.PROTO_META_ARCH_SSD,
        NMSMetaArchitectures.YOLOV5_SEG: integrated_hw_graph_base_pb2.PROTO_META_ARCH_YOLOV5_SEG,
        NMSMetaArchitectures.YOLOV8: integrated_hw_graph_base_pb2.PROTO_META_ARCH_YOLOV8,
    }

    POSTPROCESS_TYPE_TO_PB = {
        PostprocessType.NMS: integrated_hw_graph_base_pb2.PROTO_NMS_TYPE,
        PostprocessType.LOGITS: integrated_hw_graph_base_pb2.PROTO_LOGITS_TYPE,
        PostprocessType.IOU: integrated_hw_graph_base_pb2.PROTO_IOU_TYPE,
        PostprocessType.RESIZE: integrated_hw_graph_base_pb2.PROTO_RESIZE_TYPE,
        PostprocessType.BBOX_DECODER: integrated_hw_graph_base_pb2.PROTO_BBOX_DECODER_TYPE,
    }

    ENGINE_TYPE_TO_PB = {
        PostprocessTarget.NN_CORE: integrated_hw_graph_base_pb2.PROTO_NN_CORE,
        PostprocessTarget.AUTO: integrated_hw_graph_base_pb2.PROTO_AUTO,
        PostprocessTarget.CPU: integrated_hw_graph_base_pb2.PROTO_CPU,
    }

    ENGINE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_NN_CORE: PostprocessTarget.NN_CORE,
        integrated_hw_graph_base_pb2.PROTO_AUTO: PostprocessTarget.AUTO,
        integrated_hw_graph_base_pb2.PROTO_CPU: PostprocessTarget.CPU,
    }

    CONVERSION_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_PRE_INFER_TF_RGB: FormatConversionType.tf_rgb_to_hailo_rgb,
        integrated_hw_graph_base_pb2.PROTO_PRE_INFER_TF_RGBX: FormatConversionType.tf_rgbx_to_hailo_rgb,
        integrated_hw_graph_base_pb2.PROTO_PRE_INFER_MIPI_RGB888: FormatConversionType.mipi_rgb888_to_hailo_rgb,
        integrated_hw_graph_base_pb2.PROTO_PRE_INFER_MIPI_BAYER_RGGB_RGB: FormatConversionType.mipi_bayer_rggb_to_hailo_rgb,
        integrated_hw_graph_base_pb2.PROTO_PRE_INFER_MIPI_BAYER_BGGR_RGB: FormatConversionType.mipi_bayer_bggr_to_hailo_rgb,
        integrated_hw_graph_base_pb2.PROTO_PRE_INFER_MIPI_BAYER_GRBG_RGB: FormatConversionType.mipi_bayer_grbg_to_hailo_rgb,
        integrated_hw_graph_base_pb2.PROTO_PRE_INFER_MIPI_BAYER_GBRG_RGB: FormatConversionType.mipi_bayer_gbrg_to_hailo_rgb,
        integrated_hw_graph_base_pb2.PROTO_HAILO_RGB_TO_TF_RGB: FormatConversionType.hailo_rgb_to_tf_rgb,
        integrated_hw_graph_base_pb2.PROTO_YUY2_TO_HAILO_YUV: FormatConversionType.yuy2_to_hailo_yuv,
        integrated_hw_graph_base_pb2.PROTO_TWELVE_TO_EIGHT_BIT: FormatConversionType.twelve_to_eight_bit,
        integrated_hw_graph_base_pb2.PROTO_TWELVE_TO_SIXTEEN_BIT: FormatConversionType.twelve_to_sixteen_bit,
        integrated_hw_graph_base_pb2.PROTO_SIXTEEN_TO_TWELVE_BIT: FormatConversionType.sixteen_to_twelve_bit,
        integrated_hw_graph_base_pb2.PROTO_HAILO_RGB_TO_PPU: FormatConversionType.hailo_rgb_to_ppu,
        integrated_hw_graph_base_pb2.PROTO_PPU_TO_HAILO_RGB: FormatConversionType.ppu_to_hailo_rgb,
        integrated_hw_graph_base_pb2.PROTO_FEATURES_TO_WIDTH_FEATURES_RESHAPE: FormatConversionType.features_to_width_features,
        integrated_hw_graph_base_pb2.PROTO_HAILO_RGB_TO_F8CR: FormatConversionType.hailo_rgb_to_f8cr,
        integrated_hw_graph_base_pb2.PROTO_F8CR_TO_HAILO_RGB: FormatConversionType.f8cr_to_hailo_rgb,
        integrated_hw_graph_base_pb2.PROTO_FLAT_TO_FRAMES_RESHAPE: FormatConversionType.flat_to_frames,
        integrated_hw_graph_base_pb2.PROTO_FRAMES_TO_FLAT: FormatConversionType.frames_to_flat,
        integrated_hw_graph_base_pb2.PROTO_TRANSPOSE_WIDTH_FEATURES: FormatConversionType.transpose_width_features,
        integrated_hw_graph_base_pb2.PROTO_TRANSPOSE_MATMUL: FormatConversionType.transpose_matmul,
        integrated_hw_graph_base_pb2.PROTO_SPATIAL_FLATTEN: FormatConversionType.spatial_flatten,
        integrated_hw_graph_base_pb2.PROTO_SPATIAL_EXPAND: FormatConversionType.spatial_expand,
        integrated_hw_graph_base_pb2.PROTO_SPATIAL_RESHAPE: FormatConversionType.spatial_reshape,
        integrated_hw_graph_base_pb2.PROTO_HXF_TO_W_TRANSPOSED: FormatConversionType.hxf_to_w_transposed,
        integrated_hw_graph_base_pb2.PROTO_F_TO_HXW_TRANSPOSED: FormatConversionType.f_to_hxw_transposed,
        integrated_hw_graph_base_pb2.PROTO_FCR_TO_C8FR: FormatConversionType.fcr_to_c8fr,
        integrated_hw_graph_base_pb2.PROTO_F8CR_TO_FCR: FormatConversionType.f8cr_to_fcr,
        integrated_hw_graph_base_pb2.PROTO_C8FR_TO_FRAMES: FormatConversionType.c8fr_to_frames,
        integrated_hw_graph_base_pb2.PROTO_RESHAPE_1XW0_TO_HXW: FormatConversionType.reshape_1xw0_to_hxw,
        integrated_hw_graph_base_pb2.PROTO_NV12_TO_HAILO_YUV: FormatConversionType.nv12_to_hailo_yuv,
        integrated_hw_graph_base_pb2.PROTO_NV21_TO_HAILO_YUV: FormatConversionType.nv21_to_hailo_yuv,
        integrated_hw_graph_base_pb2.PROTO_HAILO_RGB_TO_LCU: FormatConversionType.hailo_rgb_to_lcu,
        integrated_hw_graph_base_pb2.PROTO_I420_TO_HAILO_YUV: FormatConversionType.i420_to_hailo_yuv,
        integrated_hw_graph_base_pb2.PROTO_TRANSPOSE_HEIGHT_WIDTH: FormatConversionType.transpose_height_width,
        integrated_hw_graph_base_pb2.PROTO_RESHAPE_HEIGHT_FEATURES: FormatConversionType.reshape_height_features,
        integrated_hw_graph_base_pb2.PROTO_RESHAPE_POST_EW_MULT: FormatConversionType.reshape_post_ew_mult,
    }
    CONVERSION_TYPE_TO_PB = {v: k for k, v in CONVERSION_PB_TO_TYPE.items()}

    HW_LAYER_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_RESIZE_BILINEAR: HWLayerType.lcu,
        integrated_hw_graph_base_pb2.PROTO_RESIZE_NEAREST_NEIGHBOR: HWLayerType.ppu,
    }
    HW_LAYER_TYPE_TO_PB = {v: k for k, v in HW_LAYER_PB_TO_TYPE.items()}

    SUBCLUSTERS_16x4_POLICY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_16x4_ALLOWED: Subclusters16x4Policy.allowed,
        integrated_hw_graph_base_pb2.PROTO_16x4_DISABLED: Subclusters16x4Policy.disabled,
        integrated_hw_graph_base_pb2.PROTO_16x4_ENABLED: Subclusters16x4Policy.enabled,
    }
    SUBCLUSTERS_16x4_POLICY_TYPE_TO_PB = {v: k for k, v in SUBCLUSTERS_16x4_POLICY_PB_TO_TYPE.items()}

    L2_WEIGHTS_POLICY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_USE_L2_WEIGHTS_ALLOWED: UseL2WeightsPolicy.allowed,
        integrated_hw_graph_base_pb2.PROTO_USE_L2_WEIGHTS_DISABLED: UseL2WeightsPolicy.disabled,
        integrated_hw_graph_base_pb2.PROTO_USE_L2_WEIGHTS_ENABLED: UseL2WeightsPolicy.enabled,
    }
    L2_WEIGHTS_POLICY_TYPE_TO_PB = {v: k for k, v in L2_WEIGHTS_POLICY_PB_TO_TYPE.items()}

    COMPRESS_L3_WEIGHTS_POLICY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_COMPRESS_L3_WEIGHTS_ALLOWED: CompressL3WeightsPolicy.allowed,
        integrated_hw_graph_base_pb2.PROTO_COMPRESS_L3_WEIGHTS_DISABLED: CompressL3WeightsPolicy.disabled,
        integrated_hw_graph_base_pb2.PROTO_COMPRESS_L3_WEIGHTS_ENABLED: CompressL3WeightsPolicy.enabled,
    }
    COMPRESS_L3_WEIGHTS_POLICY_TYPE_TO_PB = {v: k for k, v in COMPRESS_L3_WEIGHTS_POLICY_PB_TO_TYPE.items()}

    ENABLE_EW_ADD_OPTIMZATION_POLICY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_ENABLE_EW_ADD_OPTIMIZATION_ALLOWED: EnableEwAddOptimizationPolicy.allowed,
        integrated_hw_graph_base_pb2.PROTO_ENABLE_EW_ADD_OPTIMIZATION_DISABLED: EnableEwAddOptimizationPolicy.disabled,
        integrated_hw_graph_base_pb2.PROTO_ENABLE_EW_ADD_OPTIMIZATION_ENABLED: EnableEwAddOptimizationPolicy.enabled,
    }
    ENABLE_EW_ADD_OPTIMZATION_POLICY_TYPE_TO_PB = {v: k for k, v in ENABLE_EW_ADD_OPTIMZATION_POLICY_PB_TO_TYPE.items()}

    PARALLEL_ACTIVATION_POLICY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_PARALLEL_ACTIVATION_ALLOWED: ParallelActivationPolicy.allowed,
        integrated_hw_graph_base_pb2.PROTO_PARALLEL_ACTIVATION_DISABLED: ParallelActivationPolicy.disabled,
        integrated_hw_graph_base_pb2.PROTO_PARALLEL_ACTIVATION_ENABLED: ParallelActivationPolicy.enabled,
    }
    PARALLEL_ACTIVATION_POLICY_TYPE_TO_PB = {v: k for k, v in PARALLEL_ACTIVATION_POLICY_PB_TO_TYPE.items()}

    SUBCLUSTERS_NO_CONTEXTS_POLICY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_NO_CONTEXTS_ALLOWED: SubclustersNoContextsPolicy.allowed,
        integrated_hw_graph_base_pb2.PROTO_NO_CONTEXTS_DISABLED: SubclustersNoContextsPolicy.disabled,
        integrated_hw_graph_base_pb2.PROTO_NO_CONTEXTS_ENABLED: SubclustersNoContextsPolicy.enabled,
    }

    SUBCLUSTERS_NO_CONTEXTS_POLICY_TYPE_TO_PB = {v: k for k, v in SUBCLUSTERS_NO_CONTEXTS_POLICY_PB_TO_TYPE.items()}

    BALANCE_OUTPUT_MULTISPLIT_POLICY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_BALANCE_OUTPUT_MULTISPLIT_ALLOWED: BalanceOutputMultisplitPolicy.allowed,
        integrated_hw_graph_base_pb2.PROTO_BALANCE_OUTPUT_MULTISPLIT_DISABLED: BalanceOutputMultisplitPolicy.disabled,
        integrated_hw_graph_base_pb2.PROTO_BALANCE_OUTPUT_MULTISPLIT_ENABLED: BalanceOutputMultisplitPolicy.enabled,
    }
    BALANCE_OUTPUT_MULTISPLIT_POLICY_TYPE_TO_PB = {v: k for k, v in BALANCE_OUTPUT_MULTISPLIT_POLICY_PB_TO_TYPE.items()}

    RESIZE_BILINEAR_STREAMING_POLICY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_RESIZE_BILINEAR_STREAMING_ALLOWED: ResizeBilinearStreamingPolicy.allowed,
        integrated_hw_graph_base_pb2.PROTO_RESIZE_BILINEAR_STREAMING_DISABLED: ResizeBilinearStreamingPolicy.disabled,
        integrated_hw_graph_base_pb2.PROTO_RESIZE_BILINEAR_STREAMING_ENABLED: ResizeBilinearStreamingPolicy.enabled,
    }

    RESIZE_BILINEAR_STREAMING_POLICY_TYPE_TO_PB = {v: k for k, v in RESIZE_BILINEAR_STREAMING_POLICY_PB_TO_TYPE.items()}

    RESIZE_BILINEAR_PIXELS_MODE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_RESIZE_BILINEAR_PIXELS_MODE_DISABLED: ResizeBilinearPixelsMode.disabled,
        integrated_hw_graph_base_pb2.PROTO_RESIZE_BILINEAR_PIXELS_MODE_ALIGN_CORNERS: ResizeBilinearPixelsMode.align_corners,
        integrated_hw_graph_base_pb2.PROTO_RESIZE_BILINEAR_PIXELS_MODE_HALF_PIXELS: ResizeBilinearPixelsMode.half_pixels,
    }

    RESIZE_BILINEAR_PIXELS_MODE_TYPE_TO_PB = {v: k for k, v in RESIZE_BILINEAR_PIXELS_MODE_PB_TO_TYPE.items()}

    TWO_LINE_BUFFER_MODE_POLICY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_TWO_LINE_BUFFER_MODE_ALLOWED: TwoLineBufferModePolicy.allowed,
        integrated_hw_graph_base_pb2.PROTO_TWO_LINE_BUFFER_MODE_DISABLED: TwoLineBufferModePolicy.disabled,
        integrated_hw_graph_base_pb2.PROTO_TWO_LINE_BUFFER_MODE_ENABLED: TwoLineBufferModePolicy.enabled,
    }

    TWO_LINE_BUFFER_MODE_POLICY_TYPE_TO_PB = {v: k for k, v in TWO_LINE_BUFFER_MODE_POLICY_PB_TO_TYPE.items()}

    PRECISION_SPLIT_MODE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_PRECISION_SPLIT_MODE_NORMAL: PrecisionSplitMode.NORMAL,
        integrated_hw_graph_base_pb2.PROTO_PRECISION_SPLIT_MODE_PIXELS: PrecisionSplitMode.PIXELS,
    }
    PRECISION_SPLIT_MODE_TYPE_TO_PB = {v: k for k, v in PRECISION_SPLIT_MODE_PB_TO_TYPE.items()}

    BIAS_MODE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_SINGLE_SCALE_DECOMPOSITION: BiasMode.single_scale_decomposition,
        integrated_hw_graph_base_pb2.PROTO_DOUBLE_SCALE_INITIALIZATION: BiasMode.double_scale_initialization,
        integrated_hw_graph_base_pb2.PROTO_DOUBLE_SCALE_DECOMPOSITION: BiasMode.double_scale_decomposition,
    }
    BIAS_MODE_TYPE_TO_PB = {v: k for k, v in BIAS_MODE_PB_TO_TYPE.items()}

    def DumpPrecisionMode(self, precision_mode):
        pb_precision_mode = integrated_hw_graph_base_pb2.ProtoNodePrecisionMode()
        if precision_mode is None:
            pb_precision_mode

        if precision_mode.input_precision_mode() in [
            PrecisionMode.a8_w8_a8,
            PrecisionMode.a8_w8_a16,
            PrecisionMode.a8_w4_a8,
            PrecisionMode.a8_w4_a16,
        ]:
            pb_precision_mode.input_data_precision_mode = integrated_hw_graph_base_pb2.PROTO_DATA_PRECISION_MODE_8BIT
        elif precision_mode.input_precision_mode() in [
            PrecisionMode.a16_w16_a16,
            PrecisionMode.a16_w16_a8,
            PrecisionMode.a16_w8_a8,
            PrecisionMode.a16_w8_a16,
            PrecisionMode.a16_w4_a16,
            PrecisionMode.a16_w4_a8,
        ]:
            pb_precision_mode.input_data_precision_mode = integrated_hw_graph_base_pb2.PROTO_DATA_PRECISION_MODE_16BIT
        else:
            raise ProtobufExportError(f"Unsupported input precision mode: {precision_mode.input_precision_mode()}")

        if precision_mode.output_precision_mode() in [
            PrecisionMode.a8_w8_a8,
            PrecisionMode.a8_w4_a8,
            PrecisionMode.a16_w16_a8,
            PrecisionMode.a16_w8_a8,
            PrecisionMode.a16_w4_a8,
        ]:
            pb_precision_mode.output_data_precision_mode = integrated_hw_graph_base_pb2.PROTO_DATA_PRECISION_MODE_8BIT
        elif precision_mode.output_precision_mode() in [
            PrecisionMode.a16_w16_a16,
            PrecisionMode.a8_w8_a16,
            PrecisionMode.a8_w4_a16,
            PrecisionMode.a16_w8_a16,
            PrecisionMode.a16_w4_a16,
        ]:
            pb_precision_mode.output_data_precision_mode = integrated_hw_graph_base_pb2.PROTO_DATA_PRECISION_MODE_16BIT
        else:
            raise ProtobufExportError(f"Unsupported output precision mode: {precision_mode.output_precision_mode()}")

        if precision_mode == PrecisionMode.a8_w4_exp:
            pb_precision_mode.weights_precision_mode = (
                integrated_hw_graph_base_pb2.PROTO_WEIGHTS_PRECISION_MODE_4BIT_EXPONENTIAL
            )
        elif precision_mode.weight_bits() == 8:
            pb_precision_mode.weights_precision_mode = integrated_hw_graph_base_pb2.PROTO_WEIGHTS_PRECISION_MODE_8BIT
        elif precision_mode.weight_bits() == 4:
            pb_precision_mode.weights_precision_mode = integrated_hw_graph_base_pb2.PROTO_WEIGHTS_PRECISION_MODE_4BIT
        elif precision_mode.weight_bits() == 16:
            pb_precision_mode.weights_precision_mode = integrated_hw_graph_base_pb2.PROTO_WEIGHTS_PRECISION_MODE_16BIT
        else:
            raise ProtobufExportError(f"Unsupported weight precision mode: {precision_mode.weight_bits()}")

        return pb_precision_mode

    def LoadPrecisionMode(self, proto_precision_mode):
        pb = integrated_hw_graph_base_pb2
        A8 = pb.PROTO_DATA_PRECISION_MODE_8BIT
        A16 = pb.PROTO_DATA_PRECISION_MODE_16BIT
        W4 = pb.PROTO_WEIGHTS_PRECISION_MODE_4BIT
        W8 = pb.PROTO_WEIGHTS_PRECISION_MODE_8BIT
        W16 = pb.PROTO_WEIGHTS_PRECISION_MODE_16BIT

        PRECISION_MODE_MAPPING = {
            (A8, W8, A8): PrecisionMode.a8_w8_a8,
            (A8, W4, A8): PrecisionMode.a8_w4_a8,
            (A8, W8, A16): PrecisionMode.a8_w8_a16,
            (A8, W4, A16): PrecisionMode.a8_w4_a16,
            (A16, W8, A8): PrecisionMode.a16_w8_a8,
            (A16, W16, A8): PrecisionMode.a16_w16_a8,
            (A16, W4, A8): PrecisionMode.a16_w4_a8,
            (A16, W4, A16): PrecisionMode.a16_w4_a16,
            (A16, W8, A16): PrecisionMode.a16_w8_a16,
            (A16, W16, A16): PrecisionMode.a16_w16_a16,
        }

        key = (
            proto_precision_mode.input_data_precision_mode,
            proto_precision_mode.weights_precision_mode,
            proto_precision_mode.output_data_precision_mode,
        )
        if key not in PRECISION_MODE_MAPPING:
            raise ProtobufExportError(f"Unsupported precision mode: {key}")

        return PRECISION_MODE_MAPPING[key]

    STR_TO_BOOL_TO_PB = {
        True: True,
        "True": True,
        False: False,
        "False": False,
    }

    ALLOCATOR_AGENT_TYPE_TO_PB = {
        AllocatorAgent.LAZY: integrated_hw_graph_base_pb2.ProtoAllocatorAgent.PROTO_ALLOCATOR_AGENT_LAZY,
        AllocatorAgent.MELG: integrated_hw_graph_base_pb2.ProtoAllocatorAgent.PROTO_ALLOCATOR_AGENT_MELG,
        AllocatorAgent.MIGDALODON: integrated_hw_graph_base_pb2.ProtoAllocatorAgent.PROTO_ALLOCATOR_AGENT_MIGDALODON,
        AllocatorAgent.AUTOMATIC: integrated_hw_graph_base_pb2.ProtoAllocatorAgent.PROTO_ALLOCATOR_AGENT_AUTOMATIC,
    }

    AUTO_VARIABLE_POLICY_TO_PB = {
        AutoVariablePolicy.MANUAL: integrated_hw_graph_base_pb2.ProtoAutoVariablePolicy.PROTO_MANUAL,
        AutoVariablePolicy.AUTOMATIC: integrated_hw_graph_base_pb2.ProtoAutoVariablePolicy.PROTO_AUTOMATIC,
    }

    ALLOCATOR_STRATEGY_TYPE_TO_PB = {
        AllocatorStrategy.LAZY: integrated_hw_graph_base_pb2.ProtoAllocatorStrategy.PROTO_LAZY,
        AllocatorStrategy.BLIND_SEARCH: integrated_hw_graph_base_pb2.ProtoAllocatorStrategy.PROTO_BLIND_SEARCH,
        AllocatorStrategy.POSITIVE_SEARCH: integrated_hw_graph_base_pb2.ProtoAllocatorStrategy.PROTO_POSITIVE_SEARCH,
    }

    PARTIAL_ROW_BUFFERS_TYPE_TO_PB = {
        PartialRowBuffersType.PARTIAL_ROW_BUFFERS_CONSTRAINED: integrated_hw_graph_base_pb2.ProtoPartialRowBuffers.PROTO_PARTIAL_ROW_BUFFERS_CONSTRAINED,
        PartialRowBuffersType.PARTIAL_ROW_BUFFERS_ENABLED: integrated_hw_graph_base_pb2.ProtoPartialRowBuffers.PROTO_PARTIAL_ROW_BUFFERS_ENABLED,
        PartialRowBuffersType.PARTIAL_ROW_BUFFERS_DISABLED: integrated_hw_graph_base_pb2.ProtoPartialRowBuffers.PROTO_PARTIAL_ROW_BUFFERS_DISABLED,
    }
    PARTIAL_ROW_BUFFERS_PB_TO_TYPE = {v: k for k, v in PARTIAL_ROW_BUFFERS_TYPE_TO_PB.items()}

    ENABLE_FIXER_POLICY_TYPE_TO_PB = {
        EnableFixerPolicy.DISABLED: integrated_hw_graph_base_pb2.ProtoEnableFixer.PROTO_ENABLE_FIXER_DISABLED,
        EnableFixerPolicy.ENABLED: integrated_hw_graph_base_pb2.ProtoEnableFixer.PROTO_ENABLE_FIXER_ENABLED,
        EnableFixerPolicy.MAX_ADJACENTS: integrated_hw_graph_base_pb2.ProtoEnableFixer.PROTO_ENABLE_FIXER_MAX_ADJACENTS,
    }
    ENABLE_FIXER_PB_TO_TYPE = {v: k for k, v in ENABLE_FIXER_POLICY_TYPE_TO_PB.items()}

    BUILDER_EXIT_POINT_TYPE_TO_PB = {
        BuilderExitPoint.POST_WINDOWS: integrated_hw_graph_base_pb2.ProtoBuilderExitPoint.PROTO_BUILDER_EXIT_POINT_POST_WINDOWS,
        BuilderExitPoint.POST_RACEHORSE: integrated_hw_graph_base_pb2.ProtoBuilderExitPoint.PROTO_BUILDER_EXIT_POINT_POST_RACEHORSE,
        BuilderExitPoint.POST_DEFUSE: integrated_hw_graph_base_pb2.ProtoBuilderExitPoint.PROTO_BUILDER_EXIT_POINT_POST_DEFUSE,
        BuilderExitPoint.POST_EXPANSION: integrated_hw_graph_base_pb2.ProtoBuilderExitPoint.PROTO_BUILDER_EXIT_POINT_POST_EXPANSION,
        BuilderExitPoint.POST_PARTITION: integrated_hw_graph_base_pb2.ProtoBuilderExitPoint.PROTO_BUILDER_EXIT_POINT_POST_PARTITION,
        BuilderExitPoint.POST_SPLIT: integrated_hw_graph_base_pb2.ProtoBuilderExitPoint.PROTO_BUILDER_EXIT_POINT_POST_SPLIT,
        BuilderExitPoint.POST_ALLOCATION: integrated_hw_graph_base_pb2.ProtoBuilderExitPoint.PROTO_BUILDER_EXIT_POINT_POST_ALLOCATION,
        BuilderExitPoint.POST_COMPILATION: integrated_hw_graph_base_pb2.ProtoBuilderExitPoint.PROTO_BUILDER_EXIT_POINT_POST_COMPILATION,
        BuilderExitPoint.POST_CAT: integrated_hw_graph_base_pb2.ProtoBuilderExitPoint.PROTO_BUILDER_EXIT_POINT_POST_CAT,
        BuilderExitPoint.POST_FORMULA: integrated_hw_graph_base_pb2.ProtoBuilderExitPoint.PROTO_BUILDER_EXIT_POINT_POST_FORMULA,
    }

    CONTEXT_SWITCH_MODE_TYPE_TO_PB = {
        ContextSwitchMode.MANUAL: integrated_hw_graph_base_pb2.ProtoContextSwitchMode.PROTO_CONTEXT_SWITCH_MODE_MANUAL,
        ContextSwitchMode.ALLOWED: integrated_hw_graph_base_pb2.ProtoContextSwitchMode.PROTO_CONTEXT_SWITCH_MODE_ALLOWED,
        ContextSwitchMode.ENABLED: integrated_hw_graph_base_pb2.ProtoContextSwitchMode.PROTO_CONTEXT_SWITCH_MODE_ENABLED,
        ContextSwitchMode.DISABLED: integrated_hw_graph_base_pb2.ProtoContextSwitchMode.PROTO_CONTEXT_SWITCH_MODE_DISABLED,
    }

    CONTEXT_SWITCH_PARTITIONER_TYPE_TO_PB = {
        Partitioner.SLOTH: integrated_hw_graph_base_pb2.ProtoContextSwitchPartitioner.PROTO_CONTEXT_SWITCH_PARTITIONER_SLOTH,
        Partitioner.SLOTTER: integrated_hw_graph_base_pb2.ProtoContextSwitchPartitioner.PROTO_CONTEXT_SWITCH_PARTITIONER_SLOTTER,
    }

    TOPOSORT_MODE_TYPE_TO_PB = {
        ToposortMode.DFS: integrated_hw_graph_base_pb2.ProtoToposortMode.PROTO_TOPOSORT_MODE_DFS,
        ToposortMode.DEPTHWISE: integrated_hw_graph_base_pb2.ProtoToposortMode.PROTO_TOPOSORT_MODE_DEPTHWISE,
    }

    LOGGER_VERBOSITY_TYPE_TO_PB = {
        LoggerVerbosity.TRACE: integrated_hw_graph_base_pb2.ProtoLoggerVerbosity.PROTO_LOGGER_VERBOSITY_TRACE,
        LoggerVerbosity.DEBUG: integrated_hw_graph_base_pb2.ProtoLoggerVerbosity.PROTO_LOGGER_VERBOSITY_DEBUG,
        LoggerVerbosity.INFO: integrated_hw_graph_base_pb2.ProtoLoggerVerbosity.PROTO_LOGGER_VERBOSITY_INFO,
        LoggerVerbosity.WARNING: integrated_hw_graph_base_pb2.ProtoLoggerVerbosity.PROTO_LOGGER_VERBOSITY_WARNING,
        LoggerVerbosity.ERROR: integrated_hw_graph_base_pb2.ProtoLoggerVerbosity.PROTO_LOGGER_VERBOSITY_ERROR,
        LoggerVerbosity.CRITICAL: integrated_hw_graph_base_pb2.ProtoLoggerVerbosity.PROTO_LOGGER_VERBOSITY_CRITICAL,
        LoggerVerbosity.OFF: integrated_hw_graph_base_pb2.ProtoLoggerVerbosity.PROTO_LOGGER_VERBOSITY_OFF,
    }

    RESOLVER_REVERT_STRATEGY_TYPE_TO_PB = {
        ResolverRevertStrategy.PATIENT: integrated_hw_graph_base_pb2.ProtoResolverRevertStrategy.PROTO_RESOLVER_REVERT_STRATEGY_PATIENT,
        ResolverRevertStrategy.ASAP: integrated_hw_graph_base_pb2.ProtoResolverRevertStrategy.PROTO_RESOLVER_REVERT_STRATEGY_ASAP,
    }

    RESOLVER_REVERT_STRATEGY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.ProtoResolverRevertStrategy.PROTO_RESOLVER_REVERT_STRATEGY_PATIENT: ResolverRevertStrategy.PATIENT,
        integrated_hw_graph_base_pb2.ProtoResolverRevertStrategy.PROTO_RESOLVER_REVERT_STRATEGY_ASAP: ResolverRevertStrategy.ASAP,
    }

    ALLOCATOR_AGENT_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.ProtoAllocatorAgent.PROTO_ALLOCATOR_AGENT_LAZY: AllocatorAgent.LAZY,
        integrated_hw_graph_base_pb2.ProtoAllocatorAgent.PROTO_ALLOCATOR_AGENT_MELG: AllocatorAgent.MELG,
        integrated_hw_graph_base_pb2.ProtoAllocatorAgent.PROTO_ALLOCATOR_AGENT_MIGDALODON: AllocatorAgent.MIGDALODON,
        integrated_hw_graph_base_pb2.ProtoAllocatorAgent.PROTO_ALLOCATOR_AGENT_AUTOMATIC: AllocatorAgent.AUTOMATIC,
    }

    ALLOCATOR_STRATEGY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.ProtoAllocatorStrategy.PROTO_LAZY: AllocatorStrategy.LAZY,
        integrated_hw_graph_base_pb2.ProtoAllocatorStrategy.PROTO_BLIND_SEARCH: AllocatorStrategy.BLIND_SEARCH,
        integrated_hw_graph_base_pb2.ProtoAllocatorStrategy.PROTO_POSITIVE_SEARCH: AllocatorStrategy.POSITIVE_SEARCH,
    }

    BUILDER_EXIT_POINT_PB_TO_TYPE = {v: k for k, v in BUILDER_EXIT_POINT_TYPE_TO_PB.items()}

    CONTEXT_SWITCH_MODE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.ProtoContextSwitchMode.PROTO_CONTEXT_SWITCH_MODE_MANUAL: ContextSwitchMode.MANUAL,
        integrated_hw_graph_base_pb2.ProtoContextSwitchMode.PROTO_CONTEXT_SWITCH_MODE_ALLOWED: ContextSwitchMode.ALLOWED,
        integrated_hw_graph_base_pb2.ProtoContextSwitchMode.PROTO_CONTEXT_SWITCH_MODE_ENABLED: ContextSwitchMode.ENABLED,
        integrated_hw_graph_base_pb2.ProtoContextSwitchMode.PROTO_CONTEXT_SWITCH_MODE_DISABLED: ContextSwitchMode.DISABLED,
    }

    CONTEXT_SWITCH_PARTITIONER_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.ProtoContextSwitchPartitioner.PROTO_CONTEXT_SWITCH_PARTITIONER_SLOTH: Partitioner.SLOTH,
        integrated_hw_graph_base_pb2.ProtoContextSwitchPartitioner.PROTO_CONTEXT_SWITCH_PARTITIONER_SLOTTER: Partitioner.SLOTTER,
    }

    TOPOSORT_MODE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.ProtoToposortMode.PROTO_TOPOSORT_MODE_DFS: ToposortMode.DFS,
        integrated_hw_graph_base_pb2.ProtoToposortMode.PROTO_TOPOSORT_MODE_DEPTHWISE: ToposortMode.DEPTHWISE,
    }

    RESOURCES_STRATEGY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.ProtoResourcesStrategy.PROTO_RESOURCES_GREEDY: ResourcesStrategy.GREEDY,
        integrated_hw_graph_base_pb2.ProtoResourcesStrategy.PROTO_RESOURCES_OPTIMIZED: ResourcesStrategy.OPTIMIZED,
        integrated_hw_graph_base_pb2.ProtoResourcesStrategy.PROTO_RESOURCES_AUTOMATIC: ResourcesStrategy.AUTOMATIC,
    }

    RESOURCES_STRATEGY_TYPE_TO_PB = {v: k for k, v in RESOURCES_STRATEGY_PB_TO_TYPE.items()}

    RESOURCES_OBJECTIVE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.ProtoResourcesObjective.PROTO_REQUIRED_FPS: ResourcesObjective.REQUIRED_FPS,
        integrated_hw_graph_base_pb2.ProtoResourcesObjective.PROTO_MAX_FPS: ResourcesObjective.MAX_FPS,
        integrated_hw_graph_base_pb2.ProtoResourcesObjective.PROTO_MIN_LATENCY: ResourcesObjective.MIN_LATENCY,
        integrated_hw_graph_base_pb2.ProtoResourcesObjective.PROTO_DONKEY: ResourcesObjective.DONKEY,
        integrated_hw_graph_base_pb2.ProtoResourcesObjective.PROTO_AUTOMATIC_OBJECTIVE: ResourcesObjective.AUTOMATIC,
    }

    RESOURCES_OBJECTIVE_TYPE_TO_PB = {v: k for k, v in RESOURCES_OBJECTIVE_PB_TO_TYPE.items()}

    BUFFERS_FORMAT_TYPE_TO_PB = {
        LayerBuffersFormat.consecutive: integrated_hw_graph_base_pb2.ProtoLayerBuffersFormat.PROTO_BUFFERS_FORMAT_CONSECUTIVE,
        LayerBuffersFormat.row_per_cut: integrated_hw_graph_base_pb2.ProtoLayerBuffersFormat.PROTO_BUFFERS_FORMAT_ROW_PER_CUT,
        LayerBuffersFormat.automatic: integrated_hw_graph_base_pb2.ProtoLayerBuffersFormat.PROTO_BUFFERS_FORMAT_AUTOMATIC,
        LayerBuffersFormat.divide_by_rows: integrated_hw_graph_base_pb2.ProtoLayerBuffersFormat.PROTO_BUFFERS_FORMAT_DIVIDE_BY_ROWS,
        LayerBuffersFormat.divide_by_words: integrated_hw_graph_base_pb2.ProtoLayerBuffersFormat.PROTO_BUFFERS_FORMAT_DIVIDE_BY_WORDS,
        LayerBuffersFormat.offset: integrated_hw_graph_base_pb2.ProtoLayerBuffersFormat.PROTO_BUFFERS_FORMAT_OFFSET,
    }

    BUFFERS_FORMAT_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.ProtoLayerBuffersFormat.PROTO_BUFFERS_FORMAT_CONSECUTIVE: LayerBuffersFormat.consecutive,
        integrated_hw_graph_base_pb2.ProtoLayerBuffersFormat.PROTO_BUFFERS_FORMAT_ROW_PER_CUT: LayerBuffersFormat.row_per_cut,
        integrated_hw_graph_base_pb2.ProtoLayerBuffersFormat.PROTO_BUFFERS_FORMAT_AUTOMATIC: LayerBuffersFormat.automatic,
        integrated_hw_graph_base_pb2.ProtoLayerBuffersFormat.PROTO_BUFFERS_FORMAT_DIVIDE_BY_ROWS: LayerBuffersFormat.divide_by_rows,
        integrated_hw_graph_base_pb2.ProtoLayerBuffersFormat.PROTO_BUFFERS_FORMAT_DIVIDE_BY_WORDS: LayerBuffersFormat.divide_by_words,
        integrated_hw_graph_base_pb2.ProtoLayerBuffersFormat.PROTO_BUFFERS_FORMAT_OFFSET: LayerBuffersFormat.offset,
    }

    USE_SEQUENCER_POLICY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.ProtoUseSequencer.PROTO_USE_SEQUENCER_ALLOWED: UseSequencerPolicy.allowed,
        integrated_hw_graph_base_pb2.ProtoUseSequencer.PROTO_USE_SEQUENCER_DISABLED: UseSequencerPolicy.disabled,
        integrated_hw_graph_base_pb2.ProtoUseSequencer.PROTO_USE_SEQUENCER_ENABLED: UseSequencerPolicy.enabled,
    }

    ENABLE_LCU_FROM_SEQUENCER_POLICY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.ProtoEnableLcuFromSequencer.PROTO_ENABLE_LCU_FROM_SEQUENCER_DISABLED: EnableLcuFromSequencerPolicy.disabled,
        integrated_hw_graph_base_pb2.ProtoEnableLcuFromSequencer.PROTO_ENABLE_LCU_FROM_SEQUENCER_ENABLED: EnableLcuFromSequencerPolicy.enabled,
    }

    PARAMS_LOAD_TIME_COMPRESSION_POLICY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.ProtoParamsLoadTimeCompression.PROTO_PARAMS_LOAD_TIME_COMPRESSION_ALLOWED: ParamsLoadTimeCompressionPolicy.allowed,
        integrated_hw_graph_base_pb2.ProtoParamsLoadTimeCompression.PROTO_PARAMS_LOAD_TIME_COMPRESSION_DISABLED: ParamsLoadTimeCompressionPolicy.disabled,
        integrated_hw_graph_base_pb2.ProtoParamsLoadTimeCompression.PROTO_PARAMS_LOAD_TIME_COMPRESSION_ENABLED: ParamsLoadTimeCompressionPolicy.enabled,
    }

    BUFFERS_TYPE_TO_PB = {
        "FULL_ROW": integrated_hw_graph_base_pb2.ProtoBuffersType.PROTO_FULL_ROW,
        "PARTIAL_ROW": integrated_hw_graph_base_pb2.ProtoBuffersType.PROTO_PARTIAL_ROW,
    }

    USE_SEQUENCER_POLICY_TYPE_TO_PB = {v: k for k, v in USE_SEQUENCER_POLICY_PB_TO_TYPE.items()}

    ENABLE_LCU_FROM_SEQUENCER_POLICY_TYPE_TO_PB = {v: k for k, v in ENABLE_LCU_FROM_SEQUENCER_POLICY_PB_TO_TYPE.items()}

    PARAMS_LOAD_TIME_COMPRESSION_POLICY_TYPE_TO_PB = {
        v: k for k, v in PARAMS_LOAD_TIME_COMPRESSION_POLICY_PB_TO_TYPE.items()
    }

    INPUT_FEATURE_SPLITTER_DEFUSE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_INPUT_FEATURE_SPLITTER_DEFUSE_ALLOWED: InputFeatureSplitterDefuse.ALLOWED,
        integrated_hw_graph_base_pb2.PROTO_INPUT_FEATURE_SPLITTER_DEFUSE_ENABLED: InputFeatureSplitterDefuse.ENABLED,
        integrated_hw_graph_base_pb2.PROTO_INPUT_FEATURE_SPLITTER_DEFUSE_DISABLED: InputFeatureSplitterDefuse.DISABLED,
    }
    INPUT_FEATURE_SPLITTER_DEFUSE_TYPE_TO_PB = {v: k for k, v in INPUT_FEATURE_SPLITTER_DEFUSE_PB_TO_TYPE.items()}

    WIDTH_SPLITTER_DEFUSE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_WIDTH_SPLITTER_DEFUSE_ALLOWED: WidthSplitterDefuse.ALLOWED,
        integrated_hw_graph_base_pb2.PROTO_WIDTH_SPLITTER_DEFUSE_ENABLED: WidthSplitterDefuse.ENABLED,
        integrated_hw_graph_base_pb2.PROTO_WIDTH_SPLITTER_DEFUSE_DISABLED: WidthSplitterDefuse.DISABLED,
    }
    WIDTH_SPLITTER_DEFUSE_TYPE_TO_PB = {v: k for k, v in WIDTH_SPLITTER_DEFUSE_PB_TO_TYPE.items()}

    FEATURE_MULTIPLIER_TYPE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.ProtoFeatureMultiplierType.PROTO_FEATURE_MULTIPLIER_USER_SPECIFIED: FeatureMultiplierType.user_specified,
        integrated_hw_graph_base_pb2.ProtoFeatureMultiplierType.PROTO_FEATURE_MULTIPLIER_SQUARE: FeatureMultiplierType.square,
        integrated_hw_graph_base_pb2.ProtoFeatureMultiplierType.PROTO_FEATURE_MULTIPLIER_YOLO: FeatureMultiplierType.yolov5,
    }
    FEATURE_MULTIPLIER_TYPE_TYPE_TO_PB = {v: k for k, v in FEATURE_MULTIPLIER_TYPE_PB_TO_TYPE.items()}

    NORMALIZATION_TYPE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.ProtoNormalizationType.PROTO_NORMALIZATION: NormalizationType.normalization,
        integrated_hw_graph_base_pb2.ProtoNormalizationType.PROTO_MUL_AND_ADD: NormalizationType.mul_and_add,
    }
    NORMALIZATION_TYPE_TO_PB = {v: k for k, v in NORMALIZATION_TYPE_PB_TO_TYPE.items()}

    AUTOMATIC_RESHAPES_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_AUTOMATIC_RESHAPES_ALLOWED: AutomaticReshapesPolicy.ALLOWED,
        integrated_hw_graph_base_pb2.PROTO_AUTOMATIC_RESHAPES_ENABLED: AutomaticReshapesPolicy.ENABLED,
        integrated_hw_graph_base_pb2.PROTO_AUTOMATIC_RESHAPES_DISABLED: AutomaticReshapesPolicy.DISABLED,
    }
    AUTOMATIC_RESHAPES_TYPE_TO_PB = {v: k for k, v in AUTOMATIC_RESHAPES_PB_TO_TYPE.items()}

    PLATFORM_PARAM_TARGET_TYPE_TO_PB = {
        PlatformTargets.ETHERNET: integrated_hw_graph_base_pb2.PROTO_PLATFORM_TARGET_ETHERNET,
        PlatformTargets.HW_SIM: integrated_hw_graph_base_pb2.PROTO_PLATFORM_TARGET_HW_SIM,
        PlatformTargets.HW_SIM_MC: integrated_hw_graph_base_pb2.PROTO_PLATFORM_TARGET_HW_SIM_MC,
        PlatformTargets.HW_SIM_FC: integrated_hw_graph_base_pb2.PROTO_PLATFORM_TARGET_HW_SIM_FC,
    }

    PLATFORM_PARAM_TARGET_PB_TO_TYPE = {v: k for k, v in PLATFORM_PARAM_TARGET_TYPE_TO_PB.items()}

    PLATFORM_PARAM_HINTS_TYPE_TO_PB = {
        PlatformHints.LOW_PCIE_BANDWIDTH: integrated_hw_graph_base_pb2.PROTO_PLATFORM_HINT_LOW_PCIE_BANDWIDTH,
    }

    PLATFORM_PARAM_HINTS_PB_TO_TYPE = {v: k for k, v in PLATFORM_PARAM_HINTS_TYPE_TO_PB.items()}

    KO_POLICY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_KO_ALLOWED: KoPolicy.ALLOWED,
        integrated_hw_graph_base_pb2.PROTO_KO_DISABLED: KoPolicy.DISABLED,
        integrated_hw_graph_base_pb2.PROTO_KO_ENABLED: KoPolicy.ENABLED,
    }
    KO_POLICY_TYPE_TO_PB = {v: k for k, v in KO_POLICY_PB_TO_TYPE.items()}

    ALIGN_CCWS_SECTION_POLICY_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_SHOULD_ALIGN_CCWS_SECTION_ALLOWED: ShouldAlignCcwsSection.allowed,
        integrated_hw_graph_base_pb2.PROTO_SHOULD_ALIGN_CCWS_SECTION_DISABLED: ShouldAlignCcwsSection.disabled,
        integrated_hw_graph_base_pb2.PROTO_SHOULD_ALIGN_CCWS_SECTION_ENABLED: ShouldAlignCcwsSection.enabled,
    }
    ALIGN_CCWS_SECTION_POLICY_TYPE_TO_PB = {v: k for k, v in ALIGN_CCWS_SECTION_POLICY_PB_TO_TYPE.items()}

    CONCAT_AXIS_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_CONCAT_AXIS_FEATURES: ConcatAxis.features,
        integrated_hw_graph_base_pb2.PROTO_CONCAT_AXIS_SPATIAL_H: ConcatAxis.spatial_h,
        integrated_hw_graph_base_pb2.PROTO_CONCAT_AXIS_SPATIAL_W: ConcatAxis.spatial_w,
    }
    CONCAT_AXIS_TYPE_TO_PB = {v: k for k, v in CONCAT_AXIS_PB_TO_TYPE.items()}

    SHAPE_SPLITTER_TYPE_PB_TO_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_SHAPE_SPLITT_TYPE_HEIGHT: ShapeSplitterType.SPLIT_HEIGHT,
        integrated_hw_graph_base_pb2.PROTO_SHAPE_SPLITT_TYPE_WIDTH: ShapeSplitterType.SPLIT_WIDTH,
        integrated_hw_graph_base_pb2.PROTO_SHAPE_SPLITT_TYPE_FEATURES: ShapeSplitterType.SPLIT_FEATURES,
    }
    SHAPE_SPLITTER_TYPE_TYPE_TO_PB = {v: k for k, v in SHAPE_SPLITTER_TYPE_PB_TO_TYPE.items()}

    IO_TYPE_PB_TO_HN_TYPE = {
        integrated_hw_graph_base_pb2.PROTO_STANDARD: IOType.STANDARD,
        integrated_hw_graph_base_pb2.PROTO_CACHE: IOType.CACHE,
    }
    IO_TYPE_HN_TO_PB_TYPE = {v: k for k, v in IO_TYPE_PB_TO_HN_TYPE.items()}

    def __init__(self):
        self.integrated_hw_graph_pb2 = integrated_hw_graph_pb2
        self.integrated_hw_graph_base_pb2 = integrated_hw_graph_base_pb2
        self.protobuf_exporter = ProtobufExporter
        self.protobuf_importer = ProtobufImporter
