from enum import Enum

from hailo_sdk_client.allocator.resources_params import AutoDouble, convert_to_auto_double
from hailo_sdk_common.hailo_nn.tools_params import ToolsParams, convert_str_to_bool, convert_to_int


class OptimizationLevel(Enum):
    ZERO = 0
    ONE = 1
    TWO = 2


DEFAULT_OPTIMIZATION_LEVEL = OptimizationLevel.ONE


def get_max_compiler_optimization_level():
    return max(level.value for level in OptimizationLevel)


class BufferCalcCommand(Enum):
    OPTIMIZE_BUFFERS = "optimize_buffers"
    SPLIT_AWARE_OPTIMIZE_BUFFERS = "split_aware_optimize_buffers"
    BUFFER_CALC_FPS = "buffer_calc_fps"
    SAVE_LATENCY_TIMELINE = "save_latency_timeline"


def convert_str_to_optimization_level(optimization_level):
    optimization_level = int(optimization_level)

    return OptimizationLevel(optimization_level)


class PerformanceParams(ToolsParams):
    DEFAULT_PARAMS = {
        "fps": None,
        "optimization_level": DEFAULT_OPTIMIZATION_LEVEL,
        "optimize_for_batch": 0,
        "optimize_for_power": False,
        "low_pcie_bandwidth_value": AutoDouble("automatic"),
    }
    ALLOWED_VALUES = {
        "optimization_level": ["0", "1", "max"],
    }
    VALUE_CONVERTERS = {
        "fps": convert_to_int,
        "optimization_level": convert_str_to_optimization_level,
        "optimize_for_batch": convert_to_int,
        "optimize_for_power": convert_str_to_bool,
        "low_pcie_bandwidth_value": convert_to_auto_double,
    }
    INTERNAL_PARAMS = ["optimize_for_batch", "low_pcie_bandwidth_value"]

    def __init__(self):
        super().__init__()
        self.set_default_params()
        self._fps = None
        self._components = []

    def to_pb(self, pb, pb_wrapper):
        self.set_pb_field(pb.fps, "fps")
        self.set_pb_field(pb.optimization_level, "optimization_level")
        self.set_pb_field(pb.optimize_for_batch, "optimize_for_batch")
        self.set_pb_field(pb.optimize_for_power, "optimize_for_power")
        self.set_auto_pb_field(pb.low_pcie_bandwidth_value, "low_pcie_bandwidth_value")

    def from_pb(self, pb, pb_wrapper):
        performance_params_pb = pb
        self.get_pb_field(performance_params_pb, "fps")
        self.get_pb_field(performance_params_pb, "optimization_level")
        self.get_pb_field(performance_params_pb, "optimize_for_batch")
        self.get_pb_field(performance_params_pb, "optimize_for_power")
        self.get_auto_pb_field(performance_params_pb, "low_pcie_bandwidth_value", AutoDouble)
