#!/usr/bin/env python

from enum import Enum

from hailo_sdk_client.sdk_backend.sdk_backend_exceptions import AllocatorScriptParserException
from hailo_sdk_common.hailo_nn.tools_params import (
    ToolsParams,
    allowed_str_for_enumerate,
    convert_value_str_to_list_enumerate,
)
from hailo_sdk_common.versions import LATEST_HEF_VERSION


class PlatformTargets(Enum):
    DEFAULT = "default"
    ETHERNET = "ethernet"
    HW_SIM = "hw_sim"
    HW_SIM_MC = "hw_sim_mc"
    HW_SIM_FC = "hw_sim_fc"

    def __str__(self):
        return self.value


class PlatformHints(Enum):
    DEFAULT = "default"
    LOW_PCIE_BANDWIDTH = "low_pcie_bandwidth"


version_alias_dict = {
    "v1": 1,
    "v2": 2,
    "v3": 3,
    "v4": 4,
    "v5": 5,
    "latest": LATEST_HEF_VERSION,
}


def convert_hef_version_input(version_from_alls):
    if version_from_alls in version_alias_dict:
        return version_alias_dict[version_from_alls]
    version_from_alls = int(version_from_alls)
    if version_from_alls in version_alias_dict.values():
        return version_from_alls
    raise AllocatorScriptParserException(f"Recieved a bad HEF version {version_from_alls}")


class PlatformParams(ToolsParams):
    TARGETS_STR = "targets"
    HEF_VERSION_STR = "hef_version"
    HINTS_STR = "hints"

    COMMENTED_PARAMS = [
        TARGETS_STR,
        HINTS_STR,
    ]

    DEFAULT_PARAMS = {
        TARGETS_STR: [],
        HEF_VERSION_STR: LATEST_HEF_VERSION,
        HINTS_STR: [],
    }
    ALLOWED_VALUES = {
        TARGETS_STR: allowed_str_for_enumerate(PlatformTargets),
        HINTS_STR: allowed_str_for_enumerate(PlatformHints),
    }
    VALUE_CONVERTERS = {
        TARGETS_STR: convert_value_str_to_list_enumerate(PlatformTargets),
        HEF_VERSION_STR: convert_hef_version_input,
        HINTS_STR: convert_value_str_to_list_enumerate(PlatformHints),
    }

    def __init__(self, **kwargs):
        super().__init__()
        self.set_default_params()
        self.override_params_from_kwargs(**kwargs)

    def to_pb(self, pb, pb_wrapper):
        self.extend_pb_field(pb, PlatformParams.TARGETS_STR, pb_wrapper.PLATFORM_PARAM_TARGET_TYPE_TO_PB)
        self.set_pb_field(pb, PlatformParams.HEF_VERSION_STR)
        self.extend_pb_field(pb, PlatformParams.HINTS_STR, pb_wrapper.PLATFORM_PARAM_HINTS_TYPE_TO_PB)

    def from_pb(self, pb, pb_wrapper):
        self.get_pb_field(pb, PlatformParams.HEF_VERSION_STR)
        self.get_pb_list_field(pb, PlatformParams.TARGETS_STR, pb_wrapper.PLATFORM_PARAM_TARGET_PB_TO_TYPE)
        self.get_pb_list_field(pb, PlatformParams.HINTS_STR, pb_wrapper.PLATFORM_PARAM_HINTS_PB_TO_TYPE)
