#!/usr/bin/env python

from enum import Enum

from hailo_sdk_common.hailo_nn.tools_params import (
    AutoVariablePolicy,
    ToolsParams,
    allowed_str_for_enumerate,
    convert_str_to_enumerate,
    convert_to_float,
)


class ResourcesStrategy(Enum):
    GREEDY = "greedy"
    OPTIMIZED = "optimized"
    AUTOMATIC = "automatic"


class ResourcesObjective(Enum):
    REQUIRED_FPS = "required_fps"
    MAX_FPS = "max_fps"
    MIN_LATENCY = "min_latency"
    AUTOMATIC = "automatic"
    DONKEY = "random"


class AutoDouble:
    def __init__(self, val):
        if val == "automatic":
            self._policy = AutoVariablePolicy.AUTOMATIC
            self._val = 0  # uninitialized
        elif type(val) in [int, float]:
            self._policy = AutoVariablePolicy.MANUAL
            self._val = convert_to_float(val)
        elif isinstance(val, str):
            self._policy = AutoVariablePolicy.MANUAL
            self._val = float(val)
        elif type(val) is AutoDouble:
            self._policy = val.policy()
            self._val = val.val()
        else:
            raise ValueError("Value must be a number or 'automatic'")

    def __str__(self):
        return str(self._val)

    def policy(self):
        return self._policy

    def val(self):
        return self._val


def convert_to_auto_double(val):
    return AutoDouble(val)


class ResourcesParams(ToolsParams):
    DEFAULT_PARAMS = {
        "strategy": ResourcesStrategy.AUTOMATIC,
        "objective": ResourcesObjective.AUTOMATIC,
        "max_utilization": AutoDouble("automatic"),
        "max_control_utilization": AutoDouble("automatic"),
        "max_compute_utilization": AutoDouble("automatic"),
        "max_compute_16bit_utilization": AutoDouble("automatic"),
        "max_memory_utilization": AutoDouble("automatic"),
        "max_input_aligner_utilization": AutoDouble("automatic"),
        "max_apu_utilization": AutoDouble("automatic"),
    }
    ALLOWED_VALUES = {
        "strategy": allowed_str_for_enumerate(ResourcesStrategy),
        "objective": allowed_str_for_enumerate(ResourcesObjective),
    }
    VALUE_CONVERTERS = {
        "strategy": convert_str_to_enumerate(ResourcesStrategy),
        "objective": convert_str_to_enumerate(ResourcesObjective),
        "max_utilization": convert_to_auto_double,
        "max_control_utilization": convert_to_auto_double,
        "max_compute_utilization": convert_to_auto_double,
        "max_compute_16bit_utilization": convert_to_auto_double,
        "max_memory_utilization": convert_to_auto_double,
        "max_input_aligner_utilization": convert_to_auto_double,
        "max_apu_utilization": convert_to_auto_double,
    }

    def __init__(self):
        super().__init__()
        self.set_default_params()

    def to_pb(self, pb, pb_wrapper):
        self.set_pb_field(pb, "strategy", pb_wrapper.RESOURCES_STRATEGY_TYPE_TO_PB)
        self.set_pb_field(pb, "objective", pb_wrapper.RESOURCES_OBJECTIVE_TYPE_TO_PB)
        self.set_auto_pb_field(pb, pb_wrapper, "max_utilization")
        self.set_auto_pb_field(pb, pb_wrapper, "max_control_utilization")
        self.set_auto_pb_field(pb, pb_wrapper, "max_compute_utilization")
        self.set_auto_pb_field(pb, pb_wrapper, "max_compute_16bit_utilization")
        self.set_auto_pb_field(pb, pb_wrapper, "max_memory_utilization")
        self.set_auto_pb_field(pb, pb_wrapper, "max_input_aligner_utilization")
        self.set_auto_pb_field(pb, pb_wrapper, "max_apu_utilization")

    def from_pb(self, pb, pb_wrapper):
        self.get_pb_field(pb, "strategy", pb_wrapper.RESOURCES_STRATEGY_PB_TO_TYPE)
        self.get_pb_field(pb, "objective", pb_wrapper.RESOURCES_OBJECTIVE_PB_TO_TYPE)
        self.get_auto_pb_field(pb, "max_utilization", AutoDouble)
        self.get_auto_pb_field(pb, "max_control_utilization", AutoDouble)
        self.get_auto_pb_field(pb, "max_compute_utilization", AutoDouble)
        self.get_auto_pb_field(pb, "max_compute_16bit_utilization", AutoDouble)
        self.get_auto_pb_field(pb, "max_memory_utilization", AutoDouble)
        self.get_auto_pb_field(pb, "max_input_aligner_utilization", AutoDouble)
        self.get_auto_pb_field(pb, "max_apu_utilization", AutoDouble)
