import copy
import glob
import io
import json
import os
import re
import tarfile
import tempfile
import time
from enum import Enum
from pathlib import Path
from typing import Optional

import onnx
from packaging import version

from hailo_model_optimization.acceleras.utils.acceleras_definitions import PostprocessTarget
from hailo_model_optimization.acceleras.utils.params_loader import ParamSerializationType, load_params, save_params
from hailo_model_optimization.tools.mo_script_parser import OptimizationFlavorsInfo
from hailo_model_optimization.tools.orchestator import FlowCheckPoint
from hailo_sdk_client.exposed_definitions import NNFramework, States
from hailo_sdk_client.sdk_backend.modification_config import ModificationsConfig
from hailo_sdk_client.sdk_backend.script_parser.nms_postprocess_command import NMS_CONFIG_FILE_FAKE_HAR_PATH
from hailo_sdk_client.tools.core_postprocess.nms_postprocess import NMSConfig, NMSMetaData
from hailo_sdk_common import get_version
from hailo_sdk_common.hailo_nn.hn_definitions import NMSMetaArchitectures
from hailo_sdk_common.model_params.model_params import ModelParams
from hailo_sdk_common.targets.inference_targets import ParamsKinds


class HailoArchiveSaveException(Exception):
    """Raised when trying to serialize invalid runner."""


class HailoArchiveLoadException(Exception):
    """Raised when trying to deserialize corrupted HAR file."""


class HARMetaDataKeys(str, Enum):
    STATE = "state"
    FORCE_WEIGHTLESS_MODEL = "force_weightless_model"
    MODEL_NAME = "model_name"
    SDK_VERSION = "sdk_version"
    HW_ARCH = "hw_arch"
    NMS_ENGINE = "nms_engine"
    NMS_META_ARCH = "nms_meta_arch"
    MO_FLAVOR = "mo_flavor"
    FLAVOR_CONFIG = "flavor_config"


class HARFileNames(str, Enum):
    METADATA = "metadata"
    HN = "hn"
    NATIVE_HN = "native_hn"
    FP_HN = "fp_hn"
    PARAMS = "params"
    PARAM_AFTER_BN = "param_after_bn"
    PARAMS_FP_OPT = "params_fp_opt"
    PARAMS_HAILO_OPTIMIZED = "params_hailo_optimized"
    PARAMS_TRANSLATED = "params_translated"
    PARAMS_STATISTICS = "params_statistics"
    MODEL_SCRIPT = "model_script"
    AUTO_MODEL_SCRIPT = "auto_model_script"
    ORIGINAL_MODEL = "original_model"
    HEF = "hef"
    ORIGINAL_MODEL_META = "original_model_meta"
    PREPROCESS = "preprocess"
    POSTPROCESS = "postprocess"
    NMS_CONFIG = "nms_config"
    MODIFICATIONS_META_DATA = "modifications_meta_data"
    MODIFICATIONS_PARAMS = "modifications_params"
    FLOW_MEMENTO = "flow_memento"
    FLOW_MEMENTO_BUILDER = "flow_memento_builder"
    LORA_WEIGHTS_META_DATA = "lora_weights_meta_data"


class HailoArchive:
    """Hailo Archive representation."""

    LEGACY_DEFAULT_HW_ARCH = "hailo8"

    def __init__(
        self,
        state,
        original_model_path=None,
        hn=None,
        native_hn=None,
        fp_hn=None,
        model_name=None,
        params=None,
        params_after_bn=None,
        params_fp_opt=None,
        params_hailo_opt=None,
        params_translated=None,
        params_statistics=None,
        model_script=None,
        auto_model_script=None,
        force_weightless_model=False,
        hef=None,
        hw_arch=None,
        original_model_meta=None,
        preprocess_model=None,
        postprocess_model=None,
        nms_metadata=None,
        mo_flavor=None,
        flavor_config=None,
        modifications_meta_data=None,
        optimization_flow_memento=None,
        lora_weights_metadata=None,
    ):
        """
        Hailo Archive constructor.

        Args:
            state (:class:`~hailo_sdk_client.exposed_definitions.States`): The state of the runner to
                archive.
            original_model_path (str, optional): Path for the original model of the runner to
                archive.
            hn (:class:`~hailo_sdk_common.hailo_nn.hailo_nn.HailoNN`, optional): The HN of the
                runner to archive.
            native_hn (:class:`~hailo_sdk_common.hailo_nn.hailo_nn.HailoNN`, optional): The native HN of the
                runner to archive.
            fp_hn (:class:`~hailo_sdk_common.hailo_nn.hailo_nn.HailoNN`, optional): The full-precision HN of the
                runner to archive.
            model_name (str, optional): The model name of the runner to archive.
            params (:class:`~hailo_sdk_common.model_params.model_params.ModelParams`, optional): The
                params of the runner to archive.
            params_after_bn (:class:`~hailo_sdk_common.model_params.model_params.ModelParams`, optional): The
                native params after BN to archive.
            params_fp_opt (:class:`~hailo_sdk_common.model_params.model_params.ModelParams`, optional): The
                native params FP optimized to archive.
            params_translated (:class:`~hailo_sdk_common.model_params.model_params.ModelParams`, optional): The
                translated params of the runner to archive.
            params_statistics (:class:`~hailo_sdk_common.model_params.model_params.ModelParams`, optional): The
                statistics params of the runner to archive.
            model_script (str, optional): The model script loaded to the runner to archive.
            auto_model_script (str, optional): The auto-generated model script for the archive.
            force_weightless_model (bool, optional): Whether the runner to archive is forcing weightless model.
                Defaults to False.
            hef (str, optional): The data of the HEF containing the HW representation of the runner to archive.
            hw_arch (str, optional): Hardware architecture to be used. Defaults to None.
            original_model_meta (dict, optional): Metadata of the original model to archive.
            preprocess_model (bytes, optional): ONNX model of the pre-process ops to archive.
            postprocess_model (bytes, optional): ONNX model of the post-process ops to archive.

        """
        self._state = state
        self._original_model_path = original_model_path
        self._hn = hn
        self._native_hn = native_hn
        self._fp_hn = fp_hn
        self._model_name = model_name if model_name else "model"
        self._params = params
        self._params_after_bn = params_after_bn
        self._params_fp_opt = params_fp_opt
        self._params_hailo_opt = params_hailo_opt
        self._params_translated = params_translated
        self._params_statistics = params_statistics
        self._model_script = model_script
        self._auto_model_script = auto_model_script
        self._force_weightless_model = force_weightless_model
        self._hef = hef
        self._hw_arch = hw_arch
        self._metadata = {
            HARMetaDataKeys.STATE: self._state,
            HARMetaDataKeys.FORCE_WEIGHTLESS_MODEL: self._force_weightless_model,
            HARMetaDataKeys.MODEL_NAME: self._model_name,
            HARMetaDataKeys.SDK_VERSION: get_version("hailo_sdk_client"),
            HARMetaDataKeys.HW_ARCH: self._hw_arch,
        }
        self._original_model_meta = original_model_meta
        self._preprocess_model = preprocess_model
        self._postprocess_model = postprocess_model
        self._nms_metadata = nms_metadata
        self._mo_flavor = mo_flavor
        self._flavor_config = flavor_config
        self._modifications_meta_data = modifications_meta_data
        self._optimization_flow_memento = optimization_flow_memento
        self._lora_weights_metadata = lora_weights_metadata

    @property
    def state(self):
        return self._state

    @property
    def original_model_path(self):
        return self._original_model_path

    @property
    def hn(self):
        return self._hn

    @property
    def native_hn(self):
        return self._native_hn

    @property
    def fp_hn(self):
        return self._fp_hn

    @property
    def model_name(self):
        return self._model_name

    @property
    def params(self):
        return self._params

    @property
    def params_after_bn(self):
        return self._params_after_bn

    @property
    def params_fp_opt(self):
        return self._params_fp_opt

    @property
    def params_hailo_opt(self):
        return self._params_hailo_opt

    @property
    def params_translated(self):
        return self._params_translated

    @property
    def params_statistics(self):
        return self._params_statistics

    @property
    def model_script(self):
        return self._model_script

    @property
    def auto_model_script(self):
        return self._auto_model_script

    @property
    def force_weightless_model(self):
        return self._force_weightless_model

    @property
    def hef(self):
        return self._hef

    @property
    def hw_arch(self):
        return self._hw_arch

    @property
    def original_model_meta(self):
        return self._original_model_meta

    @property
    def preprocess_model(self):
        return self._preprocess_model

    @property
    def postprocess_model(self):
        return self._postprocess_model

    @property
    def nms_metadata(self):
        return self._nms_metadata

    @property
    def nms_config_file(self):
        return self.nms_metadata.config_file if self.nms_metadata else None

    @property
    def mo_flavor(self):
        return self._mo_flavor

    @property
    def flavor_config(self):
        return self._flavor_config

    @property
    def modifications_meta_data(self):
        return self._modifications_meta_data

    @property
    def lora_weights_metadata(self):
        return self._lora_weights_metadata

    def save(
        self,
        path,
        compressed=False,
        save_original_model=False,
        compilation_only=False,
        params_serialization: ParamSerializationType = ParamSerializationType.NPZ,
    ):
        """
        Save the Hailo Archive as a HAR file.

        Args:
            path (str): The path for the generated HAR file.
            compressed (bool, optional): Whether to compress the archive. Defaults to False.
            save_original_model (bool, optional): Whether to save the original model (TF/ONNX) in the HAR file.
                Defaults to False.
            compilation_only (bool, optional): Whether to save a HAR containing the minimum information required for
                compilation. Defaults to False.
            params_serialization (:class:`~hailo_model_optimization.acceleras.utils.acceleras_definitions.ParamSerializationType`, optional):
                The serialization type of the params. Defaults to ParamSerializationType.NPZ. Using HDF5 will allow faster load time, but won't work with netron.

        """
        mode = "w" if not compressed else "w:gz"
        params_suffix = params_serialization.clean_suffix()
        with tarfile.open(path, mode) as har_file:
            if save_original_model and not compilation_only:
                self._add_original_model_to_har(har_file)

            if self.state not in [States.UNINITIALIZED, States.ORIGINAL_MODEL]:
                hn = self.hn if isinstance(self.hn, str) else self.hn.to_hn(self.hn.name)
                self._add_data_to_har(har_file, hn.encode(), "hn", HARFileNames.HN)

            self._add_params_to_har(
                har_file, self.params_translated, f"q.{params_suffix}", HARFileNames.PARAMS_TRANSLATED
            )

            if self.model_script is not None:
                if self.state in [States.HAILO_MODEL]:
                    self._add_nms_metadata_before_apply_to_har(har_file)

                # removes nms config file paths from alls script for extracting nms from har when loading it in future
                self._model_script = re.sub(
                    r'nms_postprocess\(".*"',
                    f'nms_postprocess("{NMS_CONFIG_FILE_FAKE_HAR_PATH}"',
                    self.model_script,
                )
                self._add_data_to_har(har_file, self.model_script.encode(), "alls", HARFileNames.MODEL_SCRIPT)

            if self.state in [States.COMPILED_MODEL, States.COMPILED_SLIM_MODEL]:
                self._add_data_to_har(har_file, self.hef, "hef", HARFileNames.HEF)
                self._add_data_to_har(
                    har_file,
                    self.auto_model_script.encode(),
                    "auto.alls",
                    HARFileNames.AUTO_MODEL_SCRIPT,
                )

            if self.nms_metadata and self.nms_metadata.config_file:
                self._add_nms_metadata_to_har(
                    har_file,
                    self.nms_metadata.config_file,
                    self.nms_metadata.meta_arch,
                    self.nms_metadata.engine,
                )

            if self.mo_flavor:
                self._metadata[HARMetaDataKeys.MO_FLAVOR] = self.mo_flavor.dict()

            if self.flavor_config:
                self._metadata[HARMetaDataKeys.FLAVOR_CONFIG] = self.flavor_config

            if self.state in [States.QUANTIZED_SLIM_MODEL, States.COMPILED_SLIM_MODEL] or compilation_only:
                regular_to_slim = {
                    States.QUANTIZED_MODEL: States.QUANTIZED_SLIM_MODEL,
                    States.COMPILED_MODEL: States.COMPILED_SLIM_MODEL,
                }
                self._metadata[HARMetaDataKeys.STATE] = regular_to_slim.get(self.state, self.state)
                self._add_data_to_har(
                    har_file,
                    json.dumps(self._metadata).encode(),
                    "metadata.json",
                    HARFileNames.METADATA,
                )
                return

            if self.state in [
                States.FP_OPTIMIZED_MODEL,
                States.QUANTIZED_BASE_MODEL,
                States.QUANTIZED_MODEL,
                States.COMPILED_MODEL,
            ]:
                encoded_native_hn = self._native_hn.to_hn(self._native_hn.name).encode()
                self._add_data_to_har(har_file, encoded_native_hn, "native.hn", HARFileNames.NATIVE_HN)

            if self._optimization_flow_memento:
                self._add_flow_memento_to_har(har_file, self._optimization_flow_memento)

            if self.state in [States.QUANTIZED_BASE_MODEL, States.QUANTIZED_MODEL, States.COMPILED_MODEL]:
                fp_hn = self.fp_hn.to_hn(self.fp_hn.name).encode()
                self._add_data_to_har(har_file, fp_hn, "fp.hn", HARFileNames.FP_HN)

            self._add_params_to_har(har_file, self._params, params_suffix, HARFileNames.PARAMS)
            self._add_params_to_har(har_file, self._params_after_bn, f"bn.{params_suffix}", HARFileNames.PARAM_AFTER_BN)
            self._add_params_to_har(har_file, self._params_fp_opt, f"fpo.{params_suffix}", HARFileNames.PARAMS_FP_OPT)
            self._add_params_to_har(
                har_file, self._params_hailo_opt, f"ho.{params_suffix}", HARFileNames.PARAMS_HAILO_OPTIMIZED
            )
            self._add_params_to_har(
                har_file, self._params_statistics, f"stats.{params_suffix}", HARFileNames.PARAMS_STATISTICS
            )

            if self._original_model_meta:
                data_copy = copy.deepcopy(self._original_model_meta)
                if self._original_model_meta.get("parsing_report"):
                    data_copy["parsing_report"] = data_copy["parsing_report"].dict()
                self._add_data_to_har(
                    har_file,
                    json.dumps(data_copy).encode(),
                    "original_model_meta.json",
                    HARFileNames.ORIGINAL_MODEL_META,
                )

            if self.modifications_meta_data and (
                self.modifications_meta_data.inputs
                or self.modifications_meta_data.outputs
                or self.modifications_meta_data.tracker.layers
            ):
                self._add_data_to_har(
                    har_file,
                    self.modifications_meta_data.json().encode(),
                    "modifications_meta_data.json",
                    HARFileNames.MODIFICATIONS_META_DATA,
                )
                if self.modifications_meta_data.tracker.has_modification_params():
                    self._add_params_to_har(
                        har_file,
                        self.modifications_meta_data.tracker.get_modification_params(),
                        f"modification_params.{params_suffix}",
                        HARFileNames.MODIFICATIONS_PARAMS,
                    )

            if self.lora_weights_metadata and self.state in [States.QUANTIZED_BASE_MODEL, States.QUANTIZED_MODEL]:
                self._add_data_to_har(
                    har_file,
                    json.dumps(self.lora_weights_metadata).encode(),
                    "lora_weights_meta_data.json",
                    HARFileNames.LORA_WEIGHTS_META_DATA,
                )

            self._add_onnx_model_to_har(har_file, self._preprocess_model, HARFileNames.PREPROCESS)
            self._add_onnx_model_to_har(har_file, self._postprocess_model, HARFileNames.POSTPROCESS)

            self._add_data_to_har(
                har_file,
                json.dumps(self._metadata).encode(),
                "metadata.json",
                HARFileNames.METADATA,
            )

    def _add_nms_metadata_before_apply_to_har(self, har_file):
        match = re.match(
            r"nms_postprocess\(\"(?P<config_path>.*)\", meta_arch=(?P<meta_arch>\w*)(, engine=(?P<engine>\w*))?",
            self.model_script,
        )
        if match is None:
            return

        nms_values = match.groupdict()
        config_path = nms_values["config_path"]
        meta_arch = nms_values["meta_arch"]
        engine = nms_values["engine"]

        if meta_arch is None:
            raise HailoArchiveSaveException(
                "Model script command nms_postprocess is invalid: meta_arch argument is "
                "missing. Please make sure to provide it.",
            )

        if not os.path.exists(config_path):
            raise HailoArchiveSaveException(
                f"Model script command nms_postprocess command is invalid: Post-process "
                f"config file isn't found in {config_path}. Please make sure the given path"
                f" is relative to the .alls file location, or it is a global path.",
            )
        with open(config_path) as f:
            config_file = json.load(f)

        self._add_nms_metadata_to_har(har_file, config_file, meta_arch, engine)

    def _add_to_har(self, har, buffer, file_ext, size, filename_key):
        filename = f"{self._model_name}.{file_ext}"
        info = tarfile.TarInfo(filename)
        info.size = size
        info.mtime = time.time()
        har.addfile(info, buffer)
        self._metadata[filename_key] = filename

    def _add_data_to_har(self, har, data, file_ext, filename_key):
        buffer = io.BytesIO(initial_bytes=data)
        self._add_to_har(har, buffer, file_ext, len(data), filename_key)

    def _add_original_model_to_har(self, har_file):
        if self._original_model_path:
            self._metadata[HARFileNames.ORIGINAL_MODEL] = []
            original_model_ext = os.path.splitext(self._original_model_path)[1]
            if original_model_ext == ".ckpt":
                for file in glob.glob(f"{self._original_model_path}.*"):
                    filename = f"{self._model_name}.ckpt{os.path.splitext(file)[-1]}"
                    har_file.add(file, arcname=filename)
                    self._metadata[HARFileNames.ORIGINAL_MODEL].append(filename)
            else:
                filename = f"{self._model_name}{original_model_ext}"
                har_file.add(self._original_model_path, arcname=filename)
                self._metadata[HARFileNames.ORIGINAL_MODEL].append(filename)

    def _add_params_to_har(self, har_file, params, file_ext, filename_key):
        if params is not None:
            buffer = io.BytesIO()
            serialize_type = "." + file_ext.split(".")[-1]
            save_params(buffer, dict(iter(params.items())), type_=serialize_type)
            buffer.seek(0)
            self._add_to_har(har_file, buffer, file_ext, buffer.getbuffer().nbytes, filename_key)

    def _add_onnx_model_to_har(self, har_file, onnx_model, filename_key):
        if onnx_model is not None:
            buffer = io.BytesIO()
            onnx.save_model(onnx_model, buffer)
            buffer.seek(0)
            self._add_to_har(har_file, buffer, f"{filename_key}.onnx", buffer.getbuffer().nbytes, filename_key)

    def _add_nms_metadata_to_har(self, har_file, config_file, meta_arch, engine):
        """
        Stores nms config information
        """
        self._add_data_to_har(har_file, json.dumps(config_file).encode(), "nms.json", HARFileNames.NMS_CONFIG)
        self._metadata[HARMetaDataKeys.NMS_META_ARCH] = meta_arch
        self._metadata[HARMetaDataKeys.NMS_ENGINE] = engine

    def _add_flow_memento_to_har(self, har_file, memento: FlowCheckPoint):
        flow_memento = memento.node_checkpoint.checkpoint
        har_file.add(flow_memento.base_path, arcname=HARFileNames.FLOW_MEMENTO.value)
        self._add_data_to_har(
            har_file,
            memento.json().encode(),
            f"{HARFileNames.FLOW_MEMENTO_BUILDER.value}.json",
            HARFileNames.FLOW_MEMENTO_BUILDER,
        )

    @classmethod
    def load(cls, har_path, temp_dir=None):
        """
        Load a given HAR file to a HAR objet.

        Args:
            har_path (str): Path for the HAR file to load.
            temp_dir (str, optional): Path for directory for the extracted original model (TF/ONNX)
                files.

        Returns:
            :class:`~hailo_sdk_common.hailo_archive.hailo_archive.HailoArchive`: The generated
            HailoArchive object.

        """
        with HailoArchiveLoader(har_path) as har_loader:
            return har_loader.load(temp_dir=temp_dir)


class HailoArchiveLoader:
    """Loader for loading a Hailo Archive object from HAR path."""

    def __init__(self, path):
        self._path = path
        self._har_file = tarfile.open(self._path, "r:*")
        self._set_metadata()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self._har_file.close()

    def load(self, temp_dir=None):
        state = self.get_state()
        model_name = self.get_model_name()
        force_weightless_model = self._metadata[HARMetaDataKeys.FORCE_WEIGHTLESS_MODEL]
        original_model_path = self.extract_original_model(temp_dir)
        hn = self.get_hn(must_exist=True) if self.is_hailo_model() else None
        slim_mode = state in [States.QUANTIZED_SLIM_MODEL, States.COMPILED_SLIM_MODEL]
        native_hn = self.get_native_hn(must_exist=self.is_native_hn_in_har() and not slim_mode)
        fp_hn = self.get_fp_hn(must_exist=self.is_fp_hn_in_har() and not slim_mode)
        params = self.get_params(ParamsKinds.NATIVE, must_exist=False)
        params_after_bn = self.get_params(ParamsKinds.NATIVE_FUSED_BN, must_exist=False)
        params_fp_opt = self.get_params(ParamsKinds.FP_OPTIMIZED, must_exist=False)
        params_hailo_opt = self.get_params(ParamsKinds.HAILO_OPTIMIZED, must_exist=False)
        params_translated = self.get_params(ParamsKinds.TRANSLATED, must_exist=True) if self.is_quantized() else None
        params_statistics = self.get_params(ParamsKinds.STATISTICS, must_exist=False)
        model_script = self.get_model_script(must_exist=False)
        auto_model_script = self.get_auto_model_script(must_exist=True) if self.is_compiled() else None
        hef = self.get_hef(must_exist=self.get_sdk_version() is not None) if self.is_compiled() else None
        hw_arch = self.get_hw_arch()
        original_model_meta = self.get_original_model_meta()
        preprocess_model = self.get_preprocess_model()
        postprocess_model = self.get_postprocess_model()
        nms_meta_arch = self._metadata.get(HARMetaDataKeys.NMS_META_ARCH, None)
        nms_config_file = self.get_nms_config_file()
        nms_metadata = None
        if nms_meta_arch and nms_config_file:
            nms_engine = self.get_nms_engine()
            nms_metadata = NMSMetaData(
                NMSConfig.from_json(nms_config_file, nms_meta_arch),
                nms_meta_arch,
                nms_engine,
                nms_config_file,
            )
            self._metadata[HARMetaDataKeys.NMS_ENGINE] = nms_engine
            self._metadata[HARMetaDataKeys.NMS_META_ARCH] = nms_meta_arch

        mo_flavor = None
        if self._metadata.get(HARMetaDataKeys.MO_FLAVOR):
            mo_flavor = OptimizationFlavorsInfo(**self._metadata[HARMetaDataKeys.MO_FLAVOR])

        flavor_config = self._metadata.get(HARMetaDataKeys.FLAVOR_CONFIG, None)

        modifications_meta_data = None
        modifications_meta_data_file = self.get_modifications_meta_data_file()
        if modifications_meta_data_file:
            modifications_meta_data = ModificationsConfig(**modifications_meta_data_file)
            modifications_params_file = self._metadata.get(
                HARFileNames.MODIFICATIONS_PARAMS, f"{HARFileNames.MODIFICATIONS_PARAMS.value}.npz"
            )
            modifications_params = self._get_modifications_params_from_har(modifications_params_file, must_exist=False)
            if modifications_params:
                modifications_meta_data.tracker.set_modification_params(modifications_params)

        # Memento
        flow_memento = self.get_flow_memento()

        # Lora weights metadata
        lora_weights_metadata = self.get_lora_weights_metadata_file()

        # backwards compatibility
        if state in [States.FP_OPTIMIZED_MODEL, States.QUANTIZED_MODEL, States.COMPILED_MODEL] and not fp_hn:
            fp_hn = hn

        return HailoArchive(
            state,
            original_model_path,
            hn,
            native_hn,
            fp_hn,
            model_name,
            params,
            params_after_bn,
            params_fp_opt,
            params_hailo_opt,
            params_translated,
            params_statistics,
            model_script,
            auto_model_script,
            force_weightless_model,
            hef,
            hw_arch,
            original_model_meta,
            preprocess_model,
            postprocess_model,
            nms_metadata,
            mo_flavor,
            flavor_config,
            modifications_meta_data,
            optimization_flow_memento=flow_memento,
            lora_weights_metadata=lora_weights_metadata,
        )

    def _set_metadata(self):
        filenames = [x for x in self._har_file.getnames() if x.endswith(f"{HARFileNames.METADATA.value}.json")]
        if len(filenames) != 1:
            raise HailoArchiveLoadException("Metadata file is missing in HAR")
        else:
            filename = filenames[0]
        metadata = json.loads(self._har_file.extractfile(filename).read().decode())
        self._metadata = {}
        for key, value in metadata.items():
            key = HARFileNames(key) if hasattr(HARFileNames, key.upper()) else HARMetaDataKeys(key)
            if key == HARMetaDataKeys.NMS_META_ARCH and value is not None:
                value = NMSMetaArchitectures(value)
            elif key == HARMetaDataKeys.NMS_ENGINE and value is not None:
                value = PostprocessTarget(value)
            elif key == HARMetaDataKeys.STATE:
                value = States(value)
            self._metadata[key] = value

        allowed_states = [x for x in States if x != States.UNINITIALIZED]
        if self._metadata[HARMetaDataKeys.STATE] not in allowed_states:
            raise HailoArchiveLoadException(f'The state must be one of {", ".join(allowed_states)}')

    def _get_from_har(self, filename, must_exist=True, decode=True):
        if filename in self._har_file.getnames():
            fp = self._har_file.extractfile(filename)
            data = fp.read()
            return data.decode() if decode else data

        elif must_exist:
            raise HailoArchiveLoadException(f"{filename} not in Hailo Archive")

    def _get_params_from_har(self, filename, must_exist=True):
        files_in_har = self._har_file.getnames()

        if filename in files_in_har:
            suffix = os.path.splitext(filename)[-1].lower()
            return ModelParams(load_params(self._har_file.extractfile(filename), suffix))

        elif must_exist:
            raise HailoArchiveLoadException(f"{filename} not in Hailo Archive")

    def _get_modifications_params_from_har(self, filename, must_exist=False):
        files_in_har = self._har_file.getnames()

        if filename in files_in_har:
            suffix = os.path.splitext(filename)[-1].lower()
            return load_params(self._har_file.extractfile(filename), suffix)

        elif must_exist:
            raise HailoArchiveLoadException(f"{filename} not in Hailo Archive")

    def get_state(self):
        return States(self._metadata[HARMetaDataKeys.STATE])

    def get_sdk_version(self):
        sdk_version = self._metadata.get(HARMetaDataKeys.SDK_VERSION, None)
        if sdk_version:
            return version.Version(sdk_version)

    def is_hailo_model(self):
        hn_states = [x for x in States if x not in [States.UNINITIALIZED, States.ORIGINAL_MODEL]]
        return self.get_state() in hn_states

    def is_native_hn_in_har(self):
        har_version = self.get_sdk_version()
        exists_in_version = har_version is not None and (har_version >= version.Version("3.11"))
        exists_in_state = self.get_state() in [States.FP_OPTIMIZED_MODEL, States.QUANTIZED_MODEL, States.COMPILED_MODEL]
        return exists_in_version and exists_in_state

    def is_fp_hn_in_har(self):
        har_version = self.get_sdk_version()
        exists_in_version = har_version is not None and (har_version > version.Version("3.26.dev0"))
        exists_in_state = self.get_state() in [States.QUANTIZED_MODEL, States.COMPILED_MODEL]
        return exists_in_version and exists_in_state

    def is_quantized(self):
        return self.get_state() in [
            States.QUANTIZED_MODEL,
            States.QUANTIZED_BASE_MODEL,
            States.QUANTIZED_SLIM_MODEL,
            States.COMPILED_MODEL,
            States.COMPILED_SLIM_MODEL,
        ]

    def is_compiled(self):
        return self.get_state() in [States.COMPILED_MODEL, States.COMPILED_SLIM_MODEL]

    def extract_original_model(self, dest_dir):
        original_model_members = self._metadata.get(
            HARFileNames.ORIGINAL_MODEL,
            [member for member in self._har_file.getnames() if member.startswith(HARFileNames.ORIGINAL_MODEL)],
        )

        if original_model_members and dest_dir is None:
            raise HailoArchiveLoadException(
                "dest_dir must be given when the archive contains an original model (TF/ONNX)",
            )

        original_model_path = None
        for member in original_model_members:
            self._har_file.extract(member, path=dest_dir)
            prefix, _ = os.path.splitext(member)
            _, inner_ext = os.path.splitext(prefix)
            # Remove suffix for ckpt files.
            filename = prefix if inner_ext else member
            original_model_path = os.path.join(dest_dir, filename)

        return original_model_path

    def get_model_name(self):
        return self._metadata.get(HARMetaDataKeys.MODEL_NAME, None)

    def get_hw_arch(self):
        if not self.get_sdk_version() or self.get_sdk_version() < version.Version("3.16"):
            return self._metadata.get(HARMetaDataKeys.HW_ARCH, None)
        return self._metadata[HARMetaDataKeys.HW_ARCH]

    def get_hn(self, must_exist=False):
        filename = self._metadata.get(HARFileNames.HN, f"{self.get_model_name()}.hn")
        return self._get_from_har(filename, must_exist=must_exist)

    def get_native_hn(self, must_exist=False):
        filename = self._metadata.get(HARFileNames.NATIVE_HN, f"{self.get_model_name()}.native.hn")
        return self._get_from_har(filename, must_exist=must_exist)

    def get_fp_hn(self, must_exist=False):
        filename = self._metadata.get(HARFileNames.FP_HN, f"{self.get_model_name()}.fp.hn")
        return self._get_from_har(filename, must_exist=must_exist)

    def get_params(self, params_kind=ParamsKinds.NATIVE, must_exist=False):
        if params_kind == ParamsKinds.NATIVE:
            filename = self._metadata.get(HARFileNames.PARAMS, f"{HARFileNames.PARAMS.value}.npz")
            return self._get_params_from_har(filename, must_exist=must_exist)

        if params_kind == ParamsKinds.NATIVE_FUSED_BN:
            filename = self._metadata.get(HARFileNames.PARAM_AFTER_BN, f"{HARFileNames.PARAM_AFTER_BN.value}.npz")
            return self._get_params_from_har(filename, must_exist=must_exist)

        if params_kind == ParamsKinds.FP_OPTIMIZED:
            # TODO: what happens with old har?
            filename = self._metadata.get(HARFileNames.PARAMS_FP_OPT, f"{HARFileNames.PARAMS_FP_OPT.value}.fpo.npz")
            return self._get_params_from_har(filename, must_exist=must_exist)

        if params_kind == ParamsKinds.HAILO_OPTIMIZED:
            filename = self._metadata.get(
                HARFileNames.PARAMS_HAILO_OPTIMIZED,
                f"{HARFileNames.PARAMS_HAILO_OPTIMIZED.value}.ho.npz",
            )
            return self._get_params_from_har(filename, must_exist=must_exist)

        if params_kind == ParamsKinds.TRANSLATED:
            filename = self._metadata.get(
                HARFileNames.PARAMS_TRANSLATED, f"{HARFileNames.PARAMS_TRANSLATED.value}.q.npz"
            )
            return self._get_params_from_har(filename, must_exist=must_exist)

        if params_kind == ParamsKinds.STATISTICS:
            filename = self._metadata.get(
                HARFileNames.PARAMS_STATISTICS,
                f"{HARFileNames.PARAMS_STATISTICS.value}.stats.npz",
            )
            return self._get_params_from_har(filename, must_exist=False)

    def get_model_script(self, must_exist=False):
        # After adding filenames to the metadata and before the fix in 3.11, the file under model script key is the auto
        # model script
        if (
            HARFileNames.MODEL_SCRIPT in self._metadata
            and self.get_sdk_version() <= version.Version("3.10")
            and self.get_state() == States.COMPILED_MODEL
        ):
            return None

        filename = self._metadata.get(HARFileNames.MODEL_SCRIPT, f"{HARFileNames.MODEL_SCRIPT.value}.alls")
        return self._get_from_har(filename, must_exist=must_exist)

    def get_auto_model_script(self, must_exist=False):
        # After adding filenames to the metadata and before the fix in 3.11, the file under model script key is the auto
        # model script
        if (
            HARFileNames.MODEL_SCRIPT in self._metadata
            and self.get_sdk_version() <= version.Version("3.10")
            and self.get_state() == States.COMPILED_MODEL
        ):
            filename = self._metadata[HARFileNames.MODEL_SCRIPT]
        else:
            filename = self._metadata.get(
                HARFileNames.AUTO_MODEL_SCRIPT, f"{HARFileNames.MODEL_SCRIPT.value}.auto.alls"
            )

        return self._get_from_har(filename, must_exist=must_exist)

    def get_hef(self, must_exist=False):
        filename = self._metadata.get(HARFileNames.HEF, f"{self.get_model_name()}.hef")
        return self._get_from_har(filename, must_exist=must_exist, decode=False)

    def get_original_model_meta(self):
        filename = self._metadata.get(HARFileNames.ORIGINAL_MODEL_META, None)
        if filename:
            original_model_meta = json.loads(self._har_file.extractfile(filename).read().decode())
            if "framework" not in original_model_meta:
                original_model_meta["framework"] = str(NNFramework.ONNX)

            # Support legacy postprocess_io_map field
            if "postprocess_io_map" in original_model_meta:
                original_model_meta["inverse_postprocess_io_map"] = {
                    v: k for k, v in original_model_meta["postprocess_io_map"].items()
                }

            return original_model_meta

    def get_flow_memento(self) -> Optional[FlowCheckPoint]:
        """Search for the memento on the Har if it finds it returns one else will return None"""
        har_file = tarfile.open(self._path, "r:*")
        temp_dir = Path(tempfile.mkdtemp())
        memento = None
        for member in har_file.getmembers():
            # Check if the member's path starts with the directory name you want to extract
            if member.name.startswith(HARFileNames.FLOW_MEMENTO.value):
                # Extract this member to the temporary directory
                har_file.extract(member, path=temp_dir)
            elif HARFileNames.FLOW_MEMENTO.value in member.name:
                memento = FlowCheckPoint.parse_raw(har_file.extractfile(member.name).read().decode())
        if memento:
            flow_dir = temp_dir / HARFileNames.FLOW_MEMENTO.value
            memento.node_checkpoint.checkpoint.replace_base_path(flow_dir, recursive=True)
        else:
            temp_dir.rmdir()
        return memento

    def get_onnx_model(self, filename_key):
        filename = self._metadata.get(filename_key, None)
        if filename:
            return onnx.load_model(self._har_file.extractfile(filename))

    def get_preprocess_model(self):
        return self.get_onnx_model(HARFileNames.PREPROCESS)

    def get_postprocess_model(self):
        return self.get_onnx_model(HARFileNames.POSTPROCESS)

    def get_nms_config_file(self):
        filename = self._metadata.get(HARFileNames.NMS_CONFIG, None)
        if filename:
            return json.loads(self._har_file.extractfile(filename).read().decode())

    def get_modifications_meta_data_file(self):
        filename = self._metadata.get(HARFileNames.MODIFICATIONS_META_DATA, None)
        if filename:
            return json.loads(self._har_file.extractfile(filename).read().decode())

    def get_lora_weights_metadata_file(self):
        filename = self._metadata.get(HARFileNames.LORA_WEIGHTS_META_DATA, None)
        if filename:
            return json.loads(self._har_file.extractfile(filename).read().decode())

    def get_nms_engine(self):
        return self._metadata.get(HARMetaDataKeys.NMS_ENGINE, None)

    def get_nms_meta_arch(self):
        return self._metadata.get(HARMetaDataKeys.NMS_META_ARCH, None)

    def list(self, verbose=False):
        """
        Print the files in the archive. See `tarfile.list`.
        """
        self._har_file.list(verbose=verbose)
