#!/usr/bin/env python
import copy

from hailo_sdk_common.paths_manager.paths import BaseConfigDirs, BaseConfigPaths, SDKPaths


class ConfigDirs(BaseConfigDirs):
    DIRS_BUILD_INTERNAL = {
        "outputs_dir": ["outputs"],
        "hw_sim_params_auto_dir": ["system_test"],
        "weights_dir": ["data", "weights"],
        "inputs_dir": ["data", "inputs"],
    }

    DIR_SDK_INTERNAL = {
        "hw_sim_params_templates_dir": ["system_test"],
    }

    DIR_BOTH_INTERNAL = {
        "hw_sim_params_meta_dir": ["system_test"],
    }

    DIRS_BUILD_ONLY = {
        "bin_dir": ["bin"],
        "quantized_params_dir": ["quantized_params"],
    }

    DIRS_SDK_ONLY = {
        "templates_dir": ["templates"],
    }

    DIRS_BOTH = {
        "map_dir": ["mapping", "{hw_arch}"],
    }

    def __init__(self, hw_arch):
        super().__init__(hw_arch)
        self._paths = SDKPaths()
        if self.is_internal:
            type(self).DIRS_BUILD_ONLY.update(type(self).DIRS_BUILD_INTERNAL)
            type(self).DIRS_SDK_ONLY.update(type(self).DIR_SDK_INTERNAL)
            type(self).DIRS_BOTH.update(type(self).DIR_BOTH_INTERNAL)
        self._dirs = {}
        for d in [type(self).DIRS_BUILD_ONLY, type(self).DIRS_SDK_ONLY, type(self).DIRS_BOTH]:
            self._dirs.update(self._format_dirs(d))

    @property
    def is_internal(self):
        return self._paths.is_internal


class ConfigPaths(BaseConfigPaths):
    PATHS_INTERNAL = {
        "system_tests_auto": ["{hw_sim_params_auto_dir}", "{model_name}.auto"],
        "system_tests_meta": ["{hw_sim_params_meta_dir}", "{model_name}.meta"],
        "system_tests_templates_db": ["{hw_sim_params_templates_dir}", "templates_db.json"],
    }

    PATHS = copy.deepcopy(BaseConfigPaths.PATHS)
    PATHS.update(
        {
            "probe_script": ["{bin_dir}", "sdk_waves.tcl"],
            "tags_script": ["{bin_dir}", "sdk_tags.tcl"],
            "map_script": ["{map_dir}", "{model_name}.maps"],
            "map_diagram": ["{map_dir}", "{network_name}"],
            "hn_diagram": ["{map_dir}", "{network_name}_hn"],
            "network_graph": ["{map_dir}", "{network_name}.network_graph.pb"],
            "quantized_params": ["{quantized_params_dir}", "{network_name}_params.q.npz"],
            "mapped_graph": ["{map_dir}", "{network_name}.mapped_graph.pb"],
            "boot_jlf": ["{bin_dir}", "{model_name}.boot.jlf"],
            "config_jlf": ["{bin_dir}", "{model_name}.inference.mem"],
            "params_jlf": ["{bin_dir}", "{model_name}.l3_params.mem"],
            "meta_data_jlf": ["{bin_dir}", "{model_name}.meta_data.jlf"],
            "compilation_output_proto": ["{bin_dir}", "{model_name}_jlfs.proto"],
            "hef": ["{bin_dir}", "{model_name}.hef"],
            "compiler_statistics": ["{map_dir}", "{model_name}.compiler_stats.pb"],
        },
    )

    def __init__(self, hw_arch, model_name):
        super().__init__(hw_arch, model_name)
        self._paths = SDKPaths()
        self._config_dirs = ConfigDirs(hw_arch)
        # We don't want to use multiple inheritance to get both ConfigDirs and BaseConfigPaths stuff
        # instead we inherit from BaseConfigPaths and copy interesting properties from ConfigDirs manually
        type(self).DIRS_BUILD_ONLY = self._config_dirs.DIRS_BUILD_ONLY
        type(self).DIRS_SDK_ONLY = self._config_dirs.DIRS_SDK_ONLY
        type(self).DIRS_BOTH = self._config_dirs.DIRS_BOTH
        if self.is_internal:
            type(self).PATHS.update(type(self).PATHS_INTERNAL)
        self._dirs = {}
        for d in [type(self).DIRS_BUILD_ONLY, type(self).DIRS_SDK_ONLY, type(self).DIRS_BOTH]:
            self._dirs.update(self._format_dirs(d))

    @property
    def is_internal(self):
        return self._paths.is_internal


class RunnerWithDirs:
    def __init__(self, hw_arch):
        self._config_dirs = ConfigDirs(hw_arch)


class RunnerWithPaths:
    def __init__(self, hw_arch, model_name):
        self._config_paths = ConfigPaths(hw_arch, model_name)
