#!/bin/bash
readonly table_file="system_reqs_table.log"
readonly log_file="system_reqs_results.log"
readonly log_boundary=" | "
readonly log_found="V$log_boundary"
readonly log_missing="X$log_boundary"
readonly log_warning=" $log_boundary"
readonly req_os="Ubuntu"
readonly req_releases=("20.04" "22.04" "24.04")
readonly req_mem=16
readonly rec_mem=32
readonly req_arch="x86_64"
readonly req_driver=560
readonly req_cuda=12.5
readonly req_cudnn=9

readonly EX_ARGUMENT_ERROR=2
readonly SYSTEM_CHECK_OUTPUT="hailo_system_check.txt"
readonly TABLE_FILE="system_reqs_table.log"

python_version=$(python3 -c "import sys; vinfo=sys.version_info; print('{}.{}'.format(vinfo.major, vinfo.minor))")
readonly apt_reqs=(
    "python3-tk"
    "graphviz"
    "libgraphviz-dev"
    "python${python_version}-dev"
)

declare -a cpu_commands

cpu_commands[0]='avx;0'

declare -a cpu_commands_reasons

cpu_commands_reasons[0]='install TensorFlow'

declare -a compilation_variables

compilation_variables[0]='CC'
compilation_variables[1]='CXX'
compilation_variables[2]='LD'
compilation_variables[3]='AS'
compilation_variables[4]='AR'
compilation_variables[5]='LN'
compilation_variables[6]='DUMP'
compilation_variables[7]='CPY'

error=false

# Logging mechanism

DEBUG=0
INFO=1
WARNING=2
ERROR=3

declare -a LEVEL_NAMES=("Debug" "Info" "Warning" "Error")
declare -a LEVEL_COLORS=("\033[0m" "\033[1;36m" "\033[1;33m" "\033[0;31m")
readonly NC="\033[0m"

function init_logging() {
    readonly LOG_DIR="./.install_logs"
    mkdir -p $LOG_DIR
    readonly LOG_FILE="$(realpath ${LOG_DIR})/hailo_installation_log_$(date +"%Y.%m.%dT%R:%S").log"
    touch $LOG_FILE
}

function emit() {
    level=$1
    message=$2
    log_message="[${LEVEL_NAMES[$level]}] $message"
    color_log_message="[${LEVEL_COLORS[$level]}${LEVEL_NAMES[$level]}${NC}] $message"
    if [ $level -ge $DEBUG ]
    then
        echo -e $log_message &>> $LOG_FILE 2>&1
    fi
    if [ $level -ge $INFO ]
    then
        echo -e $color_log_message
    fi
}

function log_info() {
    emit $INFO "$1"
}

function log_warning() {
    emit $WARNING "$1"
}

function log_error() {
    emit $ERROR "$1"
    emit $ERROR "See full log: ${LOG_FILE}"
    # Exit on error, second argument is the error code.
    if [ "$2" != "0" ]; then
        exit $2
    fi
}

function check_os() {
    # Get OS Distribution:
    dist_id="$(lsb_release -a 2>/dev/null | grep "Distributor ID" | awk '{print $3}')"
    if [[ $dist_id != $req_os ]]; then
        echo "ERROR: Unsupported OS - $dist_id. Only $req_os is supported." 1>&2
    fi

    # Get OS release:
    release="$(lsb_release -a 2> /dev/null | grep "Release" | awk '{print $2}')"
    # Check if the OS release is supported
    for rel in "${req_releases[@]}"
    do
        _supported_release=0
        if [[ "$release" == "$rel" ]]; then
            _supported_release=1
            break
        fi
    done
    if [[ ${_supported_release} == 0 ]]; then
        echo "ERROR: Unsupported release - $release. Only ${req_releases[@]} are supported." 1>&2
    fi

    echo "OS   $req_os  $dist_id    Required" >> $table_file
    echo "Release $req_releases  $release    Required" >> $table_file
}

function check_apt_packages() {
    for package in ${apt_reqs[@]}
    do
        # Find if the current requirement is installed with apt:
        found="$((apt list --installed 2>/dev/null | grep ^$package)|| echo "X")"
        if [[ $found == "X" ]]; then
            echo "ERROR: Requirement $package not found." 1>&2
            echo "$log_error Missing package: $package" >> $log_file
        else
            found="V"
            echo "$log_found Package $package found." >> $log_file
        fi
        echo "Package $package  $found  Required" >> $table_file
    done
}

function check_ram() {
    # Get total RAM size in GB:
    mem=$(free -g -t | grep Total | awk '{print $2}')

    if [ "$req_mem" -gt $mem ]; then
        error=true
        echo "ERROR: The SDK requires $req_mem GB of RAM ($rec_mem GB recommended), while this system has only $mem GB." 1>&2
        echo "$log_missing Insufficient RAM: $req_mem GB of RAM are required, only $mem GB available." >> $log_file
    else
        if [ "$rec_mem" -gt $mem ]; then
            echo "WARNING: It is recommended to have $rec_mem GB of RAM, while this system has only $mem GB." 1>&2
            echo "$log_warning Available RAM ($mem GB) below recommended amount ($rec_mem GB)." >> $log_file
        else
            echo "$log_found Available RAM ($mem GB) is sufficient, and within recommendation ($rec_mem GB)." >> $log_file
        fi
    fi
    echo "RAM(GB) $req_mem  $mem    Required" >> $table_file
    echo "RAM(GB) $rec_mem  $mem    Recommended" >> $table_file

}

function check_disk() {
    # Can't find any specifications for disk in user guide.
    disk_memory=$(df -x squashfs --total -BG | grep total | awk '{print $4}')
    echo $disk_memory
}

function check_cpu() {
    # Get CPU architecture:
    cpu_arch=$(lscpu | grep Architecture | awk '{print $2}')
    if [ $cpu_arch != $req_arch ]; then
        error=true
        echo "ERROR: CPU architecture required is $req_arch, found $cpu_arch." 1>&2
        echo "$log_missing Unsupported CPU architecture: $cpu_arch. The supported architecture is $req_arch." >> $log_file
    else
        echo "$log_found CPU architecture $cpu_arch is supported." >> $log_file
    fi
    echo "CPU-Arch    $req_arch $cpu_arch   Required" >> $table_file
}

function check_gpu() {
    #Check for GPU:
    nvidia-smi >>/dev/null 2>&1
    if [ "$?" != '0' ]; then
        gpu=false
        echo "INFO: No GPU connected."
        echo "$log_warning GPU Requirements are not checked- no GPU connected." >> $log_file
    else
        gpu=true
        # Get GPU details:
        driver_ver=$(nvidia-smi -q | grep "Driver Version" | awk '{print $4}')
        driver_ver=${driver_ver:0:3}
	cuda_ver=$(nvcc --version | grep "release" | awk '{print $5}' | tr "," " ")
        cudnn_ver_major=$(cat /usr/include/{cudnn,cudnn_version}.h 2> /dev/null | grep CUDNN_MAJOR | awk 'NR==1 {print $3}')
        cudnn_ver_minor=$(cat /usr/include/{cudnn,cudnn_version}.h 2> /dev/null | grep CUDNN_MINOR | awk 'NR==1 {print $3}')
        cudnn_ver=$cudnn_ver_major.$cudnn_ver_minor
        # Compare to requirements:
        if [ $req_driver -gt $driver_ver ]; then
            echo "WARNING: GPU driver version should be $req_drvier or higher, found $driver_ver." 1>&2
            echo "$log_missing GPU driver version should be $req_drvier or higher, found $driver_ver." >> $log_file
        else
            echo "$log_found GPU driver version is $driver_ver." >> $log_file
        fi
        if [ $req_cuda != $cuda_ver ]; then
            echo "WARNING: CUDA version should be $req_cuda or higher, found $cuda_ver." 1>&2
            echo "$log_missing CUDA version should be $req_cuda or higher, found $cuda_ver." >> $log_file
        else
            echo "$log_found CUDA version is $cuda_ver." >> $log_file
        fi
        if [ $req_cudnn != $cudnn_ver ]; then
            echo "WARNING: CUDNN version should be $req_cudnn or higher, found $cudnn_ver." 1>&2
            echo "$log_missing CUDNN version should be $req_cudnn or higher, found $cudnn_ver." >> $log_file
        else
            echo "$log_found CUDNN version is $cudnn_ver." >> $log_file
        fi
        echo "GPU-Driver    $req_driver $driver_ver Recommended" >> $table_file
        echo "CUDA    $req_cuda $cuda_ver Recommended" >> $table_file
        echo "CUDNN    $req_cudnn $cudnn_ver Recommended" >> $table_file
    fi
}

function check_compilation_variables() {
    for variable in ${compilation_variables[@]}
    do
        # verify that enviroment variables used for compilation have not been set
        if [[ ${!variable+x} ]]; then
            error=true
            is_set='set'
            echo "ERROR: Environment variable "$variable" is set to \"${!variable}\", please unset and re-run."
            echo "$log_warning Environment variable $variable is set. This variable is used in SDK installation and needs to be unset." >> $log_file

        else
            is_set='unset'
            echo "$log_found Environment variable $variable is unset." >> $log_file
        fi
        echo "Var:$variable  unset  ${is_set}  Required" >> $table_file
    done
}

function check_cpu_instructions() {
    for opcode in ${cpu_commands[@]}
    do
        IFS=";" read -r -a arr <<< "${opcode}"
        command_name="${arr[0]}"
        reason=${cpu_commands_reasons["${arr[1]}"]}
        lscpu | grep $command_name > /dev/null
        if [ "$?" != "0" ]; then
            error=true
            echo "ERROR: CPU flag $command_name is not supported in this CPU, and is required to $reason." 1>&2
            echo "$log_missing Required $command_name CPU flag is not supported in this CPU, and is required to $reason." >> $log_file
            found="X"
        else
            echo "$log_found Required $command_name CPU flag is supported." >> $log_file
            found="V"
        fi
        echo "CPU-flag    $command_name $found  Required" >> $table_file
    done
}

function check_system_requirements() {
    rm -f $log_file $table_file &> /dev/null

    echo "Component   Requirement   Found" > $table_file
    echo "========== ========== ========== ========== " >> $table_file

    echo -e "HAILO System requirements check - log\n" > $log_file

    check_os
    check_apt_packages
    check_ram
    check_cpu
    check_cpu_instructions
    check_gpu
    check_compilation_variables
}

function main() {
    init_logging
    check_system_requirements &> $SYSTEM_CHECK_OUTPUT
    req_error=false
    req_warning=false
    while read p; do
        if [[ $p == WARNING* ]]; then
            log_warning "${p#"WARNING: "}"
            req_warning=true
        elif [[ $p == INFO* ]]; then
            log_info "${p#"INFO: "}"
        elif [[ $p == ERROR* ]]; then
            log_error "${p#"ERROR: "}" 0
            req_error=true
        fi
    done < $SYSTEM_CHECK_OUTPUT
    rm $SYSTEM_CHECK_OUTPUT -rf > /dev/null
    if [[ "$req_error" == true ]] || [[ "$req_warning" == true ]]; then
        column -t -s' ' $TABLE_FILE
        rm $TABLE_FILE -rf > /dev/null
        if [[ "$req_error" == true ]]; then
            while true; do
                read -t 60 -p "System requirements check failed (see table above for details). Continue? [Y/n]" yn
                if [ "$?" != 0 ]; then
                    log_info "No response - exitting."
                    exit $EX_COMPATABILITY_ERROR
                fi
                case $yn in
                    [Yy]* ) break;;
                    [Nn]* ) exit 1;;
                    *) break;;
                esac
            done
        fi
    fi

    rm -f $table_file $log_file &> /dev/null
}

main
