import numpy as np

from hailo_model_optimization.acceleras.model_optimization_config.mo_config_model import CalibrationConfig
from hailo_model_optimization.acceleras.utils.acceleras_definitions import BiasMode, ModelOptimizationCommand
from hailo_sdk_client.allocator.estimator import Estimator
from hailo_sdk_client.sdk_backend.profiler.base_data_extractor import BaseDataExtractor
from hailo_sdk_common.hailo_nn.hn_definitions import LayerType


class OptimizationDataExtractor(BaseDataExtractor):
    FIELD_NOT_AVAILABLE = "N/A"

    class ACCURACY_FIELDS:
        WEIGHTS_HISTOGRAM = "weights_histogram"
        NUMBER_OF_BITS = "number_of_bits"
        RANGES = "ranges"
        DETAILS = "details"
        NATIVE = "native"
        NUMERIC = "numeric"
        HIST = "hist"
        BIN_EDGES = "bin_edges"
        INPUT_BITS = "input"
        WEIGHT_BITS = "weights"
        BIAS_BITS = "bias"
        OUTPUT_BITS = "output"
        EFFECTIVE_BITS = "effective"
        SHAPES = "shapes"
        INPUT_SHAPES = "input_shapes"
        OUTPUT_SHAPES = "output_shapes"
        KERNEL_SHAPE = "kernel_shape"
        BN_INFO = "bn_info"
        BN_ENABLED = "bn_enabled"
        ACTIVATION_INFO = "activation_info"
        ACTIVATION_TYPE = "activation_type"
        PARTIAL_DATA = "partial_data"
        ACTIVATIONS_HISTOGRAM = "activations_histogram"
        SNR_PER_LAYER = "snr_per_layer"
        SAMPLED_TENSORS = "sampled_tensors"
        NOISE_RESULTS = "noise_results"
        MO_ALGOS = "mo_algos"
        EQUALIZATION = "equalization"
        BIAS_CORRECTION = "bias_correction"
        ADAROUND = "adaround"
        FINETUNE = "finetune"
        ACTIVATION_CLIPPING_MIN = "clipped_activation_min"
        ACTIVATION_CLIPPING_MAX = "clipped_activation_max"
        WEIGHT_CLIPPING_MIN = "clipped_weight_min"
        WEIGHT_CLIPPING_MAX = "clipped_weight_max"
        INPUT_RANGE_STATS_MIN = "input_range_stats_min"
        INPUT_RANGE_STATS_MAX = "input_range_stats_max"
        OUTPUT_RANGE_STATS_MIN = "output_range_stats_min"
        OUTPUT_RANGE_STATS_MAX = "output_range_stats_max"
        INPUTS_RANGE_MIN_LIST = "input_range_min_list"
        INPUTS_RANGE_MAX_LIST = "input_range_max_list"
        CHECK_BN_FAILED = "check_bn_failed"
        SPARSITY = "sparsity"

    def __init__(
        self,
        hn,
        params,
        translated_params,
        hailo_optimized_params,
        statistics_params,
        mo_flavor,
        flavor_config,
        optimization_commands,
    ):
        self._hn = hn
        self._params = dict(params) if params else {}
        self._translated_params = dict(translated_params) if translated_params else {}
        self._statistics_params = dict(statistics_params) if statistics_params else {}
        self._hailo_optimized_params = dict(hailo_optimized_params) if statistics_params else {}
        self._mo_flavor = mo_flavor
        self._flavor_config = flavor_config
        self._optimization_commands = optimization_commands

    def update(self, export):
        accuracy_data = {
            self.ACCURACY_FIELDS.DETAILS: self._get_mo_details(),
            self.ACCURACY_FIELDS.WEIGHTS_HISTOGRAM: {
                self.ACCURACY_FIELDS.NATIVE: {},
                self.ACCURACY_FIELDS.NUMERIC: {},
            },
            self.ACCURACY_FIELDS.NUMBER_OF_BITS: {},
            self.ACCURACY_FIELDS.RANGES: {},
            self.ACCURACY_FIELDS.SHAPES: {},
            self.ACCURACY_FIELDS.ACTIVATION_INFO: {},
            self.ACCURACY_FIELDS.BN_INFO: {},
            self.ACCURACY_FIELDS.MO_ALGOS: {},
            self.ACCURACY_FIELDS.ACTIVATIONS_HISTOGRAM: {},
            self.ACCURACY_FIELDS.SNR_PER_LAYER: {},
            self.ACCURACY_FIELDS.SAMPLED_TENSORS: {},
            self.ACCURACY_FIELDS.SPARSITY: {},
            self.ACCURACY_FIELDS.NOISE_RESULTS: {},
            self.ACCURACY_FIELDS.PARTIAL_DATA: True,
        }

        self._update_layers_data(accuracy_data)

        # Add additional data produced by hailo layer analysis tool
        if self._statistics_params:
            accuracy_data.update(self._parse_statistics_analysis_data())

        export["accuracy_data"] = accuracy_data

    def _get_mo_details(self):
        return {
            Estimator.METADATA_FIELDS.OPTIMIZATION_LEVEL: self._get_optimization_level(),
            Estimator.METADATA_FIELDS.COMPRESSION_LEVEL: self._get_compression_level(),
            Estimator.METADATA_FIELDS.COMPRESSION_RATE: self._get_compression_rate(),
            Estimator.METADATA_FIELDS.CALIBRATION: self._get_calibration_size(),
            self.ACCURACY_FIELDS.CHECK_BN_FAILED: self._get_check_bn_failed(),
        }

    def _get_compression_level(self):
        if self._mo_flavor:
            return self._mo_flavor.compression_level

        return self.FIELD_NOT_AVAILABLE

    def _get_optimization_level(self):
        if self._mo_flavor:
            return self._mo_flavor.optimization_level

        return self.FIELD_NOT_AVAILABLE

    def _get_calibration_size(self):
        if "calibration" in self._optimization_commands:
            if "calibset_size" in self._optimization_commands["calibration"]:
                return self._optimization_commands["calibration"]["calibset_size"]

        return CalibrationConfig.get_default().calibset_size

    def _get_compression_rate(self):
        if self._flavor_config is not None:
            if ModelOptimizationCommand.compression_params.value in self._flavor_config:
                return self._flavor_config[ModelOptimizationCommand.compression_params.value]["auto_4bit_weights_ratio"]

        return self.FIELD_NOT_AVAILABLE

    def _parse_statistics_analysis_data(self):
        noise_results = {}
        activations_histogram = {
            self.ACCURACY_FIELDS.NATIVE: {},
            self.ACCURACY_FIELDS.NUMERIC: {},
        }
        sampled_tensors = {}
        sparsity = {
            self.ACCURACY_FIELDS.NATIVE: {},
            self.ACCURACY_FIELDS.NUMERIC: {},
        }
        snr_per_layer = {}
        partial_data = True

        for full_key, value in self._statistics_params.items():
            if full_key == "params_kind":
                continue
            split_key = full_key.split("/")
            layer, algo, key = "/".join(split_key[:2]), split_key[2], "/".join(split_key[3:])
            if algo != "layer_noise_analysis":
                continue
            if key.startswith("noise_results"):
                partial_data = False
                out_layer = key[len("noise_results") + 1 :]
                if out_layer not in noise_results:
                    noise_results[out_layer] = {}
                noise_results[out_layer][layer] = value[0].tolist()
            elif key.startswith("histogram"):
                if layer not in activations_histogram[self.ACCURACY_FIELDS.NATIVE]:
                    activations_histogram[self.ACCURACY_FIELDS.NATIVE][layer] = {}
                info = key[len("histogram") + 1 :]
                activations_histogram[self.ACCURACY_FIELDS.NATIVE][layer][info] = value[0].tolist()
            elif key.startswith("quantized_histogram"):
                info = key[len("quantized_histogram") + 1 :]
                if info == "unique_count":
                    pass
                else:
                    if layer not in activations_histogram[self.ACCURACY_FIELDS.NUMERIC]:
                        activations_histogram[self.ACCURACY_FIELDS.NUMERIC][layer] = {}
                    activations_histogram[self.ACCURACY_FIELDS.NUMERIC][layer][info] = value[0].tolist()
            elif key.startswith("sample"):
                if layer not in sampled_tensors:
                    sampled_tensors[layer] = {}
                info = key[len("sample") + 1 :]
                sampled_tensors[layer][info] = value[0].tolist()
            elif key.startswith("sparsity"):
                info = key[len("sparsity") + 1 :]
                sparsity[info][layer] = value[0].tolist()
            elif key == "snr":
                snr_per_layer[layer] = value[0].tolist()

        return {
            self.ACCURACY_FIELDS.ACTIVATIONS_HISTOGRAM: activations_histogram,
            self.ACCURACY_FIELDS.SNR_PER_LAYER: snr_per_layer,
            self.ACCURACY_FIELDS.SAMPLED_TENSORS: sampled_tensors,
            self.ACCURACY_FIELDS.SPARSITY: sparsity,
            self.ACCURACY_FIELDS.NOISE_RESULTS: noise_results,
            self.ACCURACY_FIELDS.PARTIAL_DATA: partial_data,
        }

    def _update_layers_data(self, accuracy_data):
        for hn_layer in self._hn:
            self._update_native_weights_histogram(hn_layer, accuracy_data)
            self._update_numeric_weights_histogram(hn_layer, accuracy_data)
            self._update_bits_data(hn_layer, accuracy_data)
            self._update_ranges(hn_layer, accuracy_data)
            self._update_shapes(hn_layer, accuracy_data)
            self._update_bn_info(hn_layer, accuracy_data)
            self._update_activation_info(hn_layer, accuracy_data)
            self._update_mo_algos(hn_layer, accuracy_data)

    def _update_native_weights_histogram(self, hn_layer, accuracy_data):
        self._update_weights_histogram(hn_layer, accuracy_data, self._params, self.ACCURACY_FIELDS.NATIVE)

    def _update_numeric_weights_histogram(self, hn_layer, accuracy_data):
        self._update_weights_histogram(
            hn_layer, accuracy_data, self._translated_params, self.ACCURACY_FIELDS.NUMERIC, power_of_two=True
        )

    def _update_weights_histogram(self, hn_layer, accuracy_data, params, key, power_of_two=False):
        res = self._get_kernel_hist(hn_layer.name, params, power_of_two=power_of_two)
        if res is not None:
            hist, bin_edges = res
            accuracy_data[self.ACCURACY_FIELDS.WEIGHTS_HISTOGRAM][key][hn_layer.name] = {
                self.ACCURACY_FIELDS.HIST: hist,
                self.ACCURACY_FIELDS.BIN_EDGES: bin_edges,
            }
        else:
            accuracy_data[self.ACCURACY_FIELDS.WEIGHTS_HISTOGRAM][key][hn_layer.name] = self.FIELD_NOT_AVAILABLE

    def _get_kernel_hist(self, hn_layer_name, params, power_of_two=False):
        kernel = self._get_kernel(hn_layer_name, params)
        if kernel is not None:
            half_bins = 2 ** np.ceil(np.log2(np.maximum(np.max(np.abs(kernel), initial=0.0), 1.0)))
            hist_range = (-half_bins, half_bins - 1) if power_of_two else None
            bins = int(np.minimum(2 * half_bins, 512)) if power_of_two else 512
            hist, bin_edges = np.histogram(kernel, bins=bins - 1, range=hist_range)
            bin_edges = [int(x) for x in bin_edges] if power_of_two else bin_edges.tolist()
            return hist.tolist(), bin_edges

    @staticmethod
    def _get_kernel(hn_layer_name, params):
        return params.get(hn_layer_name + "/kernel:0")

    def _get_kernel_scale(self, hn_layer_name):
        keys = [
            f"{hn_layer_name}/conv_op/kernel_scale:0",
            f"{hn_layer_name}/avgpool_op/kernel_scale:0",
            f"{hn_layer_name}/crosscorrelation_dw_op/input_scale:1:0",
            f"{hn_layer_name}/matmul_op/input_scale:1:0",
            f"{hn_layer_name}/mock_op/kernel_scale:0",
            f"{hn_layer_name}/reduce_sum_op/kernel_scale:0",
            f"{hn_layer_name}/kernel_scale:0",
        ]
        for full_key in keys:
            if full_key in self._translated_params:
                return self._translated_params[full_key]
            if full_key in self._hailo_optimized_params:
                return self._hailo_optimized_params[full_key]

    @staticmethod
    def _get_conv_kernel(hn_layer_name, params):
        return params.get(hn_layer_name + "/conv_kernel:0")

    def _update_bits_data(self, hn_layer, accuracy_data):
        weight_bits = self._get_weight_bits(hn_layer)

        accuracy_data[self.ACCURACY_FIELDS.NUMBER_OF_BITS][hn_layer.name] = {
            self.ACCURACY_FIELDS.INPUT_BITS: self._get_input_activation_bits(hn_layer),
            self.ACCURACY_FIELDS.WEIGHT_BITS: weight_bits,
            self.ACCURACY_FIELDS.BIAS_BITS: self._get_bias_bits(hn_layer, weight_bits),
            self.ACCURACY_FIELDS.OUTPUT_BITS: self._get_output_activation_bits(hn_layer),
            self.ACCURACY_FIELDS.EFFECTIVE_BITS: self._get_effective_bits(hn_layer),
        }

    def _update_ranges(self, hn_layer, accuracy_data):
        input_min, input_max = self._get_input_range(hn_layer)
        output_min, output_max = self._get_output_range(hn_layer)
        kernel_min, kernel_max = self._get_kernel_min_range(hn_layer), self._get_kernel_max_range(hn_layer)
        activation_clipping_min, activation_clipping_max = self._get_activation_clipping_range(hn_layer)
        weight_clipping_min, weight_clipping_max = self._get_weight_clipping_range(hn_layer)
        stats_input_min, stats_input_max = self._get_stats_input_range(hn_layer)
        stats_output_min, stats_output_max = self._get_stats_output_range(hn_layer)
        if stats_output_min != self.FIELD_NOT_AVAILABLE and np.isinf(stats_output_min):
            stats_output_min = output_min

        accuracy_data[self.ACCURACY_FIELDS.RANGES][hn_layer.name] = {
            Estimator.FIELDS.INPUT_RANGE_MIN: input_min[0],
            Estimator.FIELDS.INPUT_RANGE_MAX: input_max[0],
            Estimator.FIELDS.KERNEL_RANGE_MIN: kernel_min,
            Estimator.FIELDS.KERNEL_RANGE_MAX: kernel_max,
            Estimator.FIELDS.OUTPUT_RANGE_MIN: output_min,
            Estimator.FIELDS.OUTPUT_RANGE_MAX: output_max,
            self.ACCURACY_FIELDS.ACTIVATION_CLIPPING_MIN: activation_clipping_min,
            self.ACCURACY_FIELDS.ACTIVATION_CLIPPING_MAX: activation_clipping_max,
            self.ACCURACY_FIELDS.WEIGHT_CLIPPING_MIN: weight_clipping_min,
            self.ACCURACY_FIELDS.WEIGHT_CLIPPING_MAX: weight_clipping_max,
            self.ACCURACY_FIELDS.INPUT_RANGE_STATS_MIN: stats_input_min,
            self.ACCURACY_FIELDS.INPUT_RANGE_STATS_MAX: stats_input_max,
            self.ACCURACY_FIELDS.OUTPUT_RANGE_STATS_MIN: stats_output_min,
            self.ACCURACY_FIELDS.OUTPUT_RANGE_STATS_MAX: stats_output_max,
            self.ACCURACY_FIELDS.INPUTS_RANGE_MIN_LIST: input_min,
            self.ACCURACY_FIELDS.INPUTS_RANGE_MAX_LIST: input_max,
        }

    def _update_shapes(self, hn_layer, accuracy_data):
        layer_shapes_info = {
            self.ACCURACY_FIELDS.INPUT_SHAPES: hn_layer.input_shapes,
            self.ACCURACY_FIELDS.OUTPUT_SHAPES: hn_layer.output_shapes,
        }
        if hasattr(hn_layer, "kernel_shape") and hn_layer.kernel_shape is not None:
            layer_shapes_info[self.ACCURACY_FIELDS.KERNEL_SHAPE] = hn_layer.kernel_shape

        accuracy_data[self.ACCURACY_FIELDS.SHAPES][hn_layer.name] = layer_shapes_info

    def _update_bn_info(self, hn_layer, accuracy_data):
        accuracy_data[self.ACCURACY_FIELDS.BN_INFO][hn_layer.name] = {
            self.ACCURACY_FIELDS.BN_ENABLED: hn_layer.bn_enabled,
        }

    def _update_activation_info(self, hn_layer, accuracy_data):
        if hasattr(hn_layer, "activation") and hn_layer.activation is not None:
            activation_value = hn_layer.activation.value
        else:
            activation_value = self.FIELD_NOT_AVAILABLE

        accuracy_data[self.ACCURACY_FIELDS.ACTIVATION_INFO][hn_layer.name] = {
            self.ACCURACY_FIELDS.ACTIVATION_TYPE: activation_value,
        }

    def _update_mo_algos(self, hn_layer, accuracy_data):
        equalization_result = self._statistics_params.get(hn_layer.name + "/equalization/source", False)
        bias_correction_result = self._statistics_params.get(hn_layer.name + "/bias_correction/successfully_run", False)
        adaround_result = self._statistics_params.get(hn_layer.name + "/adaround/successfully_run", False)
        fine_tune_result = self._statistics_params.get(hn_layer.name + "/fine_tune/successfully_run", False)
        accuracy_data[self.ACCURACY_FIELDS.MO_ALGOS][hn_layer.name] = {
            self.ACCURACY_FIELDS.EQUALIZATION: bool(equalization_result),
            self.ACCURACY_FIELDS.BIAS_CORRECTION: bool(bias_correction_result),
            self.ACCURACY_FIELDS.ADAROUND: bool(adaround_result),
            self.ACCURACY_FIELDS.FINETUNE: bool(fine_tune_result),
        }

    def _get_bias_bits(self, hn_layer, weight_bits):
        bias_mode = hn_layer.precision_config.bias_mode
        if weight_bits != self.FIELD_NOT_AVAILABLE and bias_mode in [
            BiasMode.double_scale_initialization,
            BiasMode.double_scale_decomposition,
        ]:
            return 2 * weight_bits

        return weight_bits

    def _get_weight_bits(self, hn_layer):
        keys = [
            "conv_op/weight_bits",
            "avgpool_op/weight_bits",
            "conv_op_0/weight_bits",
            "conv_op_a/weight_bits",
            "mock_op/weight_bits",
            "mock_op1/weight_bits",
            "reduce_sum_op/weight_bits",
            "resize_bilinear_mac_op/weight_bits",
            "elementwise_add_op/weight_bits",
            "elementwise_sub_op/weight_bits",
            "weight_bits",
        ]
        return self._get_int_value_from_possible_keys(hn_layer, keys)

    def _get_int_value_from_possible_keys(self, hn_layer, possible_keys):
        for key in possible_keys:
            full_key = f"{hn_layer.name}/{key}:0"
            if full_key in self._translated_params:
                value = int(self._translated_params[full_key])
                # WA: set number of bits to 16 instead of 15
                return 16 if (value == 15 and "bits" in key) else value
            if full_key in self._hailo_optimized_params:
                value = int(self._hailo_optimized_params[full_key])
                # WA: set number of bits to 16 instead of 15
                return 16 if (value == 15 and "bits" in key) else value

        return self.FIELD_NOT_AVAILABLE

    def _get_input_activation_bits(self, hn_layer):
        keys = [
            "input_passthough_op/input_bits:0",
            "passthru_op_in/input_bits:0",
            "input_op/output_bits:0",
            "conv_op/input_bits:0",
            "avgpool_op/input_bits:0",
            "passthru_op_in_0/input_bits:0",
            "const_op/input_bits:0",
            "crosscorrelation_dw_op/input_bits:0",
            "flatten_op/input_bits:0",
            "conv_op_a/input_bits:0",
            "in_passthru_op/input_bits:0",
            "slice_op/input_bits:0",
            "matmul_op/input_bits:0",
            "mock_op/input_bits:0",
            "reduce_sum_op/input_bits:0",
            "resize_bilinear_mac_op/input_bits:0",
            "elementwise_add_op/input_bits:0",
            "elementwise_sub_op/input_bits:0",
            "argmax_op/input_bits:0",
            "concat_op/input_bits:0",
            "depth_to_space_op/input_bits:0",
            "padding_op/input_bits:0",
            "feature_permute_op/input_bits:0",
            "format_conversion_op/input_bits:0",
            "output_op/input_bits:0",
            "normalization_op/input_bits:0",
            "maxpool_op/input_bits:0",
            "nms_op/input_bits:0",
            "reduce_max_op/input_bits:0",
            "resize_bilinear_ppu_op/input_bits:0",
            "resize_nearest_neighbor_op/input_bits:0",
            "passthru_op/input_bits:0",
            "softmax_op/input_bits:0",
            "space_to_depth_op/input_bits:0",
            "input_bits",
            "input_activation_bits",
            "input_activation_bits_0",
        ]
        return self._get_int_value_from_possible_keys(hn_layer, keys)

    def _get_output_activation_bits(self, hn_layer):
        keys = [
            "passthru_op/output_bits:0",
            "output_op/output_bits:0",
            "act_op/output_bits:0",
            "out_slice_op_0/output_bits:0",
            "slice_op_0/output_bits:0",
            "const_op/output_bits:0",
            "argmax_op/output_bits:0",
            "concat_op/output_bits:0",
            "depth_to_space_op/output_bits:0",
            "padding_op/output_bits:0",
            "feature_permute_op/output_bits:0",
            "format_conversion_op/output_bits:0",
            "normalization_op/output_bits:0",
            "maxpool_op/output_bits:0",
            "nms_op/output_bits:0",
            "reduce_max_op/output_bits:0",
            "resize_bilinear_ppu_op/output_bits:0",
            "resize_nearest_neighbor_op/output_bits:0",
            "softmax_op/output_bits:0",
            "space_to_depth_op/output_bits:0",
            "input_op/output_bits:0",
            "output_bits",
            "output_activation_bits",
        ]
        return self._get_int_value_from_possible_keys(hn_layer, keys)

    def _get_effective_bits(self, hn_layer):
        unique_count = self._statistics_params.get(
            f"{hn_layer.name}/layer_noise_analysis/quantized_histogram/unique_count",
            None,
        )
        quantized_limvals = self._statistics_params.get(f"{hn_layer.name}/layer_noise_analysis/quantized_limvals", None)
        if unique_count is not None:
            if unique_count[0].tolist() < 1:
                return 0
            return int(np.ceil(np.log2(unique_count[0].tolist())))
        elif quantized_limvals is not None:
            # In case unique_count isn't available, use the range of the quantized_limvals to approximate it's value.
            range = quantized_limvals[0].tolist()[1] - quantized_limvals[0].tolist()[0]
            if range < 1:
                return 0
            return int(np.ceil(np.log2(range)))
        return self.FIELD_NOT_AVAILABLE

    def _get_input_range(self, hn_layer):
        ranges = []
        if len(hn_layer.inputs) > 1:
            for input_layer in hn_layer.inputs:
                ranges.append(self._get_io_range(f"{input_layer}/limvals_out:0"))
        else:
            ranges.append(self._get_io_range(f"{hn_layer.name}/limvals_in:0"))

        return list(zip(*ranges))

    def _get_output_range(self, hn_layer):
        return self._get_io_range(f"{hn_layer.name}/limvals_out:0")

    def _get_stats_input_range(self, hn_layer):
        ranges = []
        if len(hn_layer.inputs) > 0:
            for i in range(len(hn_layer.inputs)):
                key = f"{hn_layer.name}/stats/input_{i}/stats_limvals:0"
                ranges.append(self._get_io_range(key))
        else:
            ranges.append((self.FIELD_NOT_AVAILABLE,) * 2)

        return list(zip(*ranges))

    def _get_stats_output_range(self, hn_layer):
        original_key = f"{hn_layer.name}/stats/original_output_0/stats_limvals:0"
        if original_key in self._translated_params:
            return self._get_io_range(
                f"{hn_layer.name}/stats/original_output_0/stats_limvals:0",
                self._translated_params,
            )
        if original_key in self._hailo_optimized_params:
            return self._get_io_range(
                f"{hn_layer.name}/stats/original_output_0/stats_limvals:0",
                self._hailo_optimized_params,
            )
        return self._get_io_range(f"{hn_layer.name}/stats/output_0/stats_limvals:0")

    def _get_activation_clipping_range(self, hn_layer):
        return self._get_io_range(f"{hn_layer.name}/clip_statistics/clip_values", self._statistics_params)

    def _get_weight_clipping_range(self, hn_layer):
        # Note: limvals can be min/max or percentile or lossy/manual during the optimization process,
        # while the reported value should be derived from the actual kernel values after scaling.
        kernel = self._get_kernel(hn_layer.name, self._translated_params)
        if hn_layer.op == LayerType.deconv:
            kernel = self._get_conv_kernel(hn_layer.name, self._translated_params)
        kernel_scale = self._get_kernel_scale(hn_layer.name)
        if kernel is not None and kernel.size > 0 and kernel_scale is not None and kernel_scale.size > 0:
            scaled_kernel = kernel.astype(np.float32) * kernel_scale.astype(np.float32)
            return float(np.min(scaled_kernel)), float(np.max(scaled_kernel))

        return [self.FIELD_NOT_AVAILABLE] * 2

    def _get_io_range(self, io_range_key, params=None):
        params_list = [self._translated_params, self._hailo_optimized_params] if params is None else [params]
        for params_sel in params_list:
            if io_range_key in params_sel:
                min_value, max_value = params_sel[io_range_key][:2]
                return float(min_value), float(max_value)

        return [self.FIELD_NOT_AVAILABLE] * 2

    def _get_kernel_max_range(self, hn_layer):
        return self._get_kernel_range(hn_layer, np.max)

    def _get_kernel_min_range(self, hn_layer):
        return self._get_kernel_range(hn_layer, np.min)

    def _get_kernel_range(self, hn_layer, method):
        kernel = self._get_kernel(hn_layer.name, self._params)
        if kernel is not None:
            return float(method(kernel))

        return self.FIELD_NOT_AVAILABLE

    def _get_check_bn_failed(self):
        """
        Get check bn failed from params statistics, which indicates whether the calibration data not match BN info.

        Returns
            If check_bn/failed not exist for all layers returns "N/A".
            Else, if there is a layer with check_bn/failed is true, returns True.
            Otherwise, return False (check_bn/failed exist for some layers but is False for all of them).

        """
        check_bn_failed = self.FIELD_NOT_AVAILABLE
        for key in self._statistics_params.keys():
            if key.endswith("/quantization_checker/check_bn/failed"):
                check_bn_failed = bool(self._statistics_params[key])
                if check_bn_failed:
                    return check_bn_failed

        return check_bn_failed
