#!/usr/bin/env python
import collections
import fnmatch
import os
from abc import ABC, abstractmethod
from enum import Enum, IntEnum

from hailo_model_optimization.acceleras.utils.acceleras_definitions import (
    FormatConversionType,
    ModelOptimizationCommand,
)
from hailo_sdk_client.allocator.allocator_params import AllocatorParams, AllocatorStrategy
from hailo_sdk_client.allocator.context_compilation_params import ContextCompilationParams
from hailo_sdk_client.allocator.context_resources_params import ContextResourcesParams
from hailo_sdk_client.allocator.context_switch_params import ContextSwitchParams
from hailo_sdk_client.allocator.hef_params import HefParams
from hailo_sdk_client.allocator.logger_params import LoggerParams
from hailo_sdk_client.allocator.pb_wrapper import PbWrapper
from hailo_sdk_client.allocator.performance_params import (
    DEFAULT_OPTIMIZATION_LEVEL,
    OptimizationLevel,
    PerformanceParams,
    get_max_compiler_optimization_level,
)
from hailo_sdk_client.allocator.platform_params import PlatformParams
from hailo_sdk_client.allocator.resources_params import AutoDouble, ResourcesParams, convert_to_auto_double
from hailo_sdk_client.sdk_backend.sdk_backend_exceptions import AllocatorScriptParserException
from hailo_sdk_common.hailo_nn.hn_definitions import ShapeSplitterType
from hailo_sdk_common.hailo_nn.hn_layers_params import CompilationParams
from hailo_sdk_common.hailo_nn.tools_params import AutoInt, AutoVariablePolicy, convert_to_auto_int, param_str
from hailo_sdk_common.logger.logger import DeprecationVersion, default_logger
from hailo_sdk_common.versions import LATEST_HEF_VERSION

ParamInfo = collections.namedtuple("ParamInfo", "is_supported ,name, message")


class SupportedCommands(str, Enum):
    PLACE = "place"
    SHORTCUT = "shortcut"
    PORTAL = "portal"
    OUTPUT_MUX = "output_mux"
    DEFUSE = "defuse"
    DEFUSE_BLOCK = "defuse_block"
    CONCAT = "concat"
    FROM_TF = "from_tf"
    BUFFERS = "buffers"
    STRATEGY = "strategy"
    COMPILATION_PARAM = "compilation_param"
    RESOURCES_PARAM = "resources_param"
    QUANTIZATION_PARAM = ModelOptimizationCommand.quantization_param.value
    ALLOCATOR_PARAM = "allocator_param"
    COMPRESSION_PARAMS = ModelOptimizationCommand.compression_params.value
    PRINT_BUFFERS = "print_buffers"
    OPTIMIZE_BUFFERS = "optimize_buffers"
    BUFFER_CALC_PARAM = "buffer_calc_param"
    CONTEXT = "context"
    OUTPUT_LAYER = "output_layer"
    MERGE = "merge"
    FORMAT_CONVERSION = "format_conversion"
    FEATURE_SPLITTER = "feature_splitter"
    FORCE_MAPPING = "force_mapping"
    CONTEXT_SWITCH_PARAM = "context_switch_param"
    HEF_PARAM = "hef_param"
    CONTEXT_COMPILATION_PARAM = "compilation_param"
    CONTEXT_RESOURCES_PARAM = "resources_param"
    DDR = "ddr"
    L4_PORTAL = "l4_portal"
    FORCE_ROUTE = "force_route"
    LOGGER_PARAM = "logger_param"
    INTERNAL_ALLOCATOR_PARAM = "internal_allocator_param"
    COLLAPSE = "collapse"
    INTERNAL_CONTEXT_SWITCH_PARAM = "internal_context_switch_param"
    MUX_DEMUX = "mux_demux"
    PRE_QUANTIZATION_OPTIMIZATION = ModelOptimizationCommand.pre_quantization_optimization.value
    POST_QUANTIZATION_OPTIMIZATION = ModelOptimizationCommand.post_quantization_optimization.value
    MODEL_OPTIMIZATION_CONFIG = ModelOptimizationCommand.model_optimization_config.value
    MODEL_OPTIMIZATION_FLAVOR = ModelOptimizationCommand.model_optimization_flavor.value
    NETWORK_GROUP = "network_group"
    CASCADE = "cascade"
    PLATFORM_PARAM = "platform_param"
    NORMALIZATION = "normalization"
    TRANSPOSE = "transpose"
    NMS_POSTPROCESS = "nms_postprocess"
    TRANSPOSE_CONCAT = "transpose_concat"
    INPUT_CONVERSION = "input_conversion"
    CHANGE_OUTPUT_ACTIVATION = "change_output_activation"
    PERFORMANCE_PARAM = "performance_param"
    CONTEXT_PERFORMANCE_PARAM = "performance_param"
    LOGITS_LAYER = "logits_layer"
    SET_SEED = "set_seed"
    RESIZE = "resize"
    REMOVE_NODE = "remove_node"
    CONVERT_TO_DENSE = "convert_to_dense"
    SHAPE_SPLITTER = "shape_splitter"
    SET_KV_CACHE_PAIR = "set_kv_cache_pair"
    SET_KV_CACHE_GLOBAL_PARAMS = "set_kv_cache_global_params"
    MIRROR = "mirror"
    BUCKET = "bucket"
    SHARE_CONFIG = "share_config"
    ADD_RESOURCE = "add_resource"


GLOB_COMMANDS = [
    SupportedCommands.QUANTIZATION_PARAM.value,
    SupportedCommands.COMPILATION_PARAM.value,
    SupportedCommands.CONTEXT_COMPILATION_PARAM.value,
    SupportedCommands.CONTEXT_RESOURCES_PARAM.value,
    SupportedCommands.PRE_QUANTIZATION_OPTIMIZATION.value,
    SupportedCommands.POST_QUANTIZATION_OPTIMIZATION.value,
    SupportedCommands.MODEL_OPTIMIZATION_CONFIG.value,
    SupportedCommands.PERFORMANCE_PARAM.value,
    SupportedCommands.CONTEXT_PERFORMANCE_PARAM.value,
]

VOID_COMMANDS = [
    SupportedCommands.PLACE.value,
    SupportedCommands.STRATEGY.value,
    SupportedCommands.COMPRESSION_PARAMS.value,
    SupportedCommands.ALLOCATOR_PARAM.value,
    SupportedCommands.BUFFERS.value,
    SupportedCommands.BUFFER_CALC_PARAM.value,
    SupportedCommands.PRINT_BUFFERS.value,
    SupportedCommands.FORCE_MAPPING.value,
    SupportedCommands.FORCE_ROUTE.value,
    SupportedCommands.OPTIMIZE_BUFFERS.value,
    SupportedCommands.CONTEXT_SWITCH_PARAM.value,
    SupportedCommands.HEF_PARAM.value,
    SupportedCommands.CONTEXT_COMPILATION_PARAM.value,
    SupportedCommands.LOGGER_PARAM.value,
    SupportedCommands.INTERNAL_ALLOCATOR_PARAM.value,
    SupportedCommands.COLLAPSE.value,
    SupportedCommands.INTERNAL_CONTEXT_SWITCH_PARAM.value,
    SupportedCommands.RESOURCES_PARAM.value,
    SupportedCommands.PLATFORM_PARAM.value,
    SupportedCommands.TRANSPOSE.value,
    SupportedCommands.NMS_POSTPROCESS.value,
    SupportedCommands.MODEL_OPTIMIZATION_FLAVOR.value,
    SupportedCommands.CHANGE_OUTPUT_ACTIVATION.value,
    SupportedCommands.CONTEXT_PERFORMANCE_PARAM.value,
    SupportedCommands.SET_SEED.value,
    SupportedCommands.REMOVE_NODE.value,
    SupportedCommands.CONVERT_TO_DENSE.value,
    SupportedCommands.SET_KV_CACHE_PAIR.value,
    SupportedCommands.SET_KV_CACHE_GLOBAL_PARAMS.value,
    SupportedCommands.MIRROR.value,
    SupportedCommands.SHARE_CONFIG.value,
    SupportedCommands.ADD_RESOURCE.value,
]

SINGLE_RETURN_COMMANDS = [
    SupportedCommands.SHORTCUT.value,
    SupportedCommands.PORTAL.value,
    SupportedCommands.CONCAT.value,
    SupportedCommands.OUTPUT_MUX.value,
    SupportedCommands.OUTPUT_LAYER.value,
    SupportedCommands.FEATURE_SPLITTER.value,
    SupportedCommands.FROM_TF.value,
    SupportedCommands.CONTEXT.value,
    SupportedCommands.MERGE.value,
    SupportedCommands.DDR.value,
    SupportedCommands.L4_PORTAL.value,
    SupportedCommands.NETWORK_GROUP.value,
    SupportedCommands.SHAPE_SPLITTER.value,
    SupportedCommands.BUCKET.value,
]

MULTIPLE_RETURN_COMMANDS = [
    SupportedCommands.DEFUSE.value,
    SupportedCommands.MUX_DEMUX.value,
    SupportedCommands.CASCADE.value,
    SupportedCommands.NORMALIZATION.value,
    SupportedCommands.TRANSPOSE_CONCAT.value,
    SupportedCommands.INPUT_CONVERSION.value,
    SupportedCommands.FORMAT_CONVERSION.value,
    SupportedCommands.LOGITS_LAYER.value,
    SupportedCommands.RESIZE.value,
]

DICT_RETURN_COMMANDS = [
    SupportedCommands.DEFUSE_BLOCK.value,
]


class CommandsGroups(IntEnum):
    MODEL_MODIFICATIONS = 0
    QUANTIZATION_FLAVOR = 1
    QUANTIZATION = 2
    GLOBAL = 3
    NETWORK = 4
    COMPILATION_PARAMS = 5
    BUFFERS = 6
    CONTEXT = 7
    PLACE = 8
    MAPPING = 9


def convert_params_to_str(params):
    result = {}
    for k, v in params.items():
        key = k.value if isinstance(k, Enum) else str(k)
        if isinstance(v, Enum):
            value = v.value
        elif isinstance(v, list) and len(v) == 0:
            continue
        elif isinstance(v, list):
            value = [elem.value if isinstance(elem, Enum) else str(elem) for elem in v]
        else:
            value = str(v)
        result[key] = value

    return result


def validate_params_command(params, supported_params, msg_prefix, exception=AllocatorScriptParserException):
    if params and any(x not in supported_params for x in params):
        not_found = [x for x in params if x not in supported_params]
        raise exception(msg_prefix + f"params keys {not_found} are not supported.")


class ModelScriptCommand:
    def __init__(self, function_name, function_args=None, function_return_vals=None, sort_key_func=None):
        self._function_name = function_name
        self._function_args = function_args
        self._function_return_vals = function_return_vals
        self._sort_key_func = sort_key_func

    @property
    def function_name(self):
        return self._function_name

    @property
    def function_args(self):
        return self._function_args

    @property
    def function_return_vals(self):
        if self._function_return_vals is None:
            return None
        if not isinstance(self._function_return_vals, list):
            return [self._function_return_vals]
        return self._function_return_vals

    @property
    def group(self):
        return CommandsGroups.GLOBAL

    def sort_key(self):
        return int(self.group), self.inner_cmd_sort_key()

    def inner_cmd_sort_key(self):
        return 0

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_OPERANDS

    @classmethod
    def validate_data_type(cls, command_pb):
        assert (
            command_pb.data.type == cls.data_type()
        ), f"wrong data type, expected {cls.data_type()} but got {command_pb.data.type}"

    @classmethod
    def _get_layers_glob_syntax(cls, args, layers_scope_from_hn, net_scopes):
        layers = []

        for pattern in args:
            if len(net_scopes) == 1:
                pattern = cls.add_scope_to_layer(net_scopes, pattern)
            layers.extend(fnmatch.filter(layers_scope_from_hn, pattern))
        return layers

    def expand_glob(self, layers_scope_from_hn, net_scopes):
        pass

    @property
    def msg_prefix(self):
        return f"Failed to parse {self._function_name} command - "

    def validate_command(self, layers_scope_from_hn):
        raise NotImplementedError

    def get_layers(self):
        raise NotImplementedError

    def has_unfound_layers(self, layers_scope_from_hn):
        return not set(self.get_layers()).issubset(set(layers_scope_from_hn))

    def handle_unfound_layers(self, layers_scope_from_hn):
        # no need to handle, just ignore
        return False

    def should_add_scope(self):
        scope_changer_groups = [SINGLE_RETURN_COMMANDS, MULTIPLE_RETURN_COMMANDS, DICT_RETURN_COMMANDS]
        return any(self.function_name.value in group for group in scope_changer_groups)

    def add_scope(self, scope_names, force=False):
        if self.should_add_scope():
            raise NotImplementedError

    @staticmethod
    def add_scope_to_layer(scope_names, layer_name, force=False):
        layer_parts = layer_name.split("/", 1)
        if len(layer_parts) == 1:
            if isinstance(scope_names, dict):
                raise AllocatorScriptParserException("Scope names cannot be dict when scope does not exist")
            return f"{scope_names[0]}/{layer_name}"
        scope_name = [scope_names[layer_parts[0]]] if isinstance(scope_names, dict) else scope_names
        if len(scope_name) == 1 and layer_parts[0] == scope_name[0]:
            return layer_name
        if force:
            return f"{scope_name[0]}/{layer_parts[-1]}"

        raise AllocatorScriptParserException(f"Invalid scope name {layer_parts[0]} exists")

    @staticmethod
    def remove_scope_from_layer(layer_name, force=False):
        layer_parts = layer_name.split("/")
        len_parts = len(layer_parts)
        if len_parts == 1:
            return layer_name
        if len_parts == 2 or force:
            return layer_parts[-1]

        raise AllocatorScriptParserException(f"Invalid scope name {layer_name} exists")

    def _remove_scope(self, val):
        return (
            [self.remove_scope_from_layer(v) for v in val]
            if isinstance(val, collections.abc.Iterable) and not isinstance(val, str)
            else self.remove_scope_from_layer(val)
        )

    def remove_scope(self):
        pass

    def __str__(self):
        raise NotImplementedError

    def str_to_alls(self):
        return str(self)

    def scopes_in_command(self):
        return None

    def prefix_in_command(self, prefix):
        return None

    def replace_prefix(self, src_prefix, dst_prefix):
        pass


class ModelScriptCommandWithScope(ModelScriptCommand, ABC):
    def add_scope(self, scope_names, force=False):
        if not scope_names:
            return
        if self.function_return_vals:
            self._function_return_vals = [
                self.add_scope_to_layer(scope_names, function_return_val, force=force)
                for function_return_val in self.function_return_vals
            ]

    @abstractmethod
    def remove_scope(self):
        pass

    @abstractmethod
    def _all_layers(self):
        pass

    @abstractmethod
    def _replace_all_layers(self, new_values):
        pass

    def scopes_in_command(self):
        return get_scopes_set_from_layers(self._all_layers())

    def prefix_in_command(self, prefix):
        contains_prefix = False
        if self.function_return_vals:
            contains_prefix = is_prefix_in_layers_list(prefix, self.function_return_vals)
        contains_prefix = contains_prefix or is_prefix_in_layers_list(prefix, self._all_layers())
        return contains_prefix

    def replace_prefix(self, src_prefix, dst_prefix):
        self._replace_all_layers(replace_prefix_in_layers_list(src_prefix, dst_prefix, self._all_layers()))

        # replace return values
        if self.function_return_vals:
            self._function_return_vals = replace_prefix_in_layers_list(
                src_prefix, dst_prefix, self.function_return_vals
            )

    def validate_single_scope_in_command(self, error_msg):
        real_scope_names = [name for name in self.scopes_in_command() if name != ""]
        if len(real_scope_names) > 1:
            raise AllocatorScriptParserException(self.msg_prefix + error_msg)


class DefuseType(Enum):
    NORMAL = 0
    SPATIAL = 1
    DECONV = 2
    COMPUTE_LANES = 3
    NMS = 4
    MAXPOOL = 5
    RESIZE = 6
    UNSUPPORTED_PADDING = 7
    INPUT_FEATURES = 8
    SUPER_CONV = 9
    ARCH_REQUIRED = 10
    DEPTH_TO_SPACE = 11
    SPACE_TO_DEPTH = 12
    RESIZE_TRANSPOSE = 13
    SUPER_DW = 14
    CONST_INPUT = 15
    GLOBAL_AVGPOOL_TRANSPOSED_INPUT = 16
    SPATIAL_RESHAPE = 17
    DOUBLE_PRECISION_CONV = 18
    NV = 19
    I420 = 20
    SUPER_DECONV = 21
    DYNAMIC_WEIGHTS = 22
    L3_PORTAL = 23
    EW_MULT_ON_MAC = 24
    CONV_WEIGHT_GROUPS = 25

    def to_pb(self):
        if self == DefuseType.NORMAL:
            return


def get_scopes_set_from_layers(layers):
    result = list({layer.split("/")[0] if len(layer.split("/")) > 1 else "" for layer in layers})
    result.sort()
    return result


def decode_prefix(prefix):
    if "/" in prefix:
        scope, reminder = prefix.split("/", 1)
    else:
        scope, reminder = prefix, ""

    return scope, reminder


def is_prefix_in_layers_list(prefix, layers):
    scope, reminder = decode_prefix(prefix)
    return all(layer.startswith(scope) and reminder in layer for layer in layers)


def is_prefix_in_layers_list_relaxed(prefix, layers):
    scope, reminder = decode_prefix(prefix)
    return any(layer.startswith(scope) and reminder in layer for layer in layers)


def replace_prefix_in_layers_list(src_prefix, dst_prefix, layers):
    src_scope, src_reminder = decode_prefix(src_prefix)
    dst_scope, dst_reminder = decode_prefix(dst_prefix)

    if src_scope != dst_scope:
        temp = [layer.replace(src_scope + "/", dst_scope + "/") for layer in layers]
    else:
        temp = layers

    if src_reminder != dst_reminder:
        temp = [layer.replace(src_reminder, dst_reminder) for layer in temp]

    return temp


class DefuseCommand(ModelScriptCommandWithScope):
    def __init__(
        self,
        layers_to_defuse,
        defused_layers,
        num_of_defuses,
        defuse_type=DefuseType.NORMAL,
        transpose=False,
        splitter=False,
        axis=None,
    ):
        super().__init__(SupportedCommands.DEFUSE, function_return_vals=defused_layers)
        self._layers_to_defuse = layers_to_defuse
        self._defused_layers = defused_layers
        self._num_of_defuses = num_of_defuses
        self._defuse_type = defuse_type
        self._transpose = transpose
        self._splitter = splitter
        self._axis = axis

    def _all_layers(self):
        return self.layers_to_defuse + self.defused_layers

    def _replace_all_layers(self, new_values):
        self._layers_to_defuse = new_values[: len(self.layers_to_defuse)]
        self._defused_layers = new_values[len(self.layers_to_defuse) :]

    def __str__(self):
        defused_layers = ", ".join(self._defused_layers)
        defuse_type_str = {
            DefuseType.SPATIAL: "defuse_type=SPATIAL",
            DefuseType.DECONV: "defuse_type=DECONV",
            DefuseType.SUPER_DECONV: "defuse_type=SUPER_DECONV",
            DefuseType.COMPUTE_LANES: "defuse_type=COMPUTE_LANES",
            DefuseType.NMS: "defuse_type=NMS",
            DefuseType.MAXPOOL: "defuse_type=MAXPOOL",
            DefuseType.RESIZE: "defuse_type=RESIZE",
            DefuseType.UNSUPPORTED_PADDING: "defuse_type=UNSUPPORTED_PADDING",
            DefuseType.INPUT_FEATURES: "defuse_type=INPUT_FEATURES",
            DefuseType.SUPER_CONV: "defuse_type=SUPER_CONV",
            DefuseType.ARCH_REQUIRED: "defuse_type=ARCH_REQUIRED",
            DefuseType.DEPTH_TO_SPACE: "defuse_type=DEPTH_TO_SPACE",
            DefuseType.SPACE_TO_DEPTH: "defuse_type=SPACE_TO_DEPTH",
            DefuseType.RESIZE_TRANSPOSE: "defuse_type=RESIZE_TRANSPOSE",
            DefuseType.SUPER_DW: "defuse_type=SUPER_DW",
            DefuseType.CONST_INPUT: "defuse_type=CONST_INPUT",
            DefuseType.GLOBAL_AVGPOOL_TRANSPOSED_INPUT: "defuse_type=GLOBAL_AVGPOOL_TRANSPOSED_INPUT",
            DefuseType.SPATIAL_RESHAPE: "defuse_type=SPATIAL_RESHAPE",
            DefuseType.DOUBLE_PRECISION_CONV: "defuse_type=DOUBLE_PRECISION_CONV",
            DefuseType.NV: "defuse_type=NV",
            DefuseType.I420: "defuse_type=I420",
            DefuseType.DYNAMIC_WEIGHTS: "defuse_type=DYNAMIC_WEIGHTS",
            DefuseType.L3_PORTAL: "defuse_type=L3_PORTAL",
            DefuseType.EW_MULT_ON_MAC: "defuse_type=EW_MULT_ON_MAC",
            DefuseType.CONV_WEIGHT_GROUPS: "defuse_type=CONV_WEIGHT_GROUPS",
        }

        if len(self._layers_to_defuse) > 1:
            layers_to_defuse_str = "[{}]".format(", ".join(self._layers_to_defuse))
        else:
            layers_to_defuse_str = self._layers_to_defuse[0]
        res = f"{defused_layers} = defuse({layers_to_defuse_str}, {self._num_of_defuses!s}"
        if self._defuse_type != DefuseType.NORMAL:
            res = f"{res}, {defuse_type_str[self._defuse_type]}"
        if self._axis is not None and self._defuse_type == DefuseType.COMPUTE_LANES:
            defuseAxis = PbWrapper().integrated_hw_graph_base_pb2.ProtoDefuseLanesAxis
            if self._axis == defuseAxis.PROTO_DEFUSE_WIDTH:
                res = f"{res}, axis=WIDTH"
            else:
                res = f"{res}, axis=FEATURES"
        res = "{}, {}".format(res, "transpose=True") if self._transpose else f"{res}"
        return "{}, {})".format(res, "splitter=True") if self._splitter else f"{res})"

    @classmethod
    def from_tokens(cls, tokens):
        num_of_defuses = int(tokens.function_args[1])
        layers_to_defuse = tokens.function_args[0]
        if not isinstance(layers_to_defuse, list):
            layers_to_defuse = [layers_to_defuse]

        defused_layers = tokens.multiple_return_vals.asList()

        defuse_type = DefuseType.NORMAL
        transpose = False
        splitter = False
        axis = None
        found_keys = []
        index = 2
        while index < len(tokens.function_args):
            if "defuse_type" in tokens.function_args[index] and "defuse_type" not in found_keys:
                defuse_type = DefuseType[tokens.function_args[index]["defuse_type"]]
                found_keys.append("defuse_type")
            elif "transpose" in tokens.function_args[index] and "transpose" not in found_keys:
                transpose = tokens.function_args[index]["transpose"] == "True"
                found_keys.append("transpose")
            elif "splitter" in tokens.function_args[index] and "splitter" not in found_keys:
                splitter = tokens.function_args[index]["splitter"] == "True"
                found_keys.append("splitter")
            elif "axis" in tokens.function_args[index] and "axis" not in found_keys:
                if tokens.function_args[index]["axis"] == "WIDTH":
                    axis = PbWrapper().integrated_hw_graph_base_pb2.ProtoDefuseLanesAxis.PROTO_DEFUSE_WIDTH
                else:  # features axis is default
                    axis = PbWrapper().integrated_hw_graph_base_pb2.ProtoDefuseLanesAxis.PROTO_DEFUSE_FEATURES
                found_keys.append("axis")
            else:
                raise AllocatorScriptParserException(
                    f"Failed to parse {tokens.function_name} command - unexpected kwarg in Defuse command",
                )
            index += 1

        if "defuse_type" not in found_keys and num_of_defuses == 1:  # TODO: should be decided by layer type (SDK-12778)
            defuse_type = DefuseType.DECONV

        return cls(
            layers_to_defuse,
            defused_layers,
            num_of_defuses,
            defuse_type=defuse_type,
            transpose=transpose,
            splitter=splitter,
            axis=axis,
        )

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_DEFUSE

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("defuse")
        layers_to_defuse = command_pb.data.defuse.layers_to_defuse
        defused_layers = list(command_pb.result)
        pb_wrapper = PbWrapper()
        type_switcher = {
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_NORMAL: DefuseType.NORMAL,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_SPATIAL: DefuseType.SPATIAL,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_DECONV: DefuseType.DECONV,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_SUPER_DECONV: DefuseType.SUPER_DECONV,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_COMPUTE_LANES: DefuseType.COMPUTE_LANES,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_NMS: DefuseType.NMS,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_MAXPOOL: DefuseType.MAXPOOL,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_RESIZE: DefuseType.RESIZE,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_UNSUPPORTED_PADDING: DefuseType.UNSUPPORTED_PADDING,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_INPUT_FEATURES: DefuseType.INPUT_FEATURES,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_SUPER_CONV: DefuseType.SUPER_CONV,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_ARCH_REQUIRED: DefuseType.ARCH_REQUIRED,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_DEPTH_TO_SPACE: DefuseType.DEPTH_TO_SPACE,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_SPACE_TO_DEPTH: DefuseType.SPACE_TO_DEPTH,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_RESIZE_TRANSPOSE: DefuseType.RESIZE_TRANSPOSE,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_SUPER_DW: DefuseType.SUPER_DW,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_CONST_INPUT: DefuseType.CONST_INPUT,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_GLOBAL_AVGPOOL_TRANSPOSED_INPUT: DefuseType.GLOBAL_AVGPOOL_TRANSPOSED_INPUT,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_SPATIAL_RESHAPE: DefuseType.SPATIAL_RESHAPE,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_DOUBLE_PRECISION_CONV: DefuseType.DOUBLE_PRECISION_CONV,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_NV: DefuseType.NV,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_I420: DefuseType.I420,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_DYNAMIC_WEIGHTS: DefuseType.DYNAMIC_WEIGHTS,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_L3_PORTAL: DefuseType.L3_PORTAL,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_EW_MULT_ON_MAC: DefuseType.EW_MULT_ON_MAC,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_CONV_WEIGHT_GROUPS: DefuseType.CONV_WEIGHT_GROUPS,
        }

        return cls(
            layers_to_defuse,
            defused_layers,
            command_pb.data.defuse.num_of_defuses,
            type_switcher[command_pb.data.defuse.script_defuse_type],
            command_pb.data.defuse.transpose,
            command_pb.data.defuse.splitter,
            command_pb.data.defuse.axis,
        )

    @property
    def layers_to_defuse(self):
        return self._layers_to_defuse

    @property
    def layers_to_remove(self):
        if self.defuse_type == DefuseType.UNSUPPORTED_PADDING:
            return None
        else:
            return self._layers_to_defuse

    @property
    def num_of_defuses(self):
        return self._num_of_defuses

    @property
    def defuse_type(self):
        return self._defuse_type

    @property
    def transpose(self):
        return self._transpose

    @property
    def splitter(self):
        return self._splitter

    @property
    def axis(self):
        return self._axis

    @property
    def defused_layers(self):
        return self._defused_layers

    @property
    def group(self):
        return CommandsGroups.NETWORK

    @property
    def num_of_defused_layers(self):
        if len(self.layers_to_defuse) > 1:
            return len(self.layers_to_defuse) * self.num_of_defuses
        if self.defuse_type == DefuseType.UNSUPPORTED_PADDING:
            return 1
        if self.defuse_type == DefuseType.COMPUTE_LANES:
            return 3
        if self.defuse_type in [DefuseType.SUPER_CONV, DefuseType.SUPER_DW]:
            return 3
        if self.defuse_type == DefuseType.GLOBAL_AVGPOOL_TRANSPOSED_INPUT:
            return 2
        if self.defuse_type == DefuseType.NV:
            return 5
        if self.defuse_type == DefuseType.I420:
            return 6
        # Each defuse splits the nms node and it's output.
        elif self.defuse_type == DefuseType.NMS:
            return self.num_of_defuses * 2
        elif self.defuse_type in [
            DefuseType.DECONV,
            DefuseType.SUPER_DECONV,
            DefuseType.RESIZE,
            DefuseType.MAXPOOL,
            DefuseType.ARCH_REQUIRED,
            DefuseType.RESIZE_TRANSPOSE,
            DefuseType.CONST_INPUT,
            DefuseType.L3_PORTAL,
            DefuseType.EW_MULT_ON_MAC,
            DefuseType.CONV_WEIGHT_GROUPS,
        ]:
            return len(self.defused_layers)
        elif self.defuse_type in [DefuseType.INPUT_FEATURES, DefuseType.DYNAMIC_WEIGHTS]:
            return self.num_of_defuses + 2
        elif self.defuse_type in [DefuseType.SPACE_TO_DEPTH, DefuseType.DEPTH_TO_SPACE, DefuseType.SPATIAL_RESHAPE]:
            return 2
        elif self.defuse_type == DefuseType.DOUBLE_PRECISION_CONV:
            return 4 if self.splitter else 3
        else:
            return self.num_of_defuses + 1

    def get_layers(self):
        return self.layers_to_defuse

    def validate_command(self, layers_scope_from_hn):
        if self.layers_to_defuse is None or self.has_unfound_layers(layers_scope_from_hn):
            not_found_layers = set(self.layers_to_defuse) - set(layers_scope_from_hn)
            not_found_layers_names = ",".join(not_found_layers)
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Cannot find layer {not_found_layers_names} in existing layers scope.",
            )
        if self.num_of_defused_layers + int(self.transpose) > len(self.defused_layers):
            if self.defuse_type in [DefuseType.DECONV, DefuseType.SUPER_DECONV]:
                pass  # Deconv defuse layers can be 2,3,4. SDK-21044
            raise AllocatorScriptParserException(
                self.msg_prefix + "num_of_defuses should be integer and match the number of returned defused layers.",
            )
        if self.axis is not None:
            defuseAxis = PbWrapper().integrated_hw_graph_base_pb2.ProtoDefuseLanesAxis
            if (
                self.axis not in [defuseAxis.PROTO_DEFUSE_WIDTH, defuseAxis.PROTO_DEFUSE_FEATURES]
                or self.defuse_type != DefuseType.COMPUTE_LANES
            ):
                raise AllocatorScriptParserException(
                    self.msg_prefix + "axis should be one of WIDTH or FEATURES and used only for COMPUTE_LANES defuse.",
                )
        if self.defuse_type not in DefuseType:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"defuse_type should be on of: {DefuseType.__members__}",
            )
        if self.defuse_type in [DefuseType.DECONV, DefuseType.SUPER_DECONV] and self.num_of_defuses > 4:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"num of defuses for deconv layer {self.layers_to_defuse}" "must be no more than 4.",
            )
        if self.defuse_type == DefuseType.NV and self.num_of_defuses != 5:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"num of defuses for nv layer {self.layers_to_defuse}" "must be 5.",
            )
        if self.defuse_type == DefuseType.I420 and self.num_of_defuses != 6:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"num of defuses for nv layer {self.layers_to_defuse}" "must be 6.",
            )
        if (
            self.splitter
            and self.defuse_type != DefuseType.DOUBLE_PRECISION_CONV
            and self.defuse_type != DefuseType.SPATIAL
        ):
            raise AllocatorScriptParserException(
                self.msg_prefix + "splitter supported only for SPATIAL and DOUBLE_PRECISION_CONV defuses",
            )
        if self.defused_layers is not None and any(x in layers_scope_from_hn for x in self.defused_layers):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"one or more from defused_layers {self.defused_layers} is already in "
                "layers scope",
            )
        real_scope_names = [name for name in self.scopes_in_command() if name != ""]
        self.validate_single_scope_in_command("too many scopes in the command: " + ", ".join(real_scope_names))
        default_logger().debug(
            f"Parsed Defuse command, args=({self.layers_to_defuse}, {self.num_of_defuses}), return_vals={self.defused_layers}",
        )

    def to_pb(self, pb_wrapper):
        defuse_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        defuse_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_DEFUSE
        defuse_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_DEFUSE
        defuse_cmd_msg.data.defuse.num_of_defuses = self.num_of_defuses
        defuse_cmd_msg.data.defuse.transpose = self.transpose
        defuse_cmd_msg.data.defuse.splitter = self.splitter
        if self.axis is not None:
            defuse_cmd_msg.data.defuse.axis = self.axis
        if self.defuse_type == DefuseType.NORMAL:
            defuse_cmd_msg.data.defuse.script_defuse_type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_NORMAL
        elif self.defuse_type == DefuseType.SPATIAL:
            defuse_cmd_msg.data.defuse.script_defuse_type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_SPATIAL
        elif self.defuse_type == DefuseType.COMPUTE_LANES:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_COMPUTE_LANES
            )
        elif self.defuse_type == DefuseType.NMS:
            defuse_cmd_msg.data.defuse.script_defuse_type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_NMS
        elif self.defuse_type == DefuseType.MAXPOOL:
            defuse_cmd_msg.data.defuse.script_defuse_type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_MAXPOOL
        elif self.defuse_type == DefuseType.RESIZE:
            defuse_cmd_msg.data.defuse.script_defuse_type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_RESIZE
        elif self.defuse_type == DefuseType.UNSUPPORTED_PADDING:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_UNSUPPORTED_PADDING
            )
        elif self.defuse_type == DefuseType.DECONV:
            defuse_cmd_msg.data.defuse.script_defuse_type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_DECONV
        elif self.defuse_type == DefuseType.SUPER_DECONV:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_SUPER_DECONV
            )
        elif self.defuse_type == DefuseType.INPUT_FEATURES:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_INPUT_FEATURES
            )
        elif self.defuse_type == DefuseType.SUPER_CONV:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_SUPER_CONV
            )
        elif self.defuse_type == DefuseType.ARCH_REQUIRED:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_ARCH_REQUIRED
            )
        elif self.defuse_type == DefuseType.DEPTH_TO_SPACE:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_DEPTH_TO_SPACE
            )
        elif self.defuse_type == DefuseType.SPACE_TO_DEPTH:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_SPACE_TO_DEPTH
            )
        elif self.defuse_type == DefuseType.RESIZE_TRANSPOSE:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_RESIZE_TRANSPOSE
            )
        elif self.defuse_type == DefuseType.SUPER_DW:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_SUPER_DW
            )
        elif self.defuse_type == DefuseType.CONST_INPUT:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_CONST_INPUT
            )
        elif self.defuse_type == DefuseType.GLOBAL_AVGPOOL_TRANSPOSED_INPUT:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_GLOBAL_AVGPOOL_TRANSPOSED_INPUT
            )
        elif self.defuse_type == DefuseType.NV:
            defuse_cmd_msg.data.defuse.script_defuse_type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_NV
        elif self.defuse_type == DefuseType.I420:
            defuse_cmd_msg.data.defuse.script_defuse_type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_I420
        elif self.defuse_type == DefuseType.SPATIAL_RESHAPE:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_SPATIAL_RESHAPE
            )
        elif self.defuse_type == DefuseType.DOUBLE_PRECISION_CONV:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_DOUBLE_PRECISION_CONV
            )
        elif self.defuse_type == DefuseType.DYNAMIC_WEIGHTS:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_DYNAMIC_WEIGHTS
            )
        elif self.defuse_type == DefuseType.L3_PORTAL:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_L3_PORTAL
            )
        elif self.defuse_type == DefuseType.EW_MULT_ON_MAC:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_EW_MULT_ON_MAC
            )

        elif self.defuse_type == DefuseType.CONV_WEIGHT_GROUPS:
            defuse_cmd_msg.data.defuse.script_defuse_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DEFUSE_CONV_WEIGHT_GROUPS
            )
        else:
            assert ()

        for layer in self.layers_to_defuse:
            defuse_cmd_msg.data.defuse.layers_to_defuse.append(layer)
        defuse_cmd_msg.result.extend(self.defused_layers)
        return defuse_cmd_msg

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        layer_to_defuse = []
        for layer in self._layers_to_defuse:
            layer_to_defuse.append(self.add_scope_to_layer(scope_names, layer, force=force))
        self._layers_to_defuse = layer_to_defuse
        for i, layer in enumerate(self._defused_layers):
            self._defused_layers[i] = self.add_scope_to_layer(scope_names, layer, force=force)

    def remove_scope(self):
        self._defused_layers = self._remove_scope(self._defused_layers)
        self._layers_to_defuse = self._remove_scope(self._layers_to_defuse)


class DefuseBlockCommand(ModelScriptCommandWithScope):
    def __init__(self, defuse_dict, number_of_defuses):
        super().__init__(SupportedCommands.DEFUSE_BLOCK)
        self._defuse_dict = defuse_dict
        self._number_of_defuses = number_of_defuses

    def _all_layers(self):
        layers = list(self._defuse_dict.keys())
        for value in self._defuse_dict.values():
            layers.extend(value)
        return layers

    def _replace_all_layers(self, new_values):
        if len(self._defuse_dict) != 1:
            raise NotImplementedError(
                "DefuseBlockCommand for more than 1 layer with mirror command is not supported - SDK-54919"
            )
        new_orig_layer_name = new_values[0]
        new_defused_layers = tuple(new_values[1:])
        new_defsue_dict = {new_orig_layer_name: new_defused_layers}
        assert len(new_defsue_dict) == len(self._defuse_dict)
        for (key, value), (new_key, new_value) in zip(self._defuse_dict.items(), new_defsue_dict.items()):
            assert len(value) == len(new_value)
        self._defuse_dict = new_defsue_dict

    def __str__(self):
        string = "{ "

        dict_content = []
        for key, layers in self._defuse_dict.items():
            if layers:
                layers_names = ", ".join(layers)
                dict_content.append(f"{key}: [{layers_names}]")

        string += ", ".join(dict_content)
        string += " } = " + self._function_name.value + "("
        string += ", ".join(self._defuse_dict.keys())
        string += ", defuse_count=" + str(self._number_of_defuses) + ")"

        return string

    @classmethod
    def from_tokens(cls, tokens):
        num_of_defuses = None

        defuse_dict = dict({(a, tuple(b)) for a, b in list(tokens.dict_return_vals)})

        for param in tokens.function_args:
            if isinstance(param, str):
                if param not in defuse_dict:
                    defuse_dict[param] = tuple()
            elif isinstance(param, dict):
                if "defuse_count" in param:
                    num_of_defuses = int(param["defuse_count"])
                elif "defuse_type" in param:
                    pass  # TODO
                else:
                    raise AllocatorScriptParserException("Invalid defuse command param")
            else:
                raise AllocatorScriptParserException("Invalid defuse command")

        return cls(defuse_dict, num_of_defuses)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_DEFUSE_BLOCK

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("defuse_block")

        layers_to_defuses_dict = {}
        for layer in command_pb.data.defuse_block.layers:
            layers_to_defuses_dict[layer.layer] = layer.defused_layers

        number_of_defuses = command_pb.data.defuse_block.num_of_defuses

        return cls(layers_to_defuses_dict, number_of_defuses)

    @property
    def layers_to_defuse(self):
        return self._defuse_dict.keys()

    @property
    def layers_to_remove(self):
        return list(self._defuse_dict.keys())

    @property
    def num_of_defuses(self):
        return self._num_of_defuses

    @property
    def defuse_type(self):
        return self._defuse_type

    @property
    def defused_layers(self):
        layers = []
        for _, value in self._defuse_dict.items():
            layers.extend(value)
        return layers

    @property
    def group(self):
        return CommandsGroups.NETWORK

    @property
    def num_of_defused_layers(self):
        # Not accurate, can't tell how many concats and feature splitter needed
        return len(self.layers_to_defuse) * self.num_of_defuses

    def get_layers(self):
        return self.layers_to_defuse

    def validate_command(self, layers_scope_from_hn):
        if self.layers_to_defuse is None or self.has_unfound_layers(layers_scope_from_hn):
            not_found_layers = set(self.layers_to_defuse) - set(layers_scope_from_hn)
            not_found_layers_names = ",".join(not_found_layers)
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Cannot find layer {not_found_layers_names} in existing layers scope.",
            )
        if self.defused_layers is not None and any(x in layers_scope_from_hn for x in self.defused_layers):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"one or more from defused_layers {self.defused_layers} is already in "
                "layers scope",
            )
        real_scope_names = [name for name in self.scopes_in_command() if name != ""]
        self.validate_single_scope_in_command("too many scopes in the command: " + ", ".join(real_scope_names))

    def to_pb(self, pb_wrapper):
        defuse_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        defuse_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_DEFUSE_BLOCK
        defuse_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_DEFUSE_BLOCK
        defuse_cmd_msg.data.defuse_block.num_of_defuses = self._number_of_defuses

        for layer, layers in self._defuse_dict.items():
            item = defuse_cmd_msg.data.defuse_block.layers.add()
            item.layer = layer
            for defused_layer in layers:
                item.defused_layers.append(defused_layer)

        return defuse_cmd_msg

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)

        new_dict = {}
        for key, layers in self._defuse_dict.items():
            new_layers = []
            for layer in layers:
                new_layers.append(self.add_scope_to_layer(scope_names, layer, force=force))
            new_dict[self.add_scope_to_layer(scope_names, key, force=force)] = tuple(new_layers)

        self._defuse_dict = new_dict

    def remove_scope(self):
        layer_dict = {}
        for key, layers in self._defuse_dict.items():
            new_layers = []
            for layer in layers:
                new_layers.append(self._remove_scope(layer))
            layer_dict[self._remove_scope(key)] = tuple(new_layers)

        self._defuse_dict = layer_dict


class CollapseCommand(ModelScriptCommandWithScope):
    def __init__(self, layers_to_collapse):
        super().__init__(SupportedCommands.COLLAPSE)
        self._layers_to_collapse = layers_to_collapse

    def _all_layers(self):
        return self._layers_to_collapse

    def _replace_all_layers(self, new_values):
        self._layers_to_collapse = new_values

    def __str__(self):
        layer_to_collapse = self._layers_to_collapse
        return f"collapse({layer_to_collapse[0]}, {layer_to_collapse[1]})"

    @classmethod
    def from_tokens(cls, tokens):
        return cls(
            tokens.function_args,
        )

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_COLLAPSE

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        return cls(command_pb.data.collapse.layers_to_collapse)

    @property
    def group(self):
        return CommandsGroups.NETWORK

    def get_layers(self):
        return self._layers_to_collapse

    def validate_command(self, layers_scope_from_hn):
        if self._layers_to_collapse is None or self.has_unfound_layers(layers_scope_from_hn):
            not_found = [x for x in self._layers_to_collapse if x not in layers_scope_from_hn]
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Cannot find layer {not_found} in existing layers scope.",
            )
        self.validate_single_scope_in_command("Cannot collapse layers from different scopes")

    def to_pb(self, pb_wrapper):
        collapse_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        collapse_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_COLLAPSE
        collapse_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_COLLAPSE
        collapse_cmd_msg.data.collapse.layers_to_collapse.extend(self._layers_to_collapse)
        return collapse_cmd_msg

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        for i, layer in enumerate(self._layers_to_collapse):
            self._layers_to_collapse[i] = self.add_scope_to_layer(scope_names, layer, force=force)

    def remove_scope(self):
        self._layers_to_collapse = [self._remove_scope(val) for val in self._layers_to_collapse]


class PlaceCommand(ModelScriptCommandWithScope):
    def __init__(self, cluster_index, layers_to_place):
        super().__init__(SupportedCommands.PLACE)
        self._cluster_index = cluster_index
        self._layers_to_place = layers_to_place

    def __str__(self):
        layers_to_place = ", ".join(sorted(self._layers_to_place))
        return f"place({self._cluster_index}, [{layers_to_place}])"

    @classmethod
    def from_tokens(cls, tokens):
        return cls(int(tokens.function_args[0]), tokens.function_args[1])

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_PLACE

    @staticmethod
    def get_place_from_pb(command_pb):
        assert command_pb.data.HasField("place")
        cluster_index = command_pb.data.place.cluster_index
        layers_to_place = list(command_pb.data.place.layers_to_place)
        return cluster_index, layers_to_place

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        cluster_index, layers_to_place = cls.get_place_from_pb(command_pb)
        return cls(cluster_index, layers_to_place)

    @property
    def cluster_index(self):
        return self._cluster_index

    @property
    def layers_to_place(self):
        return self._layers_to_place

    @property
    def group(self):
        return CommandsGroups.PLACE

    def get_layers(self):
        return self.layers_to_place

    def validate_command(self, layers_scope_from_hn):
        if self.cluster_index < 0:
            raise AllocatorScriptParserException(self.msg_prefix + "cluster index should be a non negative integer.")
        if self.layers_to_place is None or self.has_unfound_layers(layers_scope_from_hn):
            not_found = [x for x in self.layers_to_place if x not in layers_scope_from_hn]
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Cannot find layer {not_found} in existing layers scope.",
            )
        default_logger().debug(f"Parsed Place command, args=({self.cluster_index}, {self.layers_to_place})")

    def handle_unfound_layers(self, layers_scope_from_hn):
        non_existing_layers = [x for x in self.layers_to_place if x not in layers_scope_from_hn]
        for layer in non_existing_layers:
            self.layers_to_place.remove(layer)
        return bool(self.layers_to_place)

    def to_pb(self, pb_wrapper):
        place_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        place_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_PLACE
        place_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_PLACE
        place_cmd_msg.data.place.layers_to_place.extend(self.layers_to_place)
        place_cmd_msg.data.place.cluster_index = self.cluster_index
        return place_cmd_msg

    def add_scope(self, scope_names, force=False):
        if not scope_names:
            return
        super().add_scope(scope_names, force=force)
        for i, layer in enumerate(self._layers_to_place):
            self._layers_to_place[i] = self.add_scope_to_layer(scope_names, layer, force=force)

    def remove_scope(self):
        self._layers_to_place = self._remove_scope(self._layers_to_place)

    def _replace_all_layers(self, new_values):
        for layer in self._layers_to_place:
            if layer in new_values:
                new_values.remove(layer)
        self._layers_to_place = new_values

    def prefix_in_command(self, prefix):
        # NOTE: account for improperly named layers
        return is_prefix_in_layers_list_relaxed(prefix, self._all_layers())

    def _all_layers(self):
        return self._layers_to_place


class ContextPlaceCommand(PlaceCommand):
    def __init__(self, context_name, cluster_index, layers_to_place):
        super().__init__(cluster_index, layers_to_place)
        self._context_name = context_name

    def __str__(self):
        place_str = super().__str__()
        return f"{self.context_name}.{place_str}"

    @classmethod
    def from_tokens(cls, tokens):
        return cls(tokens.object, int(tokens.function_args[0]), tokens.function_args[1])

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CONTEXT_PLACE

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        cluster_index, layers_to_place = super().get_place_from_pb(command_pb)
        context_name = command_pb.data.place.context_name
        return cls(context_name, cluster_index, layers_to_place)

    @property
    def context_name(self):
        return self._context_name

    def set_context_name(self, name):
        self._context_name = name

    def validate_command(self, layers_scope_from_hn):
        # NOTE: valid context is validated in cpp
        super().validate_command(layers_scope_from_hn)

    def to_pb(self, pb_wrapper):
        place_cmd_msg = super().to_pb(pb_wrapper)
        place_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CONTEXT_PLACE
        place_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_CONTEXT_PLACE
        place_cmd_msg.data.place.context_name = self.context_name
        return place_cmd_msg


class ShortcutCommand(ModelScriptCommandWithScope):
    def __init__(self, shortcut_layer, source_layer, dest_layers, function_name):
        super().__init__(function_name, function_return_vals=shortcut_layer)
        self._shortcut_layer = shortcut_layer
        self._source_layer = source_layer
        self._dest_layers = dest_layers if isinstance(dest_layers, list) else [dest_layers]
        self._is_portal = self.function_name == SupportedCommands.PORTAL
        self._is_ddr = self.function_name == SupportedCommands.DDR
        self._is_l4_portal = self.function_name == SupportedCommands.L4_PORTAL

    def __str__(self):
        dst_layers = ", ".join(self._dest_layers)
        dst_layers = f"[{dst_layers}]" if len(self._dest_layers) > 1 else dst_layers
        if self._is_l4_portal:
            command = "l4_portal"
        elif self._is_portal:
            command = "portal"
        elif self._is_ddr:
            command = "ddr"
        else:
            command = "shortcut"
        return f"{self._shortcut_layer} = {command}({self._source_layer}, {dst_layers})"

    def remove_scope(self):
        self._shortcut_layer = self._remove_scope(self._shortcut_layer)
        self._source_layer = self._remove_scope(self._source_layer)
        self._dest_layers = self._remove_scope(self._dest_layers)

    @staticmethod
    def function_name_to_op(function_name):
        if function_name == SupportedCommands.SHORTCUT.value:
            return SupportedCommands.SHORTCUT
        elif function_name == SupportedCommands.PORTAL.value:
            return SupportedCommands.PORTAL
        elif function_name == SupportedCommands.DDR.value:
            return SupportedCommands.DDR
        elif function_name == SupportedCommands.L4_PORTAL.value:
            return SupportedCommands.L4_PORTAL
        else:
            raise AllocatorScriptParserException("FATAL- non existing shortcut command type")

    @classmethod
    def from_tokens(cls, tokens):
        dest_layers = (
            tokens.function_args[1].asList() if hasattr(tokens.function_args[1], "asList") else tokens.function_args[1]
        )
        return cls(
            tokens.single_return_val,
            tokens.function_args[0],
            dest_layers,
            cls.function_name_to_op(tokens.function_name),
        )

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("operands")
        pb_wrapper = PbWrapper()
        if command_pb.op == pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_PORTAL:
            op = SupportedCommands.PORTAL
        elif command_pb.op == pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_DDR:
            op = SupportedCommands.DDR
        elif command_pb.op == pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_L4_PORTAL:
            op = SupportedCommands.L4_PORTAL
        else:
            op = SupportedCommands.SHORTCUT

        layer = command_pb.result[0]
        source_layer = command_pb.data.operands.operand0[0]
        dest_layers = list(command_pb.data.operands.operand1)

        return cls(layer, source_layer, dest_layers, op)

    @property
    def shortcut_layer(self):
        return self._shortcut_layer

    @property
    def source_layer(self):
        return self._source_layer

    @property
    def dest_layers(self):
        return self._dest_layers

    @property
    def group(self):
        return CommandsGroups.NETWORK

    def _all_layers(self):
        return [*self.dest_layers, self.source_layer, self.shortcut_layer]

    def _replace_all_layers(self, new_values):
        count = len(self.dest_layers)
        self._dest_layers = new_values[:count]
        self._source_layer = new_values[count]
        self._shortcut_layer = new_values[count + 1]

    def get_layers(self):
        return [self.source_layer, *self.dest_layers]

    def get_unfound_layers(self, layers_scope_from_hn):
        return [layer for layer in self.get_layers() if layer not in layers_scope_from_hn]

    def validate_command(self, layers_scope_from_hn):
        if self.shortcut_layer is None or self.shortcut_layer in layers_scope_from_hn:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Layer {self.shortcut_layer} is already in layers scope.",
            )
        if self.source_layer is None or self.has_unfound_layers(layers_scope_from_hn):
            raise AllocatorScriptParserException(
                self.msg_prefix
                + f"Cannot find layer {self.get_unfound_layers(layers_scope_from_hn)} in existing layers scope.",
            )
        if self.dest_layers is None or len(self.dest_layers) < 1:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"dest layers {self.dest_layers} must be a list of more than 0 layers.",
            )

        default_logger().debug(
            f"Parsed {self.function_name} command, args=({self.source_layer}, {self.dest_layers}), return_vals={self.shortcut_layer}",
        )

    def handle_unfound_layers(self, layers_scope_from_hn):
        if self.source_layer is None or self.source_layer not in layers_scope_from_hn:
            return False
        non_existing_layers = [x for x in self.dest_layers if x not in layers_scope_from_hn]
        for layer in non_existing_layers:
            self.dest_layers.remove(layer)
        return bool(self.dest_layers)

    def to_pb(self, pb_wrapper):
        shortcut_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        if self._is_portal:
            shortcut_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_PORTAL
        elif self._is_ddr:
            shortcut_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_DDR
        elif self._is_l4_portal:
            shortcut_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_L4_PORTAL
        else:
            shortcut_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_SHORTCUT

        shortcut_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_OPERANDS
        shortcut_cmd_msg.data.operands.operand0.append(self.source_layer)
        shortcut_cmd_msg.data.operands.operand1.extend(self.dest_layers)
        shortcut_cmd_msg.result.extend([self.shortcut_layer])
        return shortcut_cmd_msg

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        self._shortcut_layer = self.add_scope_to_layer(scope_names, self._shortcut_layer, force=force)
        self._source_layer = self.add_scope_to_layer(scope_names, self._source_layer, force=force)
        for i, layer in enumerate(self._dest_layers):
            self._dest_layers[i] = self.add_scope_to_layer(scope_names, layer, force=force)


class ConcatCommand(ModelScriptCommandWithScope):
    def __init__(self, concat_layer, layers_to_concat, dest_layer):
        super().__init__(SupportedCommands.CONCAT, function_return_vals=concat_layer)
        self._concat_layer = concat_layer
        self._layers_to_concat = layers_to_concat if isinstance(layers_to_concat, list) else [layers_to_concat]
        self._dest_layer = dest_layer

    def _all_layers(self):
        return [*self.layers_to_concat, self.concat_layer, self.dest_layer]

    def _replace_all_layers(self, new_values):
        count = len(self.layers_to_concat)
        self._layers_to_concat = new_values[:count]
        self._concat_layer = new_values[count]
        self._dest_layer = new_values[count + 1]

    def __str__(self):
        layers_to_concat = ", ".join(self._layers_to_concat)
        return f"{self._concat_layer} = concat([{layers_to_concat}], {self._dest_layer})"

    def remove_scope(self):
        self._concat_layer = self._remove_scope(self._concat_layer)
        self._layers_to_concat = self._remove_scope(self._layers_to_concat)
        self._dest_layer = self._remove_scope(self._dest_layer)

    @classmethod
    def from_tokens(cls, tokens):
        layers_to_concat = (
            tokens.function_args[0].asList() if hasattr(tokens.function_args[0], "asList") else tokens.function_args[0]
        )
        return cls(tokens.single_return_val, layers_to_concat, tokens.function_args[1])

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("operands")
        concat_layer = command_pb.result[0]
        layers_to_concat = list(command_pb.data.operands.operand0)
        dest_layer = command_pb.data.operands.operand1[0]
        return cls(concat_layer, layers_to_concat, dest_layer)

    @property
    def concat_layer(self):
        return self._concat_layer

    @property
    def layers_to_concat(self):
        return self._layers_to_concat

    @property
    def dest_layer(self):
        return self._dest_layer

    @property
    def group(self):
        return CommandsGroups.NETWORK

    def get_layers(self):
        return [self.dest_layer, *self.layers_to_concat]

    def validate_command(self, layers_scope_from_hn):
        if self.concat_layer is None or self.concat_layer in layers_scope_from_hn:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Layer {self.concat_layer} is already in layers scope.",
            )
        if self.dest_layer is None or self.dest_layer not in layers_scope_from_hn:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Cannot find layer {self.dest_layer} in existing layers scope",
            )
        if (
            self.layers_to_concat is None
            or len(self.layers_to_concat) < 1
            or self.has_unfound_layers(layers_scope_from_hn)
        ):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"layers to concat {self.layers_to_concat} must be a list of more "
                "than 0 and all layers should be in the scope",
            )
        self.validate_single_scope_in_command("Cannot add a concat layer between scopes")
        default_logger().debug(
            f"Parsed Concat command, args=({self.layers_to_concat}, {self.dest_layer}), return_vals={self.concat_layer}",
        )

    def to_pb(self, pb_wrapper):
        concat_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        concat_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_CONCAT
        concat_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_OPERANDS
        concat_cmd_msg.data.operands.operand0.extend(self.layers_to_concat)
        concat_cmd_msg.data.operands.operand1.append(self.dest_layer)
        concat_cmd_msg.result.append(self.concat_layer)
        return concat_cmd_msg

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        self._concat_layer = self.add_scope_to_layer(scope_names, self._concat_layer, force=force)
        self._dest_layer = self.add_scope_to_layer(scope_names, self._dest_layer, force=force)
        for i, layer in enumerate(self._layers_to_concat):
            self._layers_to_concat[i] = self.add_scope_to_layer(scope_names, layer, force=force)


class TransposeConcatCommand(ModelScriptCommandWithScope):
    def __init__(self, concat_layer, format_conversions):
        super().__init__(
            SupportedCommands.TRANSPOSE_CONCAT,
            function_return_vals=format_conversions,
        )
        self._concat_layer = concat_layer
        self._format_conversions = format_conversions

    def _all_layers(self):
        return [self.concat_layer, *self.format_conversions]

    def _replace_all_layers(self, new_values):
        self._concat_layer = new_values[0]
        self._format_conversions = new_values[1:]

    def __str__(self):
        format_conversions = ", ".join(self.format_conversions)
        return f"{format_conversions} = transpose_concat({self.concat_layer})"

    def remove_scope(self):
        self._format_conversions = self._remove_scope(self._format_conversions)
        self._concat_layer = self._remove_scope(self._concat_layer)

    @classmethod
    def from_tokens(cls, tokens):
        format_conversions = tokens.multiple_return_vals.asList()
        return cls(tokens.function_args[0], format_conversions)

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        concat_layer = command_pb.data.transpose_concat.concat_layer
        format_conversions = list(command_pb.result)
        return cls(concat_layer, format_conversions)

    @property
    def concat_layer(self):
        return self._concat_layer

    @property
    def format_conversions(self):
        return self._format_conversions

    @property
    def group(self):
        return CommandsGroups.NETWORK

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_TRANSPOSE_CONCAT

    def get_layers(self):
        return [self.concat_layer]

    def validate_command(self, layers_scope_from_hn):
        if self.concat_layer is None or self.concat_layer not in layers_scope_from_hn:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Layer {self.concat_layer} not exists in layers scope.",
            )

        if self.format_conversions is not None and any(x in layers_scope_from_hn for x in self.format_conversions):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"one or more from format conversions layers "
                f"{self.format_conversions} is already in layers "
                f"scope",
            )

        self.validate_single_scope_in_command("Cannot add a concat layer between scopes")
        default_logger().debug(f"Parsed Concat command, args=({self.concat_layer})")

    def to_pb(self, pb_wrapper):
        transpose_concat_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        transpose_concat_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_TRANSPOSE_CONCAT
        transpose_concat_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_TRANSPOSE_CONCAT
        transpose_concat_cmd_msg.data.transpose_concat.concat_layer = self.concat_layer
        transpose_concat_cmd_msg.result.extend(self.format_conversions)
        return transpose_concat_cmd_msg

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        self._concat_layer = self.add_scope_to_layer(scope_names, self._concat_layer, force=force)
        for i, layer in enumerate(self.format_conversions):
            self._format_conversions[i] = self.add_scope_to_layer(scope_names, layer, force=force)


class OutputMuxCommand(ModelScriptCommandWithScope):
    def __init__(self, output_mux_layer, output_layers_to_mux):
        super().__init__(SupportedCommands.OUTPUT_MUX, function_return_vals=output_mux_layer)
        self._output_mux_layer = output_mux_layer
        self._output_layers_to_mux = output_layers_to_mux

    def _all_layers(self):
        return [*self.output_layers_to_mux, self.output_mux_layer]

    def _replace_all_layers(self, new_values):
        count = len(self.output_layers_to_mux)
        self._output_layers_to_mux = new_values[:count]
        self._output_mux_layer = new_values[count]

    def __str__(self):
        output_layers_to_mux = ", ".join(self._output_layers_to_mux)
        return f"{self._output_mux_layer} = output_mux([{output_layers_to_mux}])"

    def remove_scope(self):
        self._output_mux_layer = self._remove_scope(self._output_mux_layer)
        self._output_layers_to_mux = self._remove_scope(self._output_layers_to_mux)

    @classmethod
    def from_tokens(cls, tokens):
        return cls(tokens.single_return_val, tokens.function_args[0])

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("operands")
        output_mux_layer = command_pb.result[0]
        output_layers_to_mux = list(command_pb.data.operands.operand0)
        return cls(output_mux_layer, output_layers_to_mux)

    @property
    def output_mux_layer(self):
        return self._output_mux_layer

    @property
    def output_layers_to_mux(self):
        return self._output_layers_to_mux

    @property
    def group(self):
        return CommandsGroups.NETWORK

    def get_layers(self):
        return self.output_layers_to_mux

    def validate_command(self, layers_scope_from_hn):
        if self.output_mux_layer is None or self.output_mux_layer in layers_scope_from_hn:
            raise AllocatorScriptParserException(
                self.msg_prefix
                + f"Invalid name for layer {self.output_mux_layer}- either already exists in scope or empty.",
            )
        if self.has_unfound_layers(layers_scope_from_hn):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Cannot find one of these layers {self.get_layers()} in existing layers scope.",
            )
        if self.output_layers_to_mux is None or len(self.output_layers_to_mux) < 1:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"output layers to mux {self.output_layers_to_mux} must be a list of more "
                "than 0 and all layers should be in the scope",
            )
        default_logger().debug(
            f"Parsed OutputMux command, args=({self.output_layers_to_mux}), return_vals={self.output_mux_layer}",
        )

    def to_pb(self, pb_wrapper):
        output_mux_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        output_mux_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_OUTPUT_MUX
        output_mux_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_OPERANDS
        output_mux_cmd_msg.data.operands.operand0.extend(self.output_layers_to_mux)
        output_mux_cmd_msg.result.append(self.output_mux_layer)
        return output_mux_cmd_msg

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        self._output_mux_layer = self.add_scope_to_layer(scope_names, self._output_mux_layer, force=force)
        for i, layer in enumerate(self._output_layers_to_mux):
            self._output_layers_to_mux[i] = self.add_scope_to_layer(scope_names, layer, force=force)


class CompilationParamCommand(ModelScriptCommandWithScope):
    INTERNAL_PARAMS = []
    OPTIONAL_PARAMS = ["enable_exhaustive_merge"]

    def __init__(self, input_layers, function_args=None, sort_key_func=None, **compilation_params):
        super().__init__(
            SupportedCommands.COMPILATION_PARAM,
            function_args=function_args,
            sort_key_func=sort_key_func,
        )
        self._input_layers = input_layers if isinstance(input_layers, list) else [input_layers]
        self._compilation_params = CompilationParams()
        self._compilation_params.set(compilation_params)

    def _all_layers(self):
        return self.input_layers

    def _replace_all_layers(self, new_values):
        self._input_layers = new_values

    def __str__(self):
        return self._export_params_to_string(False)

    def str_to_alls(self):
        return self._export_params_to_string(True)

    def _export_params_to_string(self, is_to_auto_alls):
        compilation_params = ", ".join(
            [
                param_str(param_k, param_v)
                for param_k, param_v in sorted(self._compilation_params.get().items())
                if not (
                    is_to_auto_alls
                    and (param_k in self.INTERNAL_PARAMS)
                    or (param_k in self.OPTIONAL_PARAMS and param_v is False)
                )
            ],
        )
        layers = ",".join(self._input_layers)
        if len(self._input_layers) > 1:
            layers = f"[{layers}]"
        elif "*" in layers:
            layers = f"{{{layers}}}"
        return (
            f"compilation_param({layers}, {compilation_params})"
            if compilation_params
            else f"compilation_param({layers})"
        )

    def remove_scope(self):
        self._input_layers = self._remove_scope(self._input_layers)

    def expand_glob(self, layers_scope_from_hn, net_scopes):
        self._input_layers = self._get_layers_glob_syntax(self._input_layers, layers_scope_from_hn, net_scopes)

    @classmethod
    def from_tokens(cls, tokens):
        args = tokens.function_args.asList()

        layers = args[0]
        compilation_params = {}
        for param in args[1:]:
            if "number_of_subclusters" in param:
                param["number_of_subclusters"] = int(param["number_of_subclusters"])
            compilation_params.update(param)

        return cls(layers, function_args=args, **compilation_params)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_COMPILATION_PARAMS

    @classmethod
    def from_pb(cls, command_pb, sort_key_func=None):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("compilation_params")
        input_layers = list(command_pb.data.compilation_params.layers)
        params = CompilationParams()
        params.from_pb(command_pb.data.compilation_params, PbWrapper())
        compilation_params = convert_params_to_str(params.get())
        return cls(input_layers, sort_key_func=sort_key_func, **compilation_params)

    @property
    def input_layers(self):
        return self._input_layers

    @property
    def compilation_params(self):
        return self._compilation_params.get()

    @property
    def group(self):
        return CommandsGroups.COMPILATION_PARAMS

    def inner_cmd_sort_key(self):
        first_input_layer = sorted(self.input_layers)[0]
        if self._sort_key_func:
            return self._sort_key_func(first_input_layer)
        return 0

    def get_layers(self):
        return self._input_layers

    def validate_command(self, layers_scope_from_hn):
        if len(self.input_layers) == 0 or self.has_unfound_layers(layers_scope_from_hn):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"layers {self.function_args[0]} could not be found in scope.",
            )
        validate_params_command(self.compilation_params, CompilationParams.DEFAULT_PARAMS.keys(), self.msg_prefix)
        self._validate_params_are_not_internal()
        default_logger().debug(
            f"Parsed CompilationParam command, args=({self.input_layers}, {self.compilation_params})",
        )

    def _validate_params_are_not_internal(self):
        internal_params = set(self.compilation_params.keys()).intersection(self.INTERNAL_PARAMS + self.OPTIONAL_PARAMS)
        if internal_params:
            # TODO: https://hailotech.atlassian.net/browse/SDK-34325
            default_logger().deprecation_warning(
                f"Params keys {internal_params} are internal and will be deprecated on future releases.",
                DeprecationVersion.FUTURE,
            )

    def to_pb(self, pb_wrapper):
        compilation_param_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        compilation_param_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_COMPILATION_PARAM
        compilation_param_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_COMPILATION_PARAMS
        compilation_param_cmd_msg.data.compilation_params.layers.extend(self.input_layers)
        self._compilation_params.to_pb(compilation_param_cmd_msg.data.compilation_params, pb_wrapper)
        return compilation_param_cmd_msg

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        for i, layer in enumerate(self._input_layers):
            self._input_layers[i] = self.add_scope_to_layer(scope_names, layer, force=force)


class MuxDemuxCommand(ModelScriptCommandWithScope):
    def __init__(self, output_mux_demux_layers, layers_to_mux, successors_to_demux):
        super().__init__(SupportedCommands.MUX_DEMUX, function_return_vals=output_mux_demux_layers)
        self._layers_to_mux = layers_to_mux
        self._output_mux_demux_layers = output_mux_demux_layers
        self._successors_to_demux = successors_to_demux

    def _all_layers(self):
        layers = self.output_mux_demux_layers + self.layers_to_mux
        if isinstance(self.successors_to_demux, list) and isinstance(self.successors_to_demux[0], list):
            for succ_list in self.successors_to_demux:
                layers.extend(succ_list)
        else:
            layers.extend(self.successors_to_demux)
        return layers

    def _replace_all_layers(self, new_values):
        count = len(self._output_mux_demux_layers)
        self._output_mux_demux_layers = new_values[:count]
        self._layers_to_mux = new_values[count : count + len(self._layers_to_mux)]
        count += len(self._layers_to_mux)

        if isinstance(self.successors_to_demux, list) and isinstance(self.successors_to_demux[0], list):
            new_successors_to_demux = []
            for succ_list in self.successors_to_demux:
                new_successors_to_demux.append(new_values[count : count + len(succ_list)])
                count += len(succ_list)
            self._successors_to_demux = new_successors_to_demux
        else:
            self._successors_to_demux = new_values[count:]

    def __str__(self):
        layers_to_mux = ", ".join(self._layers_to_mux)
        output_mux_demux_layers = ", ".join(self._output_mux_demux_layers)
        successors_to_demux = ", ".join(
            ["[{}]".format(", ".join(sub_group)) for sub_group in self.successors_to_demux],
        )
        return f"{output_mux_demux_layers} = mux_demux([{layers_to_mux}], {successors_to_demux})"

    def remove_scope(self):
        self._layers_to_mux = self._remove_scope(self._layers_to_mux)
        self._output_mux_demux_layers = self._remove_scope(self._output_mux_demux_layers)
        self._successors_to_demux = [self._remove_scope(group) for group in self._successors_to_demux]

    @classmethod
    def from_tokens(cls, tokens):
        layers_to_mux = list(tokens.function_args[0])
        succs_to_demux = list(tokens.function_args[1:])
        output_mux_demux_layers = tokens.multiple_return_vals.asList()
        return cls(output_mux_demux_layers, layers_to_mux, succs_to_demux)

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("operands")
        assert len(command_pb.result) == len(command_pb.data.operands.operand0)
        import numpy as np

        group_indices = np.cumsum([int(i) for i in command_pb.data.operands.operand0])
        layers = list(command_pb.data.operands.operand1)
        layers_to_mux = layers[: group_indices[0]]
        successors_to_demux = [layers[group_indices[i] : group_indices[i + 1]] for i in range(len(group_indices) - 1)]
        output_mux_demux_layers = list(command_pb.result)
        return cls(output_mux_demux_layers, layers_to_mux, successors_to_demux)

    @property
    def layers_to_mux(self):
        return self._layers_to_mux

    @property
    def output_mux_demux_layers(self):
        return self._output_mux_demux_layers

    @property
    def successors_to_demux(self):
        return self._successors_to_demux

    @property
    def group(self):
        return CommandsGroups.NETWORK

    def get_layers(self):
        return self.layers_to_mux + [succ for succlist in self.successors_to_demux for succ in succlist]

    def validate_command(self, layers_scope_from_hn):
        if (
            self.output_mux_demux_layers is None
            or len(self.output_mux_demux_layers) != len(self.successors_to_demux) + 1
        ):
            raise AllocatorScriptParserException(self.msg_prefix + "MuxDemux doesnt have enough return values")
        if (
            self.layers_to_mux is None
            or len(self.layers_to_mux) < 2
            or any(x not in layers_scope_from_hn for x in self.layers_to_mux)
        ):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Input layers to mux {self.layers_to_mux} must be a list of more "
                "than 2 and all layers should be in the scope",
            )
        if self.successors_to_demux is None:
            raise AllocatorScriptParserException(self.msg_prefix + "mux_demux command must have successors_to_demux")
        if any(
            succ not in layers_scope_from_hn for context_groups in self.successors_to_demux for succ in context_groups
        ):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"All the successors layers should be in the scope: {self.successors_to_demux}",
            )
        if len(set(self.layers_to_mux)) < len(self.layers_to_mux):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Input layers to mux {self.layers_to_mux} must be unique",
            )
        if any(len(set(context_group)) < len(context_group) for context_group in self.successors_to_demux):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Successor layers to demux {self.successors_to_demux} must be unique",
            )
        if len(self.successors_to_demux) == 1 and len(self.successors_to_demux[0]) != len(self.layers_to_mux):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Input layers to mux {self.layers_to_mux} must be of same length "
                f"as successor layers to demux when 1 context_group is provided {self.successors_to_demux}",
            )
        default_logger().debug(
            f"Parsed MuxDemux command, args=([{self.layers_to_mux}], {self.successors_to_demux}), return_vals={self.output_mux_demux_layers}",
        )

    def to_pb(self, pb_wrapper):
        mux_demux_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        mux_demux_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_MUX_DEMUX
        mux_demux_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_OPERANDS
        mux_demux_cmd_msg.data.operands.operand0.extend(str(len(self.layers_to_mux)))
        mux_demux_cmd_msg.data.operands.operand1.extend(self.layers_to_mux)
        for sub_group in self.successors_to_demux:
            mux_demux_cmd_msg.data.operands.operand0.extend(str(len(sub_group)))
            mux_demux_cmd_msg.data.operands.operand1.extend(sub_group)
        mux_demux_cmd_msg.result.extend(self.output_mux_demux_layers)
        return mux_demux_cmd_msg

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        for i, layer in enumerate(self._layers_to_mux):
            self._layers_to_mux[i] = self.add_scope_to_layer(scope_names, layer, force=force)
        for i, layer in enumerate(self._output_mux_demux_layers):
            self._output_mux_demux_layers[i] = self.add_scope_to_layer(scope_names, layer, force=force)
        for context_group in self._successors_to_demux:
            for i, layer in enumerate(context_group):
                context_group[i] = self.add_scope_to_layer(scope_names, layer, force=force)


class FromTFCommand(ModelScriptCommandWithScope):
    def __init__(self, layer_from_tf, original_name, hn):
        super().__init__(SupportedCommands.FROM_TF, function_return_vals=layer_from_tf)
        self._layer_from_tf = layer_from_tf
        self._original_name = original_name
        self._hn = hn
        self._hn_layer_name = self.get_hn_layer_name()

    def __str__(self):
        return f"{self._layer_from_tf} = from_tf('{self._original_name}')"

    @classmethod
    def from_tokens(cls, tokens, hn):
        return cls(tokens.single_return_val, tokens.function_args[0].replace("'", ""), hn)

    @property
    def original_name(self):
        return self._original_name

    @property
    def layer_from_tf(self):
        return self._layer_from_tf

    @property
    def group(self):
        return CommandsGroups.GLOBAL

    def get_hn_layer_name(self):
        original_name = self.original_name
        for layer in self._hn:
            if original_name in layer.original_names:
                return layer.name
        raise AllocatorScriptParserException(
            self.msg_prefix + "original layer name given to from_tf command does not exist",
        )

    def get_layers(self):
        return self.get_hn_layer_name()

    def validate_command(self, layers_scope_from_hn):
        hn_layer_name = self.get_hn_layer_name()
        if hn_layer_name not in layers_scope_from_hn:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Cannot find layer {self.layer_from_tf} in existing layers scope.",
            )
        else:
            self._hn_layer_name = hn_layer_name
            default_logger().debug(f"Parsed FromTF command, args=({self.original_name})")

    def to_pb(self, pb_wrapper):
        from_tf_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        from_tf_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_FROM_TF
        from_tf_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_FROM_TF
        from_tf_cmd_msg.data.from_tf.original_name = self.layer_from_tf
        from_tf_cmd_msg.data.from_tf.layer_from_tf = self._hn_layer_name
        return from_tf_cmd_msg

    def _all_layers(self):
        return [self._hn_layer_name, self._layer_from_tf]

    def _replace_all_layers(self, new_values):
        self._hn_layer_name = new_values[0]
        self._layer_from_tf = new_values[1]

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        self._hn_layer_name = self.add_scope_to_layer(scope_names, self._hn_layer_name, force=force)
        self._layer_from_tf = self.add_scope_to_layer(scope_names, self._layer_from_tf, force=force)

    def remove_scope(self):
        self._layer_from_tf = self._remove_scope(self._layer_from_tf)
        self._hn_layer_name = self._remove_scope(self._hn_layer_name)


class BuffersCommand(ModelScriptCommandWithScope):
    def __init__(
        self,
        source_layer,
        dest_layer,
        num_buffers,
        buffers_type="FULL_ROW",
        buffer_features_chunks=None,
        sort_key_func=None,
    ):
        super().__init__(SupportedCommands.BUFFERS, sort_key_func=sort_key_func)
        self._source_layer = source_layer
        self._dest_layer = dest_layer
        self._num_buffers = [int(num) for num in num_buffers]
        self._buffers_type = buffers_type
        if buffer_features_chunks is None:
            self._buffer_features_chunks = AutoInt("automatic")
        else:
            self._buffer_features_chunks = convert_to_auto_int(buffer_features_chunks)

    def _all_layers(self):
        return [self.source_layer, self._dest_layer]

    def _replace_all_layers(self, new_values):
        self._source_layer = new_values[0]
        self._dest_layer = new_values[1]

    def _buffers_str(self, num_buffers):
        num_buffers_string = [str(num) for num in num_buffers]
        return ", ".join(num_buffers_string)

    def __str__(self):
        cmd_str = f"buffers({self._source_layer}, {self._dest_layer}, {self._buffers_str(self._num_buffers)}, {self._buffers_type})"
        if self._buffer_features_chunks.policy() == AutoVariablePolicy.MANUAL or self._buffer_features_chunks.val() > 1:
            cmd_str = cmd_str[:-1] + param_str(", features_chunks", self._buffer_features_chunks) + ")"
        return cmd_str

    def remove_scope(self):
        self._source_layer = self._remove_scope(self._source_layer)
        self._dest_layer = self._remove_scope(self._dest_layer)

    @classmethod
    def from_tokens(cls, tokens):
        src = tokens.function_args[0]
        dst = tokens.function_args[1]
        buffers = tokens.function_args[2:]
        if isinstance(tokens.function_args[-1], str):
            buffers = tokens.function_args[2:-1]
            buffers_type = tokens.function_args[-1]
            return cls(src, dst, buffers, buffers_type)
        elif isinstance(tokens.function_args[-1], dict):
            buffers = tokens.function_args[2:-2]
            buffers_type = tokens.function_args[-2]
            features_chunks = tokens.function_args[-1]["features_chunks"]
            return cls(src, dst, buffers, buffers_type, features_chunks)
        return cls(src, dst, buffers)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_BUFFERS

    @classmethod
    def from_pb(cls, command_pb, sort_key_func=None):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("buffers")
        source_layer = command_pb.data.buffers.src
        dest_layer = command_pb.data.buffers.dst
        num_buffers = command_pb.data.buffers.number_of_buffers
        buffers_type = command_pb.data.buffers.buffers_type
        pb_wrapper = PbWrapper()
        buffer_type_enum = {
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_FULL_ROW: "FULL_ROW",
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_PARTIAL_ROW: "PARTIAL_ROW",
        }
        if (
            command_pb.data.buffers.buffer_features_chunks.policy
            == pb_wrapper.integrated_hw_graph_base_pb2.PROTO_AUTOMATIC
        ):
            features_chunks = AutoInt("automatic")
        else:
            features_chunks = convert_to_auto_int(command_pb.data.buffers.buffer_features_chunks.val)
        return cls(
            source_layer,
            dest_layer,
            num_buffers,
            buffer_type_enum[buffers_type],
            features_chunks,
            sort_key_func=sort_key_func,
        )

    @property
    def source_layer(self):
        return self._source_layer

    @property
    def dest_layer(self):
        return self._dest_layer

    @property
    def num_buffers(self):
        return self._num_buffers

    @property
    def buffers_type(self):
        return self._buffers_type

    @property
    def buffer_features_chunks(self):
        return self._buffer_features_chunks

    @property
    def group(self):
        return CommandsGroups.BUFFERS

    def inner_cmd_sort_key(self):
        if self._sort_key_func:
            return (self._sort_key_func(self.source_layer), self._sort_key_func(self.dest_layer))
        return 0

    def get_layers(self):
        return [self.source_layer, self.dest_layer]

    def validate_command(self, layers_scope_from_hn):
        if self.source_layer is None or self.dest_layer is None or self.has_unfound_layers(layers_scope_from_hn):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Cannot find one of these layers {self.get_layers()} in existing layers scope.",
            )

        if len(self.num_buffers) not in [1, 2]:
            raise AllocatorScriptParserException(
                self.msg_prefix + "number of values to describe the buffers must be 1 or 2",
            )
        for num in self.num_buffers:
            if num is None:
                raise AllocatorScriptParserException(self.msg_prefix + f"number of buffers {num} error")
        default_logger().debug(
            f"Parsed {self.function_name} command, args=({self.source_layer}, {self.dest_layer}, {self._buffers_str(self.num_buffers)}, {self._buffers_type})",
        )

        if self._buffer_features_chunks.policy() == AutoVariablePolicy.MANUAL:
            if self.buffer_features_chunks.val() is None or self.buffer_features_chunks.val() < 1:
                raise AllocatorScriptParserException(
                    self.msg_prefix + "number of buffer_features_chunks must be >= 1",
                )

    def to_pb(self, pb_wrapper):
        buffers_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        buffers_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_BUFFERS
        buffers_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_BUFFERS
        buffers_cmd_msg.data.buffers.src = self.source_layer
        buffers_cmd_msg.data.buffers.dst = self.dest_layer
        buffers_cmd_msg.data.buffers.number_of_buffers.extend(self.num_buffers)
        buffers_cmd_msg.data.buffers.buffers_type = pb_wrapper.BUFFERS_TYPE_TO_PB[self.buffers_type]
        buffers_cmd_msg.data.buffers.buffer_features_chunks.policy = pb_wrapper.AUTO_VARIABLE_POLICY_TO_PB[
            self.buffer_features_chunks.policy()
        ]
        buffers_cmd_msg.data.buffers.buffer_features_chunks.val = self.buffer_features_chunks.val()
        return buffers_cmd_msg

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        self._source_layer = self.add_scope_to_layer(scope_names, self._source_layer, force=force)
        self._dest_layer = self.add_scope_to_layer(scope_names, self._dest_layer, force=force)


class StrategyCommand(ModelScriptCommand):
    def __init__(self, strategy):
        super().__init__(SupportedCommands.STRATEGY)
        self._strategy = strategy

    def __str__(self):
        return f"strategy({self._strategy})"

    def remove_scope(self):
        pass

    @classmethod
    def from_tokens(cls, tokens):
        return cls(tokens.function_args[0])

    @property
    def strategy(self):
        return self._strategy

    @property
    def group(self):
        return CommandsGroups.GLOBAL

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        allowed_strategies = [strategy.value for strategy in AllocatorStrategy]
        if self.strategy is None or self.strategy not in allowed_strategies:
            raise AllocatorScriptParserException(self.msg_prefix + f"allocation strategy {self.strategy} not valid.")
        default_logger().debug(f"Parsed {self.function_name} command, args=({self.strategy})")

    def to_pb(self, pb_wrapper):
        strategy_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        strategy_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_STRATEGY
        strategy_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_OPERANDS
        strategy_cmd_msg.data.operands.operand0.append(self.strategy)
        return strategy_cmd_msg


class AddResourceCommand(ModelScriptCommand):
    def __init__(self, file_path, resource_name):
        super().__init__(SupportedCommands.ADD_RESOURCE)
        self._file_path = file_path
        self._resource_name = resource_name

    def __str__(self):
        return f"add_resource({self._file_path}, {self._resource_name})"

    def remove_scope(self):
        pass

    @classmethod
    def from_tokens(cls, tokens):
        return cls(tokens.function_args[0], tokens.function_args[1])

    @property
    def file_path(self):
        return self._file_path

    @property
    def resource_name(self):
        return self._resource_name

    @property
    def group(self):
        return CommandsGroups.GLOBAL

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        if not os.path.isfile(self.file_path):
            raise AllocatorScriptParserException(self.msg_prefix + f"resource {self.file_path} is not a file.")
        if not self.resource_name:
            raise AllocatorScriptParserException(self.msg_prefix + f"resource {self.file_path} is is missing a name.")
        default_logger().debug(f"Parsed {self.function_name} command, args=({self.file_path})")

    def to_pb(self, pb_wrapper):
        add_resource_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        add_resource_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_ADD_RESOURCE
        add_resource_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_ADD_RESOURCE
        add_resource_cmd_msg.data.add_resource.file_path = self.file_path
        add_resource_cmd_msg.data.add_resource.resource_name = self.resource_name
        return add_resource_cmd_msg


class AllocatorParamCommand(ModelScriptCommand):
    INTERNAL_PARAMS = [
        "agent",
        "builder_exit_point",
        "dannox",
        "enable_loose_mode",
        "enable_macros",
        "max_cluster_util",
        "strategy",
        "resolver_revert_strategy",
        "enable_mjitc",
        "resolver_revert_strategy_asap_delay",
        "microcoder_without_halts",
        "prepost_haltless",
        "optimize_buffers",
        "split_aware_optimize_buffers",
        "buffer_calc_fps",
        "save_latency_timeline",
        "optimize_one_to_many_shmifos",
        "use_minimal_buffers",
        "enable_partial_row_buffers",
        "dump_statistics",
        "resources_strategy",
        "minimal_buffers_at_fifos",
        "enable_muxer_metric",
        "muxer_slack",
        "merge_max_layer_utilization",
        "merge_max_memory_utilization",
        "enable_swapper_router",
        "enable_shiftman",
        "input_feature_splitter_defuse",
        "compilation_num_threads",
        "enable_input_phases",
        "offload_argmax",
        "success_asap",
        "num_of_workers",
        "enable_hw_padding",
        "enable_barakbak_scripts",
        "enable_l3_balloon_optimization",
        "disable_row_per_cut",
        "max_clusters_forcing",
        "number_of_active_clusters",
        "print_minimal_buffers",
        "enable_unikorn",
        "enable_dual_shmifo",
        "enable_post_split_average_buffers",
        "enable_periph",
        "enable_network_groups",
        "enable_conv_2_apus_in_racehorse_pyramids",
        "assert_mux_succeeded",
    ]

    def __init__(self, **allocator_params):
        super().__init__(SupportedCommands.ALLOCATOR_PARAM)
        self._allocator_params = AllocatorParams()
        self._allocator_params.clear()
        self._allocator_params.set(allocator_params)

    def __str__(self):
        return self._export_params_to_string(False)

    def str_to_alls(self):
        return self._export_params_to_string(True)

    def _export_params_to_string(self, is_to_auto_alls):
        allocator_params = ", ".join(
            [
                param_str(param_k, param_v)
                for param_k, param_v in sorted(self._allocator_params.get().items())
                if not (is_to_auto_alls and param_k in self.INTERNAL_PARAMS)
            ],
        )
        return f"allocator_param({allocator_params})"

    def remove_scope(self):
        pass

    @classmethod
    def from_tokens(cls, tokens):
        allocator_params = {}
        for param in tokens.function_args:
            allocator_params.update(param)
        return cls(**allocator_params)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_ALLOCATOR_PARAMS

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("allocator_params")
        params = AllocatorParams()
        params.from_pb(command_pb.data.allocator_params, PbWrapper())
        allocator_params = convert_params_to_str(params.get())
        return cls(**allocator_params)

    @property
    def allocator_params(self):
        return self._allocator_params.get()

    @property
    def group(self):
        return CommandsGroups.GLOBAL

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        validate_params_command(self.allocator_params, AllocatorParams.DEFAULT_PARAMS.keys(), self.msg_prefix)
        self._validate_params_are_not_internal()
        default_logger().debug(f"Parsed AllocatorParam command, args=({self.allocator_params})")

    def _validate_params_are_not_internal(self):
        internal_params = [param_key for param_key in self.allocator_params if param_key in self.INTERNAL_PARAMS]
        if internal_params:
            # TODO: https://hailotech.atlassian.net/browse/SDK-34325
            default_logger().deprecation_warning(
                f"Params keys {internal_params} are internal and will be deprecated on future releases.",
                DeprecationVersion.FUTURE,
            )

    def to_pb(self, pb_wrapper):
        allocator_param_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        allocator_param_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_ALLOCATOR_PARAM
        allocator_param_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_ALLOCATOR_PARAMS
        self._allocator_params.to_pb(allocator_param_cmd_msg.data.allocator_params, pb_wrapper)
        return allocator_param_cmd_msg


class InternalAllocatorParamCommand(AllocatorParamCommand):
    def validate_command(self, layers_scope_from_hn):
        validate_params_command(self.allocator_params, InternalAllocatorParamCommand.INTERNAL_PARAMS, self.msg_prefix)
        default_logger().debug(f"Parsed InternalAllocatorParam command, args=({self.allocator_params})")

    def __str__(self):
        params = ", ".join([param_str(param_k, param_v) for param_k, param_v in sorted(self.allocator_params.items())])
        return f"internal_allocator_param({params})"


class ForceMappingCommand(ModelScriptCommandWithScope):
    TYPE_TO_PROTO_HWNODE_TYPE = {
        "mem": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_MEMORY,
        "SC": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_SUBCLUSTER,
        "IB": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_INPUT,
        "OB": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_OUTPUT,
        "L": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_LAYER,
        "IA": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_IALIGNER,
        "APU": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_APU,
        "SYSIN": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_SYSINPUT,
        "SYSOUT": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_SYSOUTPUT,
        "ppmem": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_PP_MEMORY,
        "PPIB": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_PP_INPUT,
        "PPOB": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_PP_OUTPUT,
        "AGG": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_PP_AGGREGATOR,
        "PPPI": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_PP_INPUT_PORT,
        "PPPO": PbWrapper().integrated_hw_graph_base_pb2.PROTO_HWNODE_PP_OUTPUT_PORT,
    }
    PROTO_HWNODE_TYPE_TO_TYPE = {v: k for k, v in TYPE_TO_PROTO_HWNODE_TYPE.items()}

    def __init__(self, layer_name, hw_node_type, hw_node_sub_index, hw_index):
        super().__init__(SupportedCommands.FORCE_MAPPING)
        self._layer_name = layer_name
        self._hw_node_type = hw_node_type
        self._hw_node_sub_index = hw_node_sub_index
        self._hw_index = hw_index

    def _all_layers(self):
        return [self._layer_name]

    def _replace_all_layers(self, new_values):
        self._layer_name = new_values[0]

    def __str__(self):
        return f"force_mapping({self._layer_name}, {self._hw_node_type}, {self._hw_node_sub_index}, {self._hw_index})"

    def remove_scope(self):
        self._layer_name = self._remove_scope(self._layer_name)

    @classmethod
    def from_tokens(cls, tokens):
        return cls(*tokens.function_args)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_FORCE_MAPPING

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("force_mapping")

        layer_name = command_pb.data.force_mapping.layer_name
        hw_node_type = cls.PROTO_HWNODE_TYPE_TO_TYPE[command_pb.data.force_mapping.hw_node_type]
        hw_node_sub_index = float(command_pb.data.force_mapping.hw_node_sub_index)
        hw_index = float(command_pb.data.force_mapping.hw_index)

        return cls(layer_name, hw_node_type, hw_node_sub_index, hw_index)

    @property
    def hw_index(self):
        return self._hw_index

    @property
    def group(self):
        return CommandsGroups.MAPPING

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        # TODO: implement this
        pass

    def to_pb(self, pb_wrapper):
        force_mapping_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        force_mapping_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_FORCE_MAPPING
        force_mapping_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_FORCE_MAPPING
        force_mapping_cmd_msg.data.force_mapping.layer_name = self._layer_name
        force_mapping_cmd_msg.data.force_mapping.hw_node_type = self.TYPE_TO_PROTO_HWNODE_TYPE[self._hw_node_type]
        force_mapping_cmd_msg.data.force_mapping.hw_node_sub_index = int(self._hw_node_sub_index)
        force_mapping_cmd_msg.data.force_mapping.hw_index = int(self._hw_index)

        return force_mapping_cmd_msg

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        self._layer_name = self.add_scope_to_layer(scope_names, self._layer_name, force=force)


class ForceRouteCommand(ModelScriptCommandWithScope):
    def __init__(self, src_layer_name, dst_layer_name, ports, axis_upsize_index):
        super().__init__(SupportedCommands.FORCE_ROUTE)
        self._src_layer_name = src_layer_name
        self._dst_layer_name = dst_layer_name
        self._ports = ports
        self._axis_upsize_index = axis_upsize_index

    def _all_layers(self):
        return [self._src_layer_name, self._dst_layer_name]

    def _replace_all_layers(self, new_values):
        self._src_layer_name = new_values[0]
        self._dst_layer_name = new_values[1]

    def __str__(self):
        ports = ", ".join(map(str, self.ports))
        return f"force_route({self._src_layer_name}, {self._dst_layer_name}, [{ports}], axis_upsize_index={self._axis_upsize_index})"

    def remove_scope(self):
        self._src_layer_name = self._remove_scope(self._src_layer_name)
        self._dst_layer_name = self._remove_scope(self._dst_layer_name)

    @classmethod
    def from_tokens(cls, tokens):
        src_layer_name = tokens.function_args[0]
        dst_layer_name = tokens.function_args[1]
        ports = tokens.function_args[2]
        axis_upsize_index = 0
        if len(tokens.function_args) == 4:
            if "axis_upsize_index" not in tokens.function_args[3]:
                raise AllocatorScriptParserException(
                    f"Failed to parse {tokens.function_name} command - unexpected kwarg in force route command",
                )
            axis_upsize_index = tokens.function_args[3]["axis_upsize_index"]
        return cls(src_layer_name, dst_layer_name, ports, int(axis_upsize_index))

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_FORCE_ROUTE

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("force_route")

        src_layer_name = command_pb.data.force_route.src_layer_name
        dst_layer_name = command_pb.data.force_route.dst_layer_name
        ports = command_pb.data.force_route.ports
        axis_upsize_index = command_pb.data.force_route.axis_upsize_index
        return cls(src_layer_name, dst_layer_name, ports, axis_upsize_index)

    @property
    def src_layer_name(self):
        return self._src_layer_name

    @property
    def dst_layer_name(self):
        return self._dst_layer_name

    @property
    def ports(self):
        return self._ports

    @property
    def axis_upsize_index(self):
        return self._axis_upsize_index

    @property
    def group(self):
        return CommandsGroups.MAPPING

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        pass

    def to_pb(self, pb_wrapper):
        force_route_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        force_route_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_FORCE_ROUTE
        force_route_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_FORCE_ROUTE
        force_route_cmd_msg.data.force_route.src_layer_name = self.src_layer_name
        force_route_cmd_msg.data.force_route.dst_layer_name = self.dst_layer_name
        force_route_cmd_msg.data.force_route.ports.extend([int(port) for port in self.ports])
        force_route_cmd_msg.data.force_route.axis_upsize_index = self.axis_upsize_index
        return force_route_cmd_msg

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        self._src_layer_name = self.add_scope_to_layer(scope_names, self._src_layer_name, force=force)
        self._dst_layer_name = self.add_scope_to_layer(scope_names, self._dst_layer_name, force=force)


class PrintBuffersCommand(ModelScriptCommand):
    def __init__(self):
        super().__init__(SupportedCommands.PRINT_BUFFERS)

    def __str__(self):
        return "print_buffers()"

    @property
    def group(self):
        return CommandsGroups.GLOBAL

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        default_logger().debug("Parsed PrintBuffers command")

    def to_pb(self, pb_wrapper):
        print_buffers_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        print_buffers_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_PRINT_BUFFERS
        print_buffers_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_NO_DATA
        return print_buffers_cmd_msg

    def remove_scope(self):
        pass


class BufferCalcCommand(AllocatorParamCommand):
    BUFFER_CALC_PARAMS = [
        "optimize_buffers",
        "split_aware_optimize_buffers",
        "buffer_calc_fps",
        "save_latency_timeline",
    ]

    def __str__(self):
        params = ", ".join(
            [
                param_str(param_k, param_v)
                for param_k, param_v in sorted(self.allocator_params.items())
                if param_k in self.BUFFER_CALC_PARAMS
            ],
        )
        return f"buffer_calc_param({params})"

    @property
    def group(self):
        return CommandsGroups.GLOBAL

    def validate_command(self, layers_scope_from_hn):
        default_logger().debug("Parsed BufferCalc command")


class OptimizeBuffersCommand(ModelScriptCommand):
    def __init__(self):
        super().__init__(SupportedCommands.OPTIMIZE_BUFFERS)

    def __str__(self):
        return "optimize_buffers()"

    @property
    def group(self):
        return CommandsGroups.GLOBAL

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        default_logger().debug("Parsed OptimizeBuffers command")

    def to_pb(self, pb_wrapper):
        optimize_buffers_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        optimize_buffers_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_OPTIMIZE_BUFFERS
        optimize_buffers_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_NO_DATA
        return optimize_buffers_cmd_msg


class ContextCommand(ModelScriptCommandWithScope):
    def __init__(self, context_name, layers):
        super().__init__(SupportedCommands.CONTEXT)
        self._context_name = context_name
        self._layers = layers

    def _all_layers(self):
        return self.layers

    def _replace_all_layers(self, new_values):
        for layer in self._layers:
            if layer in new_values:
                new_values.remove(layer)
        self._layers = new_values

    def prefix_in_command(self, prefix):
        # NOTE: account for improperly named layers
        return is_prefix_in_layers_list_relaxed(prefix, self._all_layers())

    @property
    def context_name(self):
        return self._context_name

    @property
    def layers(self):
        return self._layers

    def set_name(self, name):
        self._context_name = name

    def remove_scope(self):
        self._layers = self._remove_scope(self._layers)

    @classmethod
    def from_tokens(cls, tokens):
        return cls(tokens.single_return_val, tokens.function_args[0])

    def get_layers(self):
        return self._layers

    def has_unfound_layers(self, layers_scope_from_hn):
        return len(self.get_unfound_layers(layers_scope_from_hn)) > 0

    def get_unfound_layers(self, layers_scope_from_hn):
        scopes_names = get_scopes_set_from_layers(layers_scope_from_hn)
        return [layer for layer in self._layers if layer not in layers_scope_from_hn and layer not in scopes_names]

    def validate_command(self, layers_scope_from_hn):
        if not isinstance(self._context_name, str):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"context name {self._context_name} is not a valid string.",
            )
        if self.has_unfound_layers(layers_scope_from_hn):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"context layers {self.get_unfound_layers(layers_scope_from_hn)} does not exist.",
            )
        if "/" in self._context_name:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"context name {self._context_name} cannot contain '/' character.",
            )
        default_logger().debug(f"Parsed Context command, args={self._layers}, return_vals={self._context_name}")

    def handle_unfound_layers(self, layers_scope_from_hn):
        for layer in self.get_unfound_layers(layers_scope_from_hn):
            self._layers.remove(layer)
        return bool(self._layers)

    def to_pb(self, pb_wrapper):
        context_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        context_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_CONTEXT
        context_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CONTEXT
        context_cmd_msg.data.context.name = self._context_name
        context_cmd_msg.data.context.layers.extend(self._layers)
        return context_cmd_msg

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CONTEXT

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("context")
        layers = list(command_pb.data.context.layers)
        context_name = command_pb.data.context.name
        return cls(context_name, layers)

    def __str__(self):
        layers = ", ".join(sorted(self._layers))
        return f"{self._context_name} = context([{layers}])"

    @property
    def group(self):
        return CommandsGroups.CONTEXT

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        for i, layer in enumerate(self._layers):
            if layer not in scope_names:
                self._layers[i] = self.add_scope_to_layer(scope_names, layer, force=force)


class MergeCommand(ModelScriptCommandWithScope):
    def __init__(self, merged_layer, layers_to_merge):
        super().__init__(SupportedCommands.MERGE, function_return_vals=merged_layer)
        self._merged_layer = merged_layer
        self._layers_to_merge = layers_to_merge

    def _all_layers(self):
        return [self.merged_layer, *self.layers_to_merge]

    def _replace_all_layers(self, new_values):
        self._merged_layer = new_values[0]
        self._layers_to_merge = new_values[1:]

    def __str__(self):
        layers_to_merge = ", ".join(self._layers_to_merge)
        return f"{self._merged_layer} = merge({layers_to_merge})"

    def remove_scope(self):
        self._merged_layer = self._remove_scope(self._merged_layer)
        self._layers_to_merge = self._remove_scope(self._layers_to_merge)

    @classmethod
    def from_tokens(cls, tokens):
        layers_to_merge = tokens.function_args.asList()
        return cls(tokens.single_return_val, layers_to_merge)

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("operands")
        merged_layer = command_pb.result[0]
        layers_to_merge = list(command_pb.data.operands.operand0)
        return cls(merged_layer, layers_to_merge)

    @property
    def merged_layer(self):
        return self._merged_layer

    @property
    def layers_to_merge(self):
        return self._layers_to_merge

    @property
    def group(self):
        return CommandsGroups.NETWORK

    def get_layers(self):
        return self.layers_to_merge

    def validate_command(self, layers_scope_from_hn):
        if self.merged_layer is None or self.merged_layer in layers_scope_from_hn:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Cannot find layer {self.merged_layer} in existing layers scope.",
            )
        if self.layers_to_merge is None or len(self.layers_to_merge) < 2:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"layers to merge {self.layers_to_merge} must be a list of more " "than 1 layer",
            )
        if self.has_unfound_layers(layers_scope_from_hn):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"not all layers ({self.layers_to_merge}) to merge are found in the scope",
            )
        self.validate_single_scope_in_command("Cannot merge layers from different scopes")
        default_logger().debug(f"Parsed Merge command, args={self.layers_to_merge}, return_vals={self.merged_layer}")

    def handle_unfound_layers(self, layers_scope_from_hn):
        non_existing_layers = [x for x in self.layers_to_merge if x not in layers_scope_from_hn]
        for layer in non_existing_layers:
            self.layers_to_merge.remove(layer)
        return not len(self.layers_to_merge) < 2

    def to_pb(self, pb_wrapper):
        merge_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        merge_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_MERGE
        merge_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_OPERANDS
        merge_cmd_msg.data.operands.operand0.extend(self.layers_to_merge)
        merge_cmd_msg.result.append(self.merged_layer)
        return merge_cmd_msg

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        self._merged_layer = self.add_scope_to_layer(scope_names, self._merged_layer, force=force)
        for i, layer in enumerate(self._layers_to_merge):
            self._layers_to_merge[i] = self.add_scope_to_layer(scope_names, layer, force=force)


class OutputLayerCommand(ModelScriptCommandWithScope):
    def __init__(self, output_layer_name, source_layer, offdevice_layer=None):
        super().__init__(SupportedCommands.OUTPUT_LAYER, function_return_vals=output_layer_name)
        self._output_layer_name = output_layer_name
        self._source_layer = source_layer
        self._offdevice_layer = offdevice_layer

    def _all_layers(self):
        return [self.output_layer_name, self.source_layer]

    def _replace_all_layers(self, new_values):
        self._output_layer_name = new_values[0]
        self._source_layer = new_values[1]
        if len(new_values) > 2:
            self._offdevice_layer = new_values[2]

    @property
    def output_layer_name(self):
        return self._output_layer_name

    def remove_scope(self):
        self._output_layer_name = self._remove_scope(self._output_layer_name)
        self._source_layer = self._remove_scope(self._source_layer)
        if self._offdevice_layer is not None:
            self._offdevice_layer = self._remove_scope(self._offdevice_layer)

    @property
    def source_layer(self):
        return self._source_layer

    @property
    def offdevice_layer(self):
        return self._offdevice_layer

    @classmethod
    def from_tokens(cls, tokens):
        return cls(
            tokens.single_return_val,
            tokens.function_args[0],
            tokens.function_args[1]["offdevice_layer"] if len(tokens.function_args) > 1 else None,
        )

    def __str__(self):
        if self._offdevice_layer is not None:
            return f"{self._output_layer_name} = output_layer({self._source_layer}, offdevice_layer={self._offdevice_layer})"
        return f"{self._output_layer_name} = output_layer({self._source_layer})"

    @property
    def group(self):
        return CommandsGroups.NETWORK

    def get_layers(self):
        return [self.source_layer]

    def validate_command(self, layers_scope_from_hn):
        if self.output_layer_name is None or self.output_layer_name in layers_scope_from_hn:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Layer {self.output_layer_name} is None or already exist in layers scope.",
            )
        if self.source_layer is None or self.has_unfound_layers(layers_scope_from_hn):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Cannot find one of these layers {self.get_layers()} in existing layers scope.",
            )
        if self.offdevice_layer is not None and self.offdevice_layer not in layers_scope_from_hn:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Cannot find these layer {self.offdevice_layer} in existing layers scope.",
            )

    def to_pb(self, pb_wrapper):
        output_layer_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        output_layer_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_OUTPUT_LAYER
        output_layer_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_OPERANDS
        output_layer_cmd_msg.data.operands.operand0.append(self.source_layer)
        if self.offdevice_layer is not None:
            output_layer_cmd_msg.data.operands.operand1.append(self.offdevice_layer)
        else:
            output_layer_cmd_msg.data.operands.operand1.append("")
        output_layer_cmd_msg.result.append(self.output_layer_name)
        return output_layer_cmd_msg

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("operands")
        output_layer_name = command_pb.result[0]
        source_layer = command_pb.data.operands.operand0[0]
        offdevice_layer = command_pb.data.operands.operand1[0] if len(command_pb.data.operands.operand1) > 0 else None
        return cls(output_layer_name, source_layer, offdevice_layer)

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        self._output_layer_name = self.add_scope_to_layer(scope_names, self._output_layer_name, force=force)
        self._source_layer = self.add_scope_to_layer(scope_names, self._source_layer, force=force)
        if self._offdevice_layer is not None:
            self._offdevice_layer = self.add_scope_to_layer(scope_names, self._offdevice_layer, force=force)


class ShapeSplitterCommand(ModelScriptCommandWithScope):
    def __init__(self, output_layer_name, split_type, original_layer, successors):
        super().__init__(
            SupportedCommands.SHAPE_SPLITTER,
            function_return_vals=output_layer_name,
        )
        self._output_layer_name = output_layer_name
        if isinstance(split_type, str):
            if split_type == ShapeSplitterType.SPLIT_HEIGHT.value:
                self._split_type = ShapeSplitterType.SPLIT_HEIGHT
            elif split_type == ShapeSplitterType.SPLIT_WIDTH.value:
                self._split_type = ShapeSplitterType.SPLIT_WIDTH
            elif split_type == ShapeSplitterType.SPLIT_FEATURES.value:
                self._split_type = ShapeSplitterType.SPLIT_FEATURES
            else:
                raise AllocatorScriptParserException(self.msg_prefix + f"split type {split_type} is not supported.")
        elif type(split_type) is ShapeSplitterType:
            self._split_type = split_type
        else:
            raise AllocatorScriptParserException(self.msg_prefix + f"split type {split_type} is not supported.")
        self._original_layer = original_layer
        self._successors = successors

    def _all_layers(self):
        return [self.output_layer_name, self.original_layer, *self.successors]

    def _replace_all_layers(self, new_values):
        self._output_layer_name = new_values[0]
        self._original_layer = new_values[1]
        self._successors = new_values[2:]

    def remove_scope(self):
        self._output_layer_name = self._remove_scope(self._output_layer_name)
        self._original_layer = self._remove_scope(self._original_layer)
        self._successors = self._remove_scope(self._successors)

    @property
    def output_layer_name(self):
        return self._output_layer_name

    @property
    def original_layer(self):
        return self._original_layer

    @property
    def successors(self):
        return self._successors

    @property
    def split_type(self):
        return self._split_type

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_SHAPE_SPLITTER

    @classmethod
    def from_tokens(cls, tokens):
        output_layer_name = tokens.single_return_val
        if tokens.function_name == SupportedCommands.FEATURE_SPLITTER.value:  # For backward compatibility
            split_type = ShapeSplitterType.SPLIT_FEATURES
            start_index = 0
        else:
            split_type = tokens.function_args[0]
            start_index = 1
        original_layer = tokens.function_args[start_index]
        successors = (
            tokens.function_args[start_index + 1].asList()
            if hasattr(tokens.function_args[start_index + 1], "asList")
            else tokens.function_args[start_index + 1]
        )
        return cls(output_layer_name, split_type, original_layer, successors)

    def __str__(self):
        successors = ", ".join(self.successors)
        return f"{self.output_layer_name} = shape_splitter({self._split_type.value}, {self.original_layer}, [{successors}])"

    @property
    def group(self):
        return CommandsGroups.NETWORK

    def get_layers(self):
        return [self.original_layer, *self.successors]

    def get_unfound_layers(self, layers_scope_from_hn):
        return [layer for layer in self.get_layers() if layer not in layers_scope_from_hn]

    def validate_command(self, layers_scope_from_hn):
        if self.output_layer_name is None or self.output_layer_name in layers_scope_from_hn:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Layer {self.output_layer_name} in None, or already exist in layers scope.",
            )
        if self.original_layer is None:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"source layer {self.original_layer} not in layers scope.",
            )
        if self.successors is None or len(self.successors) < 1:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"successors {self.successors} must be a list of more than 0 "
                "and all layers should be in the scope",
            )
        if self.has_unfound_layers(layers_scope_from_hn):
            raise AllocatorScriptParserException(
                self.msg_prefix
                + f"Cannot find layers {self.get_unfound_layers(layers_scope_from_hn)} in existing layers scope.",
            )
        self.validate_single_scope_in_command("Cannot shape split between scopes")
        default_logger().debug(
            f"Parsed {self.function_name} command, args=({self.original_layer}, {self.successors}), return_vals={self.output_layer_name}",
        )

    def to_pb(self, pb_wrapper):
        output_layer_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        output_layer_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_SHAPE_SPLITTER
        output_layer_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_SHAPE_SPLITTER
        output_layer_cmd_msg.data.shape_splitter.layer_name = self.output_layer_name
        output_layer_cmd_msg.data.shape_splitter.split_type = pb_wrapper.SHAPE_SPLITTER_TYPE_TYPE_TO_PB[
            self._split_type
        ]
        output_layer_cmd_msg.data.shape_splitter.source_layer = self.original_layer
        output_layer_cmd_msg.data.shape_splitter.dest_layers.extend(self.successors)
        output_layer_cmd_msg.result.append(self.output_layer_name)
        return output_layer_cmd_msg

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        output_layer_name = command_pb.data.shape_splitter.layer_name
        split_type = PbWrapper().SHAPE_SPLITTER_TYPE_PB_TO_TYPE[command_pb.data.shape_splitter.split_type]
        original_layer = command_pb.data.shape_splitter.source_layer
        successors = list(command_pb.data.shape_splitter.dest_layers)
        return cls(output_layer_name, split_type, original_layer, successors)

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        self._output_layer_name = self.add_scope_to_layer(scope_names, self._output_layer_name, force=force)
        self._original_layer = self.add_scope_to_layer(scope_names, self._original_layer, force=force)
        for i, layer in enumerate(self._successors):
            self._successors[i] = self.add_scope_to_layer(scope_names, layer, force=force)


class ResourcesParamCommand(ModelScriptCommand):
    INTERNAL_PARAMS = ["strategy", "objective"]

    def __init__(self, **resources_param):
        super().__init__(SupportedCommands.RESOURCES_PARAM)
        self._resources_params = ResourcesParams()
        self._resources_params.clear()
        self._resources_params.set(resources_param)

    def __str__(self):
        return self._export_params_to_string(False)

    def str_to_alls(self):
        return self._export_params_to_string(True)

    def _export_params_to_string(self, is_to_auto_alls):
        filtered_dict = {
            param_k: param_v
            for param_k, param_v in sorted(self._resources_params.get().items())
            if (param_k not in self.INTERNAL_PARAMS)
        }
        if not filtered_dict:
            return ""
        resources_params = ", ".join([param_str(param_k, param_v) for param_k, param_v in filtered_dict.items()])
        return f"resources_param({resources_params})"

    def remove_scope(self):
        pass

    @classmethod
    def from_tokens(cls, tokens):
        resources_params = {}
        for param in tokens.function_args:
            resources_params.update(param)
        return cls(**resources_params)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_RESOURCES_PARAM

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("resources_params")
        params = ResourcesParams()
        params.from_pb(command_pb.data.resources_params, PbWrapper())
        resources_params = convert_params_to_str(params.get())
        return cls(**resources_params)

    @property
    def resources_param(self):
        return self._resources_params.get()

    @property
    def group(self):
        return CommandsGroups.GLOBAL

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        validate_params_command(self.resources_param, ResourcesParams.DEFAULT_PARAMS.keys(), self.msg_prefix)
        self._validate_params_are_not_internal()
        default_logger().debug(f"Parsed ResourcesParam command, args=({self._resources_params})")

    def _validate_params_are_not_internal(self):
        internal_params = [param_key for param_key in self.resources_param if param_key in self.INTERNAL_PARAMS]
        if internal_params:
            # TODO: https://hailotech.atlassian.net/browse/SDK-34325
            default_logger().deprecation_warning(
                f"Params keys {internal_params} are internal and will be deprecated on future releases.",
                DeprecationVersion.FUTURE,
            )

    def to_pb(self, pb_wrapper):
        resources_param_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        resources_param_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_RESOURCES_PARAM
        resources_param_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_RESOURCES_PARAM
        self._resources_params.to_pb(resources_param_cmd_msg.data.resources_params, pb_wrapper)
        return resources_param_cmd_msg


class ContextSwitchParamCommand(ModelScriptCommand):
    INTERNAL_PARAMS = ["partitioner"]

    def __init__(self, **context_switch_params):
        super().__init__(SupportedCommands.CONTEXT_SWITCH_PARAM)
        self._context_switch_params = ContextSwitchParams()
        self._context_switch_params.clear()
        self._context_switch_params.set(context_switch_params)

    def __str__(self):
        return self._export_params_to_string(False)

    def str_to_alls(self):
        return self._export_params_to_string(True)

    def _export_params_to_string(self, is_to_auto_alls):
        context_switch_params = ", ".join(
            [
                param_str(param_k, param_v)
                for param_k, param_v in sorted(self._context_switch_params.get().items())
                if not (is_to_auto_alls and param_k in self.INTERNAL_PARAMS)
            ],
        )
        return f"context_switch_param({context_switch_params})"

    def remove_scope(self):
        pass

    @classmethod
    def from_tokens(cls, tokens):
        context_switch_params = {}
        for param in tokens.function_args:
            param.pop("partition_by_network_util", None)
            context_switch_params.update(param)
        if not context_switch_params:
            return None
        return cls(**context_switch_params)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CONTEXT_SWITCH_PARAM

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("context_switch_params")
        params = ContextSwitchParams()
        params.from_pb(command_pb.data.context_switch_params, PbWrapper())
        context_switch_params = convert_params_to_str(params.get())
        return cls(**context_switch_params)

    @property
    def context_switch_params(self):
        return self._context_switch_params.get()

    @property
    def group(self):
        return CommandsGroups.GLOBAL

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        validate_params_command(self.context_switch_params, ContextSwitchParams.DEFAULT_PARAMS.keys(), self.msg_prefix)
        self._validate_params_are_not_internal()
        default_logger().debug(f"Parsed ContextSwitchParam command, args=({self.context_switch_params})")

    def _validate_params_are_not_internal(self):
        internal_params = [param_key for param_key in self.context_switch_params if param_key in self.INTERNAL_PARAMS]
        if internal_params:
            # TODO: https://hailotech.atlassian.net/browse/SDK-34325
            default_logger().deprecation_warning(
                f"Params keys {internal_params} are internal and will be deprecated on future releases.",
                DeprecationVersion.FUTURE,
            )

    def to_pb(self, pb_wrapper):
        context_switch_param_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        context_switch_param_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_CONTEXT_SWITCH_PARAM
        context_switch_param_cmd_msg.data.type = (
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CONTEXT_SWITCH_PARAM
        )
        self._context_switch_params.to_pb(context_switch_param_cmd_msg.data.context_switch_params, pb_wrapper)
        return context_switch_param_cmd_msg


class InternalContextSwitchParamCommand(ContextSwitchParamCommand):
    def validate_command(self, layers_scope_from_hn):
        validate_params_command(
            self.context_switch_params,
            InternalContextSwitchParamCommand.INTERNAL_PARAMS,
            self.msg_prefix,
        )
        default_logger().debug(f"Parsed InternalContextSwitchParamCommand command, args=({self.context_switch_params})")

    def __str__(self):
        params = ", ".join(
            [param_str(param_k, param_v) for param_k, param_v in sorted(self.context_switch_params.items())],
        )
        return f"internal_context_switch_param({params})"


class HefParamCommand(ModelScriptCommand):
    INTERNAL_PARAMS = [
        "should_use_sequencer_l2_interleave",
        "enable_axis_upsize_workaround",
        "should_prioritize_sequencer_over_l3",
        "dump_debug_params",
        "dump_debug_hef_per_context",
        "delay_context_switch",
        "should_use_confifo",
        "enable_ko",
        "enable_lcu_from_sequencer",
        "num_preliminary_groups",
        "dma_engine_count",
    ]

    def __init__(self, **hef_params):
        super().__init__(SupportedCommands.HEF_PARAM)
        self._hef_params = HefParams()
        self._hef_params.clear()
        self._hef_params.set(hef_params)

    def __str__(self):
        return self._export_params_to_string(False)

    def str_to_alls(self):
        return self._export_params_to_string(True)

    def _export_params_to_string(self, is_to_auto_alls):
        hef_params = ", ".join(
            [
                param_str(param_k, param_v)
                for param_k, param_v in sorted(self._hef_params.get().items())
                if not (is_to_auto_alls and param_k in self.INTERNAL_PARAMS)
            ],
        )
        return f"hef_param({hef_params})"

    @classmethod
    def from_tokens(cls, tokens):
        hef_params = {}
        for param in tokens.function_args:
            hef_params.update(param)
        return cls(**hef_params)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_HEF_PARAMS

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("hef_params")
        params = HefParams()
        params.from_pb(command_pb.data.hef_params, PbWrapper())
        hef_params = convert_params_to_str(params.get())
        return cls(**hef_params)

    @property
    def hef_params(self):
        return self._hef_params.get()

    @property
    def group(self):
        return CommandsGroups.GLOBAL

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        validate_params_command(self.hef_params, HefParams.DEFAULT_PARAMS.keys(), self.msg_prefix)
        self._validate_params_are_not_internal()
        default_logger().debug(f"Parsed HefParams command, args=({self.hef_params})")

    def _validate_params_are_not_internal(self):
        internal_params = [param_key for param_key in self.hef_params if param_key in self.INTERNAL_PARAMS]
        if internal_params:
            # TODO: https://hailotech.atlassian.net/browse/SDK-34325
            default_logger().deprecation_warning(
                f"Params keys {internal_params} are internal and will be deprecated on future releases.",
                DeprecationVersion.FUTURE,
            )

    def to_pb(self, pb_wrapper):
        hef_param_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        hef_param_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_HEF_PARAM
        hef_param_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_HEF_PARAMS
        self._hef_params.to_pb(hef_param_cmd_msg.data.hef_params, pb_wrapper)
        return hef_param_cmd_msg

    def remove_scope(self):
        pass


class ContextCompilationParamCommand(ModelScriptCommand):
    def __init__(self, context, **compilation_params):
        super().__init__(SupportedCommands.CONTEXT_COMPILATION_PARAM)
        self._context = context
        self._compilation_params = ContextCompilationParams()
        self._compilation_params.set(compilation_params)

    def __str__(self):
        compilation_params = ", ".join(
            [param_str(param_k, param_v) for param_k, param_v in sorted(self._compilation_params.get().items())],
        )
        return f"{self._context}.compilation_param({compilation_params})"

    def remove_scope(self):
        pass

    @classmethod
    def from_tokens(cls, tokens):
        params = {}
        for param in tokens.function_args.asList():
            params.update(param)
        return cls(tokens.object, **params)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CONTEXT_COMPILATION_PARAMS

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("context_compilation_params")
        params = ContextCompilationParams()
        params.from_pb(command_pb.data.context_compilation_params, PbWrapper())
        context_compilation_params = convert_params_to_str(params.get())
        context = command_pb.data.context_compilation_params.context
        return cls(context, **context_compilation_params)

    @property
    def context(self):
        return self._context

    @property
    def compilation_params(self):
        return self._compilation_params.get()

    @property
    def group(self):
        return CommandsGroups.CONTEXT

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        # NOTE: valid context is validated in cpp
        validate_params_command(
            self.compilation_params,
            ContextCompilationParams.DEFAULT_PARAMS.keys(),
            self.msg_prefix,
        )
        default_logger().debug(
            f"Parsed ContextCompilationParam command, args=({self.context}, {self.compilation_params})",
        )

    def to_pb(self, pb_wrapper):
        context_compilation_param_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        context_compilation_param_cmd_msg.op = (
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_CONTEXT_COMPILATION_PARAM
        )
        context_compilation_param_cmd_msg.data.type = (
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CONTEXT_COMPILATION_PARAMS
        )
        context_compilation_param_cmd_msg.data.context_compilation_params.context = self.context
        self._compilation_params.to_pb(context_compilation_param_cmd_msg.data.context_compilation_params, pb_wrapper)
        return context_compilation_param_cmd_msg


class ContextResourcesParamCommand(ModelScriptCommand):
    def __init__(self, context, **context_resources_params):
        super().__init__(SupportedCommands.CONTEXT_RESOURCES_PARAM)
        self._context = context
        self._context_resources_params = ContextResourcesParams()
        self._context_resources_params.clear()
        self._context_resources_params.set(context_resources_params)

    def __str__(self):
        context_resources_params = ", ".join(
            [param_str(param_k, param_v) for param_k, param_v in sorted(self._context_resources_params.get().items())],
        )
        return f"{self._context}.resources_param({context_resources_params})"

    def remove_scope(self):
        pass

    @classmethod
    def from_tokens(cls, tokens):
        params = {}
        for param in tokens.function_args.asList():
            params.update(param)
        return cls(tokens.object, **params)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CONTEXT_RESOURCES_PARAMS

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("context_resources_params")
        params = ContextResourcesParams()
        params.from_pb(command_pb.data.context_resources_params.context_resources_params, PbWrapper())
        context_resources_params = convert_params_to_str(params.get())
        context = command_pb.data.context_resources_params.context
        return cls(context, **context_resources_params)

    @property
    def context(self):
        return self._context

    @property
    def context_resources_params(self):
        return self._context_resources_params.get()

    @property
    def group(self):
        return CommandsGroups.CONTEXT

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        # NOTE: valid context is validated in cpp
        validate_params_command(
            self.context_resources_params,
            ContextResourcesParams.DEFAULT_PARAMS.keys(),
            self.msg_prefix,
        )
        default_logger().debug(
            f"Parsed ContextResourceParam command, args=({self.context}, {self.context_resources_params})",
        )

    def to_pb(self, pb_wrapper):
        context_resources_param_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        context_resources_param_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_CONTEXT_RESOURCES_PARAM
        context_resources_param_cmd_msg.data.type = (
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CONTEXT_RESOURCES_PARAMS
        )
        context_resources_param_cmd_msg.data.context_resources_params.context = self.context
        self._context_resources_params.to_pb(
            context_resources_param_cmd_msg.data.context_resources_params.context_resources_params,
            pb_wrapper,
        )
        return context_resources_param_cmd_msg


class LoggerParamCommand(ModelScriptCommand):
    def __init__(self, logger_names, **logger_params):
        super().__init__(SupportedCommands.LOGGER_PARAM)
        self._logger_names = logger_names if isinstance(logger_names, list) else [logger_names]
        self._logger_params = LoggerParams()
        self._logger_params.clear()
        self._logger_params.set(logger_params)

    def __str__(self):
        loggers = ",".join(self._logger_names)
        loggers = f"[{loggers}]" if len(self._logger_names) > 1 else loggers
        return f"logger_param({loggers}, {self._logger_params})"

    def remove_scope(self):
        pass

    @classmethod
    def from_tokens(cls, tokens):
        logger_params = {}
        args = tokens.function_args.asList()
        for param in args[1:]:
            logger_params.update(param)
        return cls(args[0], **logger_params)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_LOGGER_PARAM

    @property
    def logger_params(self):
        return self._logger_params.get()

    @property
    def loggers(self):
        return self._logger_names

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        validate_params_command(self.logger_params, LoggerParams.DEFAULT_PARAMS.keys(), self.msg_prefix)
        default_logger().debug(f"Parsed LoggerParams command, loggers={self.loggers}, args=({self.logger_params})")

    def to_pb(self, pb_wrapper):
        logger_params_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        logger_params_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_LOGGER_PARAM
        logger_params_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_LOGGER_PARAM
        logger_params_cmd_msg.data.logger_params.loggers.extend(self._logger_names)
        self._logger_params.to_pb(logger_params_cmd_msg.data.logger_params.params, pb_wrapper)
        return logger_params_cmd_msg


class CascadeNodeType(Enum):
    SHORTCUT = "shortcut"
    PORTAL3 = "portal"
    PORTAL4 = "l4_portal"
    DDR = "ddr"

    @classmethod
    def list(cls):
        return [c.value for c in cls]


class CascadeCommandType(Enum):
    BUFFERS_LIST = 0
    WEIGHTS_LIST = 1
    TOTAl_BUFFERS = 2
    EQUAL_WEIGHTS = 3

    @classmethod
    def list(cls):
        return [c.value for c in cls]


class CascadeCommand(ModelScriptCommandWithScope):
    def __init__(self, source_layer, target_layer, cascade_types_list, arguments, cascade_layers):
        super().__init__(SupportedCommands.CASCADE, function_return_vals=cascade_layers)
        self._cascade_layers = cascade_layers
        self._source_layer = source_layer
        self._target_layer = target_layer
        self._cascade_types_list = cascade_types_list

        if (arguments is None) or (arguments == "equal_weights"):
            self._command_type = CascadeCommandType.EQUAL_WEIGHTS
            self._arguments = None
        elif "total_buffers" in arguments:
            self._command_type = CascadeCommandType.TOTAl_BUFFERS
            self._arguments = int(arguments["total_buffers"])
        elif "buffers" in arguments:
            self._command_type = CascadeCommandType.BUFFERS_LIST
            self._arguments = [int(x) for x in arguments["buffers"]]
        elif "weights" in arguments:
            self._command_type = CascadeCommandType.WEIGHTS_LIST
            self._arguments = arguments["weights"]
        else:
            raise AllocatorScriptParserException(self.msg_prefix + "unsupported format of arguments")

    def _all_layers(self):
        return [self.source_layer, self.target_layer, *self.cascade_layers]

    def _replace_all_layers(self, new_values):
        self._source_layer = new_values[0]
        self._target_layer = new_values[1]
        self._cascade_layers = new_values[2:]

    def __str__(self):
        cascade_layers = ", ".join(self._cascade_layers)
        cascade_types_list = "[" + ", ".join(self._cascade_types_list) + "]"
        if self._command_type == CascadeCommandType.EQUAL_WEIGHTS:
            res = f"{cascade_layers} = cascade({self._source_layer}, {self._target_layer}, types={cascade_types_list}, equal_weights)"
        elif self._command_type == CascadeCommandType.TOTAl_BUFFERS:
            res = f"{cascade_layers} = cascade({self._source_layer}, {self._target_layer}, types={cascade_types_list}, total_buffers={self._arguments})"
        elif self._command_type == CascadeCommandType.BUFFERS_LIST:
            res = f"{cascade_layers} = cascade({self._source_layer}, {self._target_layer}, types={cascade_types_list}, buffers={self._arguments})"
        elif self._command_type == CascadeCommandType.WEIGHTS_LIST:
            res = f"{cascade_layers} = cascade({self._source_layer}, {self._target_layer}, types={cascade_types_list}, weights={self._arguments})"
        return res

    @classmethod
    def from_tokens(cls, tokens):
        source_layer = tokens.function_args[0]
        target_layer = tokens.function_args[1]
        cascade_types_list = tokens.function_args[2]["types"]
        if len(tokens.function_args) == 3:
            arguments = None
        elif len(tokens.function_args) == 4:
            arguments = tokens.function_args[3]
        else:
            raise AllocatorScriptParserException(
                f"Failed to parse {tokens.function_name} command - too many arguments to cascade command",
            )

        cascade_layers = tokens.multiple_return_vals.asList()
        return cls(
            source_layer,
            target_layer,
            cascade_types_list,
            arguments,
            cascade_layers,
        )

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CASCADE

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("cascade")

        source_layer = command_pb.data.cascade.src
        target_layer = command_pb.data.cascade.dst
        cascade_layers = list(command_pb.result)

        pb_wrapper = PbWrapper()
        if command_pb.data.cascade.command_type == pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_BUFFERS_LIST:
            arguments = command_pb.data.cascade.buffers_list
        elif command_pb.data.cascade.command_type == pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_WEIGHTS_LIST:
            arguments = command_pb.data.cascade.weights_list
        elif (
            command_pb.data.cascade.command_type == pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_TOTAL_BUFFERS
        ):
            arguments = command_pb.data.cascade.total_buffers
        elif (
            command_pb.data.cascade.command_type == pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_EQUAL_WEIGHTS
        ):
            arguments = None

        node_type_switcher = {
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_SHORTCUT: CascadeNodeType.SHORTCUT.value,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_PORTAL3: CascadeNodeType.PORTAL3.value,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_PORTAL4: CascadeNodeType.PORTAL4.value,
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_DDR: CascadeNodeType.DDR.value,
        }
        cascade_types_list = []
        for cur_type in command_pb.data.cascade.node_types:
            cascade_types_list.append(node_type_switcher[cur_type])

        return cls(source_layer, target_layer, cascade_types_list, arguments, cascade_layers)

    @property
    def command_type(self):
        return self._command_type

    @property
    def source_layer(self):
        return self._source_layer

    @property
    def target_layer(self):
        return self._target_layer

    @property
    def cascade_types_list(self):
        return self._cascade_types_list

    @property
    def arguments(self):
        return self._arguments

    @property
    def cascade_layers(self):
        return self._cascade_layers

    @property
    def group(self):
        return CommandsGroups.NETWORK

    @property
    def num_of_cascade_layers(self):
        return len(self._cascade_types_list)

    def get_layers(self):
        return [self.source_layer, self.target_layer]

    def get_unfound_layers(self, layers_scope_from_hn):
        return [layer for layer in self.get_layers() if layer not in layers_scope_from_hn]

    def validate_command(self, layers_scope_from_hn):
        if self.source_layer is None:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Cannot find layer {self.source_layer} in existing layers scope.",
            )
        if self.target_layer is None:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"Cannot find layer {self.target_layer} in existing layers scope.",
            )
        if self.has_unfound_layers(layers_scope_from_hn):
            raise AllocatorScriptParserException(
                self.msg_prefix
                + f"Cannot find layer {self.get_unfound_layers(layers_scope_from_hn)} in existing layers scope.",
            )

        if len(self.cascade_types_list) != len(self.cascade_layers):
            raise AllocatorScriptParserException(
                self.msg_prefix + "len(cascade_layer_types) should be equal to len(output_layers)",
            )

        if self._command_type in [CascadeCommandType.BUFFERS_LIST, CascadeCommandType.WEIGHTS_LIST]:
            if len(self.cascade_layers) != len(self.arguments) - 1:
                raise AllocatorScriptParserException(
                    self.msg_prefix + "len(cascade_layer_types) shoule be equal to len(arguments)-1",
                )

        if any(x not in CascadeNodeType.list() for x in self.cascade_types_list):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"one or more types from types_list: {self.cascade_types_list} are invalid",
            )

        if any(x in layers_scope_from_hn for x in self.cascade_layers):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"one or more layer from cascade_layers: {self.cascade_layers} are already in "
                "layers scope",
            )
        self.validate_single_scope_in_command("Too many scopes in cascade")
        default_logger().debug(
            f"Parsed Cascade command, args=({self.source_layer}, {self.target_layer}, {self.cascade_types_list}, {self.arguments}), return_vals={self.cascade_layers}",
        )

    def to_pb(self, pb_wrapper):
        cascade_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        cascade_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_CASCADE
        cascade_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CASCADE
        cascade_cmd_msg.data.cascade.src = self.source_layer
        cascade_cmd_msg.data.cascade.dst = self.target_layer
        cascade_cmd_msg.data.cascade.num_of_cascade_layers = self.num_of_cascade_layers

        if self.command_type == CascadeCommandType.BUFFERS_LIST:
            cascade_cmd_msg.data.cascade.command_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_BUFFERS_LIST
            )
            cascade_cmd_msg.data.cascade.buffers_list.extend(self.arguments)
        elif self.command_type == CascadeCommandType.WEIGHTS_LIST:
            cascade_cmd_msg.data.cascade.command_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_WEIGHTS_LIST
            )
            cascade_cmd_msg.data.cascade.weights_list.extend(self.arguments)
        elif self.command_type == CascadeCommandType.TOTAl_BUFFERS:
            cascade_cmd_msg.data.cascade.command_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_TOTAL_BUFFERS
            )
            cascade_cmd_msg.data.cascade.total_buffers = self.arguments
        elif self.command_type == CascadeCommandType.EQUAL_WEIGHTS:
            cascade_cmd_msg.data.cascade.command_type = (
                pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_EQUAL_WEIGHTS
            )

        node_type_switcher = {
            CascadeNodeType.SHORTCUT.value: pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_SHORTCUT,
            CascadeNodeType.PORTAL3.value: pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_PORTAL3,
            CascadeNodeType.PORTAL4.value: pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_PORTAL4,
            CascadeNodeType.DDR.value: pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CASCADE_DDR,
        }
        for cur_type in self.cascade_types_list:
            cascade_cmd_msg.data.cascade.node_types.append(node_type_switcher[cur_type])

        cascade_cmd_msg.result.extend(self._cascade_layers)
        return cascade_cmd_msg

    def add_scope(self, scope_name, force=False):
        super().add_scope(scope_name, force=force)
        self._source_layer = self.add_scope_to_layer(scope_name, self._source_layer, force=force)
        self._target_layer = self.add_scope_to_layer(scope_name, self._target_layer, force=force)
        for i, layer in enumerate(self._cascade_layers):
            self._cascade_layers[i] = self.add_scope_to_layer(scope_name, layer, force=force)

    def remove_scope(self):
        self._source_layer = self._remove_scope(self._source_layer)
        self._target_layer = self._remove_scope(self._target_layer)
        self._cascade_layers = self._remove_scope(self._cascade_layers)


class NetworkGroupCommand(ModelScriptCommand):
    def __init__(self, network_group_name, scope_groups):
        super().__init__(SupportedCommands.NETWORK_GROUP)
        self._network_group_name = network_group_name
        self._scope_groups = scope_groups

    def __str__(self):
        scope_groups = ", ".join(f'[{", ".join(scope for scope in scope_group)}]' for scope_group in self.scope_groups)
        return f"{self.network_group_name} = network_group({scope_groups})"

    def remove_scope(self):
        pass

    @classmethod
    def from_tokens(cls, tokens):
        scope_groups = list(tokens.function_args.asList())
        return cls(tokens.single_return_val, scope_groups)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_NETWORK_GROUP

    @property
    def network_group_name(self):
        return self._network_group_name

    @property
    def scope_groups(self):
        return self._scope_groups

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        scopes_from_command = [scope for scope_group in self.scope_groups for scope in scope_group]
        if len(scopes_from_command) != len(set(scopes_from_command)):
            duplicate_scopes = [scope for scope, count in collections.Counter(scopes_from_command).items() if count > 8]
            AllocatorScriptParserException(f"{duplicate_scopes} cannot have more than 8 groups")

        scopes_from_hn = {layer.split("/", 1)[0] for layer in layers_scope_from_hn}

        invalid_scopes = set(scopes_from_command) - scopes_from_hn
        if invalid_scopes:
            raise AllocatorScriptParserException(f'{", ".join(invalid_scopes)} do not exist in the HN')

        default_logger().debug(
            f"Parsed NetworkGroupCommand command, network_group_name={self.network_group_name}, "
            f"groups=({self.scope_groups})",
        )

    def to_pb(self, pb_wrapper):
        network_group_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        network_group_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_NETWORK_GROUP
        network_group_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_NETWORK_GROUP
        network_group_cmd_msg.data.network_group.network_group_name = self.network_group_name
        for scope_group in self.scope_groups:
            proto_scope_group = pb_wrapper.integrated_hw_graph_base_pb2.ScopesGroup()
            proto_scope_group.scopes.extend(scope_group)
            network_group_cmd_msg.data.network_group.scope_groups.append(proto_scope_group)

        return network_group_cmd_msg

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        group_name = command_pb.data.network_group.network_group_name
        scope_groups = [barrier.scopes for barrier in command_pb.data.network_group.scope_groups]
        return cls(group_name, scope_groups)

    def add_scope(self, scope_names, force=False):
        pass


class MirrorCommand(ModelScriptCommand):
    def __init__(self, src_prefix, dst_prefixes):
        super(MirrorCommand, self).__init__(SupportedCommands.MIRROR)
        self._src_prefix = src_prefix
        self._dst_prefixes = dst_prefixes

    def __str__(self):
        prefixes = f'[{", ".join(self._dst_prefixes)}]'
        return f"mirror({self._src_prefix}, {prefixes})"

    def remove_scope(self):
        pass

    @classmethod
    def from_tokens(cls, tokens):
        src = tokens.function_args.asList()[0]
        dsts = tokens.function_args.asList()[1]
        return cls(src, dsts)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_MIRROR

    @property
    def src_prefix(self):
        return self._src_prefix

    @property
    def dest_prefixes(self):
        return self._dst_prefixes

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        prefixes_from_command = [self.src_prefix] + self.dest_prefixes
        if len(prefixes_from_command) != len(set(prefixes_from_command)):
            duplicate_prefixes = [
                prefix for prefix, count in collections.Counter(prefixes_from_command).items() if count > 1
            ]
            AllocatorScriptParserException(f"{duplicate_prefixes} cannot have repeated scopes in same sommand")

        scopes_from_hn = set(layer.split("/", 1)[0] for layer in layers_scope_from_hn)
        scopes_from_prefixes = set(prefix.split("/", 1)[0] for prefix in prefixes_from_command)

        invalid_scopes = scopes_from_prefixes - scopes_from_hn
        if invalid_scopes:
            raise AllocatorScriptParserException(f'{", ".join(invalid_scopes)} do not exist in the HN')

        default_logger().debug(
            f"Parsed MirrorCommand command, src_prefix={self.src_prefix}, dest_prefixes=({self.dest_prefixes})"
        )

    def to_pb(self, pb_wrapper):
        mirror_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        mirror_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_MIRROR
        mirror_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_MIRROR
        mirror_cmd_msg.data.mirror.src_prefix = self.src_prefix
        for dest_scope in self.dest_prefixes:
            mirror_cmd_msg.data.mirror.dst_prefixes.append(dest_scope)

        return mirror_cmd_msg

    @classmethod
    def from_pb(cls, command_pb):
        super(MirrorCommand, cls).validate_data_type(command_pb)
        src_prefix = command_pb.data.mirror.src_prefix
        dst_prefixes = command_pb.data.mirror.dst_prefixes
        return cls(src_prefix, dst_prefixes)

    def add_scope(self, scope_names, force=False):
        pass


class ShareConfigCommand(ModelScriptCommand):
    def __init__(self, src_node, dst_nodes):
        super(ShareConfigCommand, self).__init__(SupportedCommands.SHARE_CONFIG)
        self._src_node = src_node
        self._dst_nodes = dst_nodes

    def __str__(self):
        nodes = f'[{", ".join(self._dst_nodes)}]'
        return f"share_config({self._src_node}, {nodes})"

    def remove_scope(self):
        pass

    @classmethod
    def from_tokens(cls, tokens):
        src = tokens.function_args.asList()[0]
        dsts = tokens.function_args.asList()[1]
        return cls(src, dsts)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_SHARE_CONFIG

    @property
    def src_node(self):
        return self._src_node

    @property
    def dest_nodes(self):
        return self._dst_nodes

    def get_layers(self):
        return [self._src_node] + self._dst_nodes

    def _all_layers(self):
        return self.get_layers()

    def _replace_all_layers(self, new_values):
        self._src_node = new_values[0]
        self._dst_nodes = new_values[1:]

    def validate_command(self, layers_scope_from_hn):
        nodes_from_command = [self.src_node] + self.dest_nodes
        if len(nodes_from_command) != len(set(nodes_from_command)):
            duplicate_prefixes = [
                prefix for prefix, count in collections.Counter(nodes_from_command).items() if count > 1
            ]
            AllocatorScriptParserException(f"{duplicate_prefixes} cannot have repeated scopes in same sommand")

        scopes_from_hn = set(layer.split("/", 1)[0] for layer in layers_scope_from_hn)
        scopes_from_nodes = set(prefix.split("/", 1)[0] for prefix in nodes_from_command)

        invalid_scopes = scopes_from_nodes - scopes_from_hn
        if invalid_scopes:
            raise AllocatorScriptParserException(f'{", ".join(invalid_scopes)} do not exist in the HN')

        default_logger().debug(
            f"Parsed ShareCOnfigCommand command, src_node={self.src_node}, dest_nodes=({self.dest_nodes})"
        )

    def to_pb(self, pb_wrapper):
        share_config_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        share_config_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_SHARE_CONFIG
        share_config_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_SHARE_CONFIG
        share_config_cmd_msg.data.share_config.src_node = self.src_node
        for dest_node in self.dest_nodes:
            share_config_cmd_msg.data.share_config.dst_nodes.append(dest_node)

        return share_config_cmd_msg

    @classmethod
    def from_pb(cls, command_pb):
        super(ShareConfigCommand, cls).validate_data_type(command_pb)
        src_node = command_pb.data.share_config.src_node
        dst_nodes = command_pb.data.share_config.dst_nodes
        return cls(src_node, dst_nodes)

    def add_scope(self, scope_names, force=False):
        pass


class PlatformParamCommand(ModelScriptCommand):
    def __init__(self, **platform_params):
        super().__init__(SupportedCommands.PLATFORM_PARAM)
        self._platform_params = PlatformParams()
        self._platform_params.clear()
        self._platform_params.set(platform_params)

    def __str__(self):
        return self._export_params_to_string(False)

    def str_to_alls(self):
        return self._export_params_to_string(True)

    def _export_params_to_string(self, is_to_auto_alls):
        params = ", ".join(
            [
                param_str(param_k, param_v)
                for param_k, param_v in sorted(self.platform_params.items())
                if not (param_k in self._platform_params.COMMENTED_PARAMS and is_to_auto_alls)
            ],
        )
        commented_params = ", ".join(
            [
                param_str(param_k, param_v)
                for param_k, param_v in sorted(self.platform_params.items())
                if (param_k in self._platform_params.COMMENTED_PARAMS and is_to_auto_alls)
            ],
        )
        command_template = "platform_param({})"
        base_command = ""
        if params:
            base_command = command_template.format(params)
        if commented_params:
            base_command += "\n# " + command_template.format(commented_params)
        return base_command

    def remove_scope(self):
        pass

    @classmethod
    def from_tokens(cls, tokens):
        platform_params = {}
        for param in tokens.function_args:
            platform_params.update(param)
        return cls(**platform_params)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_PLATFORM_PARAM

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("platform_params")
        params = PlatformParams()
        params.from_pb(command_pb.data.platform_params, PbWrapper())
        platform_params = convert_params_to_str(params.get())
        return cls(**platform_params)

    @property
    def platform_params(self):
        return self._platform_params.get()

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        validate_params_command(self.platform_params, PlatformParams.DEFAULT_PARAMS.keys(), self.msg_prefix)
        if PlatformParams.HEF_VERSION_STR in self.platform_params:
            hef_version = self._platform_params.get(PlatformParams.HEF_VERSION_STR)
            current_hef_version = LATEST_HEF_VERSION
            if hef_version > current_hef_version or hef_version <= 0:
                AllocatorScriptParserException(f"Got invalid HEF version <{hef_version}>")

        default_logger().debug(f"Parsed PlatformParam command, args=({self.platform_params})")

    def to_pb(self, pb_wrapper: PbWrapper):
        platform_param_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        platform_param_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_PLATFORM_PARAM
        platform_param_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_PLATFORM_PARAM
        self._platform_params.to_pb(platform_param_cmd_msg.data.platform_params, pb_wrapper)
        return platform_param_cmd_msg


class PerformanceParamCommand(ModelScriptCommand):
    def __init__(
        self,
        components,
        fps,
        is_fps_set,
        optimization_level,
        optimize_for_batch,
        optimize_for_power,
        low_pcie_bandwidth_value=None,
    ):
        super().__init__(SupportedCommands.PERFORMANCE_PARAM)
        self._components = components
        self._fps = fps
        self._is_fps_set = is_fps_set
        self._optimization_level = optimization_level
        self._optimize_for_batch = optimize_for_batch
        self._optimize_for_power = optimize_for_power
        if low_pcie_bandwidth_value is None:
            self._low_pcie_bandwidth_value = AutoDouble("automatic")
        else:
            self._low_pcie_bandwidth_value = convert_to_auto_double(low_pcie_bandwidth_value)

    def __str__(self):
        fps = self._fps if self._is_fps_set else None
        if self._optimization_level is None:
            optimization_level = DEFAULT_OPTIMIZATION_LEVEL
        else:
            optimization_level = int(self._optimization_level.value)

        optimize_for_batch_str = f", optimize_for_batch={self._optimize_for_batch}" if self._optimize_for_batch else ""
        optimize_for_power_str = f", optimize_for_power={self._optimize_for_power}" if self._optimize_for_power else ""
        if self._low_pcie_bandwidth_value.policy() == AutoVariablePolicy.MANUAL:
            low_pcie_bandwidth_value_str = f", low_pcie_bandwidth_value={self._low_pcie_bandwidth_value.val()}"
        else:
            low_pcie_bandwidth_value_str = ""

        if len(self._components) == 0:
            if fps is None:
                return f"performance_param(compiler_optimization_level={optimization_level}{optimize_for_batch_str}{optimize_for_power_str}{low_pcie_bandwidth_value_str})"
            else:
                return f"performance_param(fps={fps})\nperformance_param(compiler_optimization_level={optimization_level}{optimize_for_batch_str}{optimize_for_power_str}{low_pcie_bandwidth_value_str})"
        layers = ",".join(self._components)
        if len(self._components) > 1:
            layers = f"[{layers}]"
        elif "*" in layers:
            layers = f"{{{layers}}}"
        return f"performance_param({layers}, fps={fps})\nperformance_param(compiler_optimization_level={optimization_level}{optimize_for_batch_str}{optimize_for_power_str}{low_pcie_bandwidth_value_str})"

    def remove_scope(self):
        pass

    @classmethod
    def from_tokens(cls, tokens):
        fps = None
        optimization_level = DEFAULT_OPTIMIZATION_LEVEL
        layers = []
        args = tokens.function_args.asList()
        optimize_for_batch = 0
        optimize_for_power = False
        low_pcie_bandwidth_value = AutoDouble("automatic")
        if not isinstance(args[0], dict):
            layers = args[0]
            fps = args[1]["fps"]
            is_fps_set = True
            if isinstance(layers, str):
                layers = [layers]
        else:
            for param in args:
                param_key = next(iter(param))
                if param_key == "fps":
                    fps = param["fps"]
                elif param_key in ("optimization_level", "compiler_optimization_level"):
                    if param_key == "optimization_level":
                        # TODO: https://hailotech.atlassian.net/browse/SDK-40693
                        default_logger().deprecation_warning(
                            "Performance parameter key 'optimization_level' is deprecated, please use "
                            "'compiler_optimization_level' instead.",
                            DeprecationVersion.FUTURE,
                        )
                    if param[param_key] == "max":
                        optimization_level = get_max_compiler_optimization_level()
                        default_logger().info(
                            f"ParsedPerformanceParam command, setting optimization_level(max={optimization_level})",
                        )
                    else:
                        optimization_level = int(param[param_key])
                elif param_key == "optimize_for_batch":
                    optimize_for_batch = int(param[param_key])
                elif param_key == "optimize_for_power":
                    optimize_for_power = bool(param[param_key])
                elif param_key == "low_pcie_bandwidth_value":
                    low_pcie_bandwidth_value = convert_to_auto_double(param[param_key])
                else:
                    raise AllocatorScriptParserException(f"{next(iter(param))} is not a legal performance parameter.")
            is_fps_set = not (fps is None or fps == "None")

        return cls(
            layers,
            fps,
            is_fps_set,
            OptimizationLevel(optimization_level),
            optimize_for_batch,
            optimize_for_power,
            low_pcie_bandwidth_value,
        )

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_PERFORMANCE_PARAM

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        components = command_pb.data.performance_param.components
        fps = command_pb.data.performance_param.fps
        is_fps_set = command_pb.data.performance_param.is_fps_set
        params = PerformanceParams()
        params.from_pb(command_pb.data.performance_param, PbWrapper())
        performance_params = params.get()
        optimization_level = performance_params["optimization_level"]
        optimize_for_batch = performance_params.get("optimize_for_batch", 0)
        optimize_for_power = performance_params.get("optimize_for_power", False)
        pb_wrapper = PbWrapper()
        if (
            command_pb.data.performance_param.low_pcie_bandwidth_value.policy
            == pb_wrapper.integrated_hw_graph_base_pb2.PROTO_AUTOMATIC
        ):
            low_pcie_bandwidth_value = AutoDouble("automatic")
        else:
            low_pcie_bandwidth_value = convert_to_auto_double(
                command_pb.data.performance_param.low_pcie_bandwidth_value.val
            )
        return cls(
            components,
            fps,
            is_fps_set,
            OptimizationLevel(optimization_level),
            optimize_for_batch,
            optimize_for_power,
            low_pcie_bandwidth_value,
        )

    @property
    def components(self):
        return self._components

    @property
    def fps(self):
        return self._fps

    @property
    def is_fps_set(self):
        return self._is_fps_set

    @property
    def optimization_level(self):
        return self._optimization_level

    @property
    def optimize_for_batch(self):
        return self._optimize_for_batch

    @property
    def optimize_for_power(self):
        return self._optimize_for_power

    @property
    def low_pcie_bandwidth_value(self):
        return self._low_pcie_bandwidth_value

    @property
    def group(self):
        return CommandsGroups.GLOBAL

    def check_layer(self, layer, layers_with_scopes):
        if "/" in layer:
            return layer in layers_with_scopes
        else:
            layers_without_scopes = {layer.split("/", 1)[1] for layer in layers_with_scopes}
            return layer in layers_without_scopes

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        # NOTE: valid context is validated in cpp
        if len(self._components) == 0:
            return
        if len(self._components) == 1:
            layer = self._components[0]
            if "*" in layer:
                scopes_from_hn = {layer.split("/", 1)[0] for layer in layers_scope_from_hn}
                if layer not in scopes_from_hn:
                    raise AllocatorScriptParserException(f"{self._components} do not exist in the HN")
            elif not self.check_layer(layer, layers_scope_from_hn):
                raise AllocatorScriptParserException(f"{self._components} do not exist in the HN")
        else:
            for layer in self._components:
                if not self.check_layer(layer, layers_scope_from_hn):
                    raise AllocatorScriptParserException(f"{layer} do not exist in the HN")

    def to_pb(self, pb_wrapper):
        performance_param_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        performance_param_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_PERFORMANCE_PARAM
        performance_param_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_PERFORMANCE_PARAM
        performance_param_cmd_msg.data.performance_param.components.extend(self._components)
        performance_param_cmd_msg.data.performance_param.is_fps_set = self._is_fps_set
        performance_param_cmd_msg.data.performance_param.optimization_level = self._optimization_level.value
        performance_param_cmd_msg.data.performance_param.optimize_for_batch = self._optimize_for_batch
        performance_param_cmd_msg.data.performance_param.optimize_for_power = self._optimize_for_power
        performance_param_cmd_msg.data.performance_param.low_pcie_bandwidth_value.policy = (
            pb_wrapper.AUTO_VARIABLE_POLICY_TO_PB[self.low_pcie_bandwidth_value.policy()]
        )
        performance_param_cmd_msg.data.performance_param.low_pcie_bandwidth_value.val = (
            self.low_pcie_bandwidth_value.val()
        )
        if self._is_fps_set:
            performance_param_cmd_msg.data.performance_param.fps = self._fps
        return performance_param_cmd_msg


class ContextPerformanceParamCommand(ModelScriptCommand):
    def __init__(self, context, fps):
        super().__init__(SupportedCommands.CONTEXT_PERFORMANCE_PARAM)
        self._context = context
        self._fps = fps

    def __str__(self):
        return f"{self._context}.performance_param(fps={self._fps})"

    def remove_scope(self):
        pass

    @classmethod
    def from_tokens(cls, tokens):
        params = {}
        for param in tokens.function_args.asList():
            params.update(param)
        return cls(tokens.object, **params)

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CONTEXT_PERFORMANCE_PARAM

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.context_performance_param.HasField("fps")
        context = command_pb.data.context_performance_param.name
        fps = command_pb.data.context_performance_param.fps
        return cls(context, fps)

    @property
    def context(self):
        return self._context

    @property
    def fps(self):
        return self._fps

    @property
    def group(self):
        return CommandsGroups.CONTEXT

    def get_layers(self):
        return []

    def validate_command(self, layers_scope_from_hn):
        # NOTE: valid context is validated in cpp
        if self._fps < 0:
            raise AllocatorScriptParserException("Got invalid FPS value")

    def to_pb(self, pb_wrapper):
        context_performance_param_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        context_performance_param_cmd_msg.op = (
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_CONTEXT_PERFORMANCE_PARAM
        )
        context_performance_param_cmd_msg.data.type = (
            pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CONTEXT_PERFORMANCE_PARAM
        )
        context_performance_param_cmd_msg.data.context_performance_param.name = self._context
        context_performance_param_cmd_msg.data.context_performance_param.fps = self._fps
        return context_performance_param_cmd_msg


class FormatConversionCommand(ModelScriptCommandWithScope):
    NV_CONVERSIONS = ["nv12_to_hailo_yuv", "nv21_to_hailo_yuv", "i420_to_hailo_yuv"]

    def __init__(
        self,
        layer_name,
        source_layer,
        dest_layers,
        conversion_type,
        first_new_dim,
        second_new_dim,
        collapse=False,
        input_windows=[],
        output_windows=[],
    ):
        super().__init__(SupportedCommands.FORMAT_CONVERSION, function_return_vals=list(layer_name))
        self._layer_name = layer_name
        self._source_layer = source_layer
        self._dest_layers = dest_layers
        self._conversion_type = conversion_type
        self._first_new_dim = first_new_dim
        self._second_new_dim = second_new_dim
        self._collapse = collapse
        self._input_windows = input_windows
        self._output_windows = output_windows

    def _all_layers(self):
        return [*self.layer_name, self.source_layer, *self.dest_layers]

    def _replace_all_layers(self, new_values):
        count = len(self.layer_name)
        self._layer_name = new_values[:count]
        self._source_layer = new_values[count]
        self._dest_layers = new_values[count + 1 :]

    def prefix_in_command(self, prefix):
        # NOTE: for some reason return value is duplicated in the command
        # which corrupts regular logic of this flow
        return is_prefix_in_layers_list_relaxed(prefix, self._all_layers())

    def remove_scope(self):
        self._layer_name = self._remove_scope(self._layer_name)
        self._source_layer = self._remove_scope(self._source_layer)
        self._dest_layers = self._remove_scope(self._dest_layers)

    @property
    def layer_name(self):
        return self._layer_name

    @property
    def source_layer(self):
        return self._source_layer

    @property
    def conversion_type(self):
        return self._conversion_type

    @property
    def dest_layers(self):
        return self._dest_layers

    @property
    def first_new_dim(self):
        return self._first_new_dim

    @property
    def second_new_dim(self):
        return self._second_new_dim

    @property
    def collapse(self):
        return self._collapse

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_FORMAT_CONVERSION

    @classmethod
    def from_tokens(cls, tokens):
        if len(tokens.function_args) == 1:
            raise AllocatorScriptParserException(
                "Failed to parse format_conversion command - command must include "
                "the source layer and conversion type.",
            )

        first_new_dim, second_new_dim = 0, 0
        collapse = False

        index = len(tokens.function_args) - 1
        flags_found = 0
        while index > 0 and isinstance(tokens.function_args[index], dict):
            flags_found += 1
            if "collapse" in tokens.function_args[index]:
                collapse = tokens.function_args[index]["collapse"] in ["True", "true", "enabled", "Enabled"]
            index -= 1
        args = tokens.function_args[:-flags_found] if flags_found else tokens.function_args

        if isinstance(args[-1], float):
            if (not isinstance(args[-2], float)) or args[-3] not in [
                "spatial_reshape",
                "hxf_to_w_transposed",
                "f_to_hxw_transposed",
                "reshape_height_features",
            ]:
                raise AllocatorScriptParserException(
                    "Failed to parse format_conversion command - command must include "
                    "2 dimensions for spatial_reshape and reshape_height_features, or 0 for the rest.",
                )

            first_new_dim = int(args[-2])
            second_new_dim = int(args[-1])
            conversion_type = args[-3]
            dst_layers = args[1:-3]

        else:
            if args[-1] == "spatial_reshape" or args[-1] == "reshape_height_features":
                raise AllocatorScriptParserException(
                    "Failed to parse format_conversion command - command must include "
                    "2 dimensions for spatial_reshape and reshape_height_features, or 0 for the rest.",
                )
            conversion_type = args[-1]
            dst_layers = args[1:-1]

        input_windows = []
        output_windows = []
        for value in tokens.function_args:
            if isinstance(value, dict):
                if "input_windows" in value:
                    input_windows = [int(v) for v in value["input_windows"]]
                if "output_windows" in value:
                    output_windows = [int(v) for v in value["output_windows"]]

        return cls(
            tokens.multiple_return_vals.asList(),
            tokens.function_args[0],
            dst_layers,
            FormatConversionType(conversion_type),
            first_new_dim,
            second_new_dim,
            collapse,
            input_windows,
            output_windows,
        )

    def __str__(self):
        command = f'{",".join(self.layer_name)} = format_conversion({self.source_layer}, '
        for dest_layer in self.dest_layers:
            command += dest_layer + ", "
        command += f"{self.conversion_type.value}"
        if (
            (self.first_new_dim, self.second_new_dim) != (0, 0)
            and (
                self.conversion_type == FormatConversionType.spatial_reshape
                or self.conversion_type == FormatConversionType.hxf_to_w_transposed
                or self.conversion_type == FormatConversionType.f_to_hxw_transposed
                or self.conversion_type == FormatConversionType.reshape_height_features
            )
        ) or self.collapse:
            if (self.first_new_dim, self.second_new_dim) != (0, 0):
                command += f", {self.first_new_dim!s}, {self.second_new_dim!s}"
            command += f", collapse={self.collapse!s}"

        if self._input_windows:
            command += f", input_windows={self._input_windows}"
        if self._output_windows:
            command += f", output_windows={self._output_windows}"
        command += ")"
        return command

    @property
    def group(self):
        return CommandsGroups.NETWORK

    def get_layers(self):
        return [self.source_layer, *self.dest_layers]

    def get_unfound_layers(self, layers_scope_from_hn):
        return [layer for layer in self.get_layers() if layer not in layers_scope_from_hn]

    def validate_command(self, layers_scope_from_hn):
        if self.layer_name is None or self.layer_name in layers_scope_from_hn:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"new layer {self.layer_name} should be a valid layer name, not existing in current "
                f"layers scope.",
            )
        if self.source_layer is None:
            raise AllocatorScriptParserException(
                self.msg_prefix + f"source layer {self.source_layer} not in layers scope.",
            )
        if self.has_unfound_layers(layers_scope_from_hn):
            raise AllocatorScriptParserException(
                self.msg_prefix
                + f"Cannot find layer {self.get_unfound_layers(layers_scope_from_hn)} in existing layers scope.",
            )
        if len(self.function_return_vals) > 1 and (self.conversion_type.value not in self.NV_CONVERSIONS):
            raise AllocatorScriptParserException(
                self.msg_prefix + "multiple return values given to single return value "
                "conversion type. See usage guidelines at: "
                "Dataflow Compiler User Guide / Building Models / Model Optimization / Model Scripts",
            )
        if len(self.function_return_vals) > 1 and (self.conversion_type.value in self.NV_CONVERSIONS):
            raise AllocatorScriptParserException(
                self.msg_prefix + f"{len(self.function_return_vals)} return values given to nv_conversion method "
                f"while 1 expected. See usage guidelines at: Dataflow Compiler User Guide / Building Models / Model Optimization / Model Scripts",
            )
        default_logger().debug(
            f"Parsed {self.function_name} command, args=({self.source_layer}), return_vals={self.layer_name}",
        )

    def to_pb(self, pb_wrapper):
        format_conversion_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        format_conversion_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_FORMAT_CONVERSION
        format_conversion_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_FORMAT_CONVERSION
        format_conversion_cmd_msg.data.format_conversion.layer_name.extend(self.layer_name)
        format_conversion_cmd_msg.data.format_conversion.source_layer = self.source_layer
        format_conversion_cmd_msg.data.format_conversion.dest_layers.extend(self.dest_layers)
        format_conversion_cmd_msg.data.format_conversion.conversion_type = pb_wrapper.CONVERSION_TYPE_TO_PB[
            self.conversion_type
        ]
        format_conversion_cmd_msg.data.format_conversion.first_new_dim = self.first_new_dim
        format_conversion_cmd_msg.data.format_conversion.second_new_dim = self.second_new_dim
        format_conversion_cmd_msg.data.format_conversion.collapse = self.collapse

        format_conversion_cmd_msg.data.format_conversion.input_windows.extend(self._input_windows)
        format_conversion_cmd_msg.data.format_conversion.output_windows.extend(self._output_windows)

        return format_conversion_cmd_msg

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("format_conversion")
        layer_name = command_pb.data.format_conversion.layer_name
        source_layer = command_pb.data.format_conversion.source_layer
        conversion_type = PbWrapper().CONVERSION_PB_TO_TYPE[command_pb.data.format_conversion.conversion_type]
        dest_layers = command_pb.data.format_conversion.dest_layers
        if conversion_type not in [FormatConversionType.spatial_reshape, FormatConversionType.reshape_height_features]:
            first_new_dim = 0
            second_new_dim = 0
        else:
            first_new_dim = command_pb.data.format_conversion.first_new_dim
            second_new_dim = command_pb.data.format_conversion.second_new_dim
        collapse = command_pb.data.format_conversion.collapse
        input_windows = command_pb.data.format_conversion.input_windows
        output_windows = command_pb.data.format_conversion.output_windows
        return cls(
            layer_name,
            source_layer,
            dest_layers,
            conversion_type,
            first_new_dim,
            second_new_dim,
            collapse,
            input_windows,
            output_windows,
        )

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        for i, layer in enumerate(self._layer_name):
            self._layer_name[i] = self.add_scope_to_layer(scope_names, layer, force=force)
        self._source_layer = self.add_scope_to_layer(scope_names, self._source_layer, force=force)
        for i, layer in enumerate(self._dest_layers):
            self._dest_layers[i] = self.add_scope_to_layer(scope_names, layer, force=force)


class RemoveNodeCommand(ModelScriptCommandWithScope):
    def __init__(self, layer_name):
        super().__init__(SupportedCommands.REMOVE_NODE)
        self._layer_name = layer_name

    def _all_layers(self):
        return self.layer_name

    def _replace_all_layers(self, new_values):
        self._layer_name = new_values

    def remove_scope(self):
        self._layer_name = self._remove_scope(self._layer_name)

    @property
    def layer_name(self):
        return self._layer_name

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_REMOVE_NODE

    @classmethod
    def from_tokens(cls, tokens):
        param = tokens.function_args.asList()
        return cls(param)

    def __str__(self):
        return f"remove_node({self._layer_name[0]})"

    @property
    def group(self):
        return CommandsGroups.NETWORK

    def get_layers(self):
        return self._layer_name

    def get_unfound_layers(self, layers_scope_from_hn):
        return [layer for layer in self.get_layers() if layer not in layers_scope_from_hn]

    def validate_command(self, layers_scope_from_hn):
        for layer in self.layer_name:
            if layer is None or layer not in layers_scope_from_hn:
                raise AllocatorScriptParserException(
                    self.msg_prefix + f"layer {layer} should be a valid layer name,",
                )

    def to_pb(self, pb_wrapper):
        remove_node_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        remove_node_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_REMOVE_NODE
        remove_node_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_REMOVE_NODE
        remove_node_cmd_msg.data.remove_node.layer_name = self.layer_name[0]
        return remove_node_cmd_msg

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("remove_node")
        layer_name = command_pb.data.remove_node.layer_name
        return cls([layer_name])

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        for i, layer in enumerate(self._layer_name):
            self._layer_name[i] = self.add_scope_to_layer(scope_names, layer, force=force)


class ConvertToDenseCommand(ModelScriptCommandWithScope):
    def __init__(self, layer_name):
        super().__init__(SupportedCommands.CONVERT_TO_DENSE)
        self._layer_name = layer_name

    def _all_layers(self):
        return self.layer_name

    def _replace_all_layers(self, new_values):
        self._layer_name = new_values

    def remove_scope(self):
        self._layer_name = self._remove_scope(self._layer_name)

    @property
    def layer_name(self):
        return self._layer_name

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CONVERT_TO_DENSE

    @classmethod
    def from_tokens(cls, tokens):
        param = tokens.function_args.asList()
        return cls(param)

    def __str__(self):
        return f"convert_to_dense({self._layer_name[0]})"

    @property
    def group(self):
        return CommandsGroups.NETWORK

    def get_layers(self):
        return self._layer_name

    def get_unfound_layers(self, layers_scope_from_hn):
        return [layer for layer in self.get_layers() if layer not in layers_scope_from_hn]

    def validate_command(self, layers_scope_from_hn):
        for layer in self.layer_name:
            if layer is None or layer not in layers_scope_from_hn:
                raise AllocatorScriptParserException(
                    self.msg_prefix + f"layer {layer} should be a valid layer name,",
                )

    def to_pb(self, pb_wrapper):
        convert_to_dense_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        convert_to_dense_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_CONVERT_TO_DENSE
        convert_to_dense_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_CONVERT_TO_DENSE
        convert_to_dense_cmd_msg.data.convert_to_dense.layer_name = self.layer_name[0]
        return convert_to_dense_cmd_msg

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("convert_to_dense")
        layer_name = command_pb.data.convert_to_dense.layer_name
        return cls([layer_name])

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        for i, layer in enumerate(self._layer_name):
            self._layer_name[i] = self.add_scope_to_layer(scope_names, layer, force=force)


class BucketCommand(ModelScriptCommandWithScope):
    def __init__(self, name, layers):
        super().__init__(SupportedCommands.BUCKET)
        self._layers = layers
        self._bucket_name = name

    def remove_scope(self):
        self._layers = self._remove_scope(self._layers)

    def _all_layers(self):
        return self._layers

    def _replace_all_layers(self, new_values):
        self._layers = new_values[len(self.layers)]

    @property
    def layers(self):
        return self._layers

    @property
    def bucket_name(self):
        return self._bucket_name

    @classmethod
    def data_type(cls):
        return PbWrapper().integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_BUCKET

    @classmethod
    def from_tokens(cls, tokens):
        name = tokens.single_return_val
        layers = tokens.function_args.asList()[0]
        return cls(name, layers)

    def __str__(self):
        layers = ", ".join(self._layers)
        msg = f"{self._bucket_name} = bucket([{layers}])"
        return msg

    @property
    def group(self):
        return CommandsGroups.NETWORK

    def get_layers(self):
        return self._layers

    def validate_command(self, layers_scope_from_hn):
        for layer in self.get_layers():
            if layer is None or layer not in layers_scope_from_hn:
                raise AllocatorScriptParserException(
                    self.msg_prefix + f"layer {layer} should be a valid layer name, to be placed in a bucket",
                )

    def to_pb(self, pb_wrapper):
        bucket_cmd_msg = pb_wrapper.integrated_hw_graph_base_pb2.ProtoAllocatorCommand()
        bucket_cmd_msg.op = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_CMD_BUCKET
        bucket_cmd_msg.data.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_DATA_TYPE_BUCKET
        bucket_cmd_msg.data.bucket.name = self._bucket_name
        bucket_cmd_msg.data.bucket.layers.extend(self._layers)
        return bucket_cmd_msg

    @classmethod
    def from_pb(cls, command_pb):
        super().validate_data_type(command_pb)
        assert command_pb.data.HasField("bucket")
        name = command_pb.data.bucket.name
        layers = command_pb.data.layers
        return cls(name, layers)

    def add_scope(self, scope_names, force=False):
        super().add_scope(scope_names, force=force)
        for j, layer in enumerate(self._layers):
            self._layers[j] = self.add_scope_to_layer(scope_names, layer, force=force)
