import json
import math
import os
import re

from hailo_model_optimization.acceleras.utils.acceleras_definitions import (
    DEFAULT_BOX_AND_OBJ_PXLS,
    NMS_ARGUMENTS_ORDER,
    BBoxDecodersInfo,
    NMSProperties,
    PostprocessTarget,
    ProtoInfo,
)
from hailo_sdk_client.sdk_backend.modification_config import NMSConfig
from hailo_sdk_client.sdk_backend.script_parser.commands import SupportedCommands
from hailo_sdk_client.sdk_backend.script_parser.model_modifications_commands import ModelModificationsCommand
from hailo_sdk_client.sdk_backend.sdk_backend_exceptions import AllocatorScriptParserException
from hailo_sdk_client.tools.core_postprocess.nms_postprocess import (
    NMSConfigPostprocessException,
    NMSMetaData,
    UnsupportedMetaArchError,
    create_nms_postprocess,
)
from hailo_sdk_common.hailo_nn.hn_definitions import ActivationType, LayerType, NMSMetaArchitectures
from hailo_sdk_common.hailo_nn.nms_postprocess_defaults import (
    DEFAULT_IOU_TH,
    DEFAULT_MAX_PROPOSALS_PER_CLASS,
    DEFAULT_NMS_OUTPUT_ORIGINAL_NAME,
    DEFAULT_SCORES_TH,
    DEFAULT_SSD_CLASSES,
)
from hailo_sdk_common.logger.logger import default_logger
from hailo_sdk_common.paths_manager.paths import SDKPaths

logger = default_logger()

MIN_SUFFICIENT_DIVISION_FACTOR = 5
NMS_CONFIG_FILE_FAKE_HAR_PATH = "{har}"
YOLOV5_SEG_NUMBER_OF_DETECTION_HEADS = 4
YOLOV6_FEATURES_OUT = 4
YOLOV6_TOTAL_OUTPUTS = 6
YOLOX_ACTIVATIONS_PER_REG_LAYER = 2
YOLOX_FEATURES_OUT = 4
YOLOX_TOTAL_OUTPUTS = 9

NN_CORE_META_ARCHS = [
    NMSMetaArchitectures.SSD,
    NMSMetaArchitectures.YOLOV5,
    NMSMetaArchitectures.CENTERNET,
    NMSMetaArchitectures.YOLOV6,
]
CPU_META_ARCHS = [
    NMSMetaArchitectures.YOLOV5,
    NMSMetaArchitectures.YOLOX,
    NMSMetaArchitectures.YOLOV5_SEG,
    NMSMetaArchitectures.SSD,
    NMSMetaArchitectures.YOLOV8,
    NMSMetaArchitectures.DAMOYOLO,
]
AUTO_SUPPORTED_META_ARCH = [NMSMetaArchitectures.YOLOV5]

YOLO_OUTPUTS_PER_BRANCH = {
    NMSMetaArchitectures.YOLOX: [3],
    NMSMetaArchitectures.YOLOV6: [2],
    NMSMetaArchitectures.YOLOV8: [1, 2],
    NMSMetaArchitectures.DAMOYOLO: [1, 2],
}
YOLO_META_ARCHS = [
    NMSMetaArchitectures.YOLOV5,
    NMSMetaArchitectures.YOLOV5_SEG,
    NMSMetaArchitectures.YOLOX,
    NMSMetaArchitectures.YOLOV6,
    NMSMetaArchitectures.YOLOV8,
    NMSMetaArchitectures.DAMOYOLO,
]
ANCHORLESS_YOLOS = [
    NMSMetaArchitectures.YOLOV6,
    NMSMetaArchitectures.YOLOV8,
    NMSMetaArchitectures.YOLOX,
    NMSMetaArchitectures.DAMOYOLO,
]

META_ARCH_OUTPUT_CHANNELS = {
    NMSMetaArchitectures.YOLOV5: 3,
    NMSMetaArchitectures.YOLOX: lambda number_of_detection_heads: number_of_detection_heads * 3,
    NMSMetaArchitectures.YOLOV5_SEG: YOLOV5_SEG_NUMBER_OF_DETECTION_HEADS,
}

CPU_DEFAULT_IOU_TH = {
    NMSMetaArchitectures.YOLOV6: 0.65,
    NMSMetaArchitectures.YOLOV8: 0.7,
    NMSMetaArchitectures.YOLOX: 0.65,
    NMSMetaArchitectures.DAMOYOLO: 0.7,
}

SUPPORTED_DFL_ON_NN_CORE_META_ARCHS = [NMSMetaArchitectures.YOLOV8]


def get_f_out_by_meta_arch(meta_arch, classes=80, num_anchors=None, num_proto=None, combined_layer=False, reg_len=16):
    if meta_arch == NMSMetaArchitectures.YOLOV6:
        return YOLOV6_FEATURES_OUT
    if meta_arch == NMSMetaArchitectures.YOLOX:
        return YOLOX_FEATURES_OUT
    if meta_arch in [NMSMetaArchitectures.YOLOV8, NMSMetaArchitectures.DAMOYOLO]:
        return reg_len * 4 + classes if combined_layer else reg_len * 4
    if num_anchors is not None:
        if meta_arch == NMSMetaArchitectures.YOLOV5:
            return (classes + DEFAULT_BOX_AND_OBJ_PXLS) * num_anchors
        if meta_arch == NMSMetaArchitectures.YOLOV5_SEG and num_proto is not None:
            return (classes + DEFAULT_BOX_AND_OBJ_PXLS + num_proto) * num_anchors
    raise UnsupportedMetaArchError(
        f"Could not calculate number of features out to {meta_arch}, make sure to provide " "all relevant arguments.",
    )


ARG_TO_TYPE = {
    "config_path": str,
    "meta_arch": str,
    "engine": str,
    "enforce_iou_threshold": str,
    "bbox_decoding_only": str,
    "nms_scores_th": float,
    "nms_iou_th": float,
    "max_proposals_per_class": int,
    "centers_scale_factor": float,
    "bbox_dimensions_scale_factor": float,
    "classes": int,
    "background_removal": str,
    "background_removal_index": int,
    "bbox_decoders": list,
    "input_division_factor": int,
    "regression_prediction_order": list,
    "image_dims": list,
    "dfl_on_nn_core": str,
    "output_original_name": str,
}


class NMSPostprocessCommand(ModelModificationsCommand):
    """
    Adds in-chip NMS post-processing to the given model.

    Args:
        config_json_path (str): Path to configuration file, containing parameters for the
            NMS post-process.
        meta_arch (:class:`~hailo_sdk_common.hailo_nn.hn_definitions.NMSMetaArchitectures`): Meta
            architecture for the NMS post-process. Use the Enum's value.
        engine (str): describes whether the post-process will be run on neural core or on HRT.
        enforce_iou_threshold (bool): if set to true, use iou in config json. Otherwise, use iou = 1 whether
            the division factor in config json different from 1. Defaults to True.
        bbox_decoding_only (bool): if set to True, performs bbox decoding only without NMS.

    Note:
    Basic assumptions:
        - Proposal generator layers are using hard-coded activations:
            * SSD - sigmoid
            * Centernet - relu

    Additional information about the configuration parameters used in the config JSON file:

    - ``nms_scores_th`` (float): Threshold used for filtering out candidates (confidence
        TF). Any box with score<TH is ignored.
    - ``nms_iou_th`` (float): Intersection over union overlap Threshold, used in the NMS
        iterative elimination process where potential duplicates of detected items are
        ignored. (not used in Centernet)
    - ``max_proposals_per_class`` (int): Fixed number of outputs allowed in this model,
        derived from hardware limitations.
    - ``centers_scale_factor`` (float): Values used for compensation of rescales done in the
        training phase (derived from faster_rcnn architecture). This param rescales anchors
        centers.
    - ``bbox_dimensions_scale_factor`` (float): Values used for compensation of rescales
        done in the training phase (derived from faster_rcnn architecture). This param
        rescales anchors dimensions.
    - ``classes`` (int): Number of detected classes, e.g., MobilenetV2-SSD trained on COCO
        has 91 classes.
    - ``background_removal`` (bool): Toggle background class removal from results, used in
        inference time.
    - ``background_removal_index`` (int): Index of background class for background removal,
        e.g., in MobilenetV2-SSD it is 0.
    - ``bbox_decoders`` (list): List of bbox decoders (anchors) for the NMS layer. Each
        model has its own number of boxes per anchor, and each anchor is described by
        location, dimensions, and inputs.
    - ``input_division_factor`` (int): Division factor of proposals sent to the NMS per
        class, instead of running NMS on all proposal together. Used to decrease memory usage
        for NMS (in case of allocation failure).
    - ``regression_prediction_order`` (list): Permutation that defines the order of inputs
        to each bbox decoder layer, from its respective regression layer. The default value
        is [0, 1, 2, 3], which translates to prediction order of [ty, tx, th, tw] (coordinates
        description of centroids anchors). Another common prediction order is typical for
        implementations based on Keras APIs, where the prediction order is usually
        [tx, ty, tw, th]: Use [1, 0, 3, 2] permutation in the config JSON.

        An example file for SSD NMS is supplied,
        see: ``sdk_client/hailo_sdk_client/tools/core_postprocess/nms_ssd_config_example.json``.

    """

    def __init__(self, args_dict, script_path, provided_config_file=None):
        super().__init__(SupportedCommands.NMS_POSTPROCESS)
        self._args = args_dict
        self._update_config_path_in_args_dict(script_path)
        self._config_file = provided_config_file
        self._nms_config = None

    def __str__(self):
        """
        Print nms_postprocess command args by nms_postprocess_command({config_path}, meta_arch={meta_arch}, ...)
        """
        kwargs = []
        if self.config_path is not None:
            kwargs.append(f'"{self.config_path}"')

        for key in NMS_ARGUMENTS_ORDER[1:]:
            if key in self._args:
                kwargs.append(f"{key.value}={self._args[key]}")

        return f'{self.function_name.value}({", ".join(kwargs)})'

    @property
    def meta_arch(self):
        return NMSMetaArchitectures(self._args[NMSProperties.META_ARCH])

    @meta_arch.setter
    def meta_arch(self, meta_arch):
        self._args[NMSProperties.META_ARCH] = meta_arch.value

    @property
    def engine(self):
        return PostprocessTarget(self._args[NMSProperties.ENGINE]) if NMSProperties.ENGINE in self._args else None

    @engine.setter
    def engine(self, engine):
        self._args[NMSProperties.ENGINE] = engine

    @property
    def enforce_iou_threshold(self):
        return self._args.get(NMSProperties.ENFORCE_IOU_THRESHOLD, "True")

    @property
    def bbox_decoding_only(self):
        return self._args.get(NMSProperties.BBOX_DECODING_ONLY, "False") == "True"

    @property
    def config_path(self):
        return self._args.get(NMSProperties.CONFIG_PATH)

    @property
    def dfl_on_nn_core(self):
        return self._args.get(NMSProperties.DFL_ON_NN_CORE, "False") == "True"

    @property
    def output_original_name(self):
        return self._args.get(NMSProperties.OUTPUT_ORIGINAL_NAME, DEFAULT_NMS_OUTPUT_ORIGINAL_NAME)

    @property
    def is_config_from_har(self):
        if self.config_path is None:
            return False

        return self.config_path.endswith(NMS_CONFIG_FILE_FAKE_HAR_PATH)

    def _is_default_or_detected_config(self):
        return self._args.get(NMSProperties.CONFIG_PATH) is None

    @classmethod
    def from_tokens(cls, tokens, script_path, provided_config_file=None):
        named_arg_seen = False
        args_dict = {}
        for i, arg in enumerate(tokens.function_args):
            if isinstance(arg, str):  # positional argument
                if named_arg_seen:
                    raise AllocatorScriptParserException(
                        f"Can't parse positional argument '{arg}' after named "
                        f"arguments. Please specify the argument name.",
                    )
                args_dict[NMS_ARGUMENTS_ORDER[i]] = arg

            elif isinstance(arg, dict):  # named argument
                named_arg_seen = True
                arg_name = next(iter(arg.keys()))
                arg_enum = NMSProperties(arg_name) if arg_name in NMSProperties._value2member_map_ else None
                if arg_enum is None or arg_enum not in NMS_ARGUMENTS_ORDER:
                    raise AllocatorScriptParserException(
                        f"No argument named {arg_name}. Please make sure to use the "
                        f"argument name as it appears in the command description.",
                    )

                args_dict.update({arg_enum: arg[arg_name]})

        return cls(args_dict, script_path, provided_config_file)

    def _update_config_path_in_args_dict(self, script_path):
        config_path = self.config_path
        if config_path is None:
            return

        config_path = config_path.replace("'", "").replace('"', "")
        if config_path.endswith(NMS_CONFIG_FILE_FAKE_HAR_PATH):
            self._args[NMSProperties.CONFIG_PATH] = config_path
            return

        if not os.path.isabs(os.path.expandvars(config_path)):
            # NMS config path is relative to the model script
            if script_path is not None:
                config_path = os.path.join(os.path.dirname(script_path), config_path)
            else:
                config_path = os.path.abspath(config_path)

        self._args[NMSProperties.CONFIG_PATH] = os.path.expandvars(config_path)

    def has_unfound_layers(self, layers_scope_from_hn):
        return False

    def validate_command(self, layers_scope_from_hn, ignore_not_found=False):
        self._validate_command_arguments()
        self._validate_nms_engine()
        self._validate_config_file_in_har()

    def _validate_command_arguments(self):
        if self._args.get(NMSProperties.META_ARCH) is None:
            msg = "Meta architecture argument is missing. Please make sure to provide it."
            raise AllocatorScriptParserException(msg)
        if NMSProperties.CONFIG_PATH in self._args and any(
            x in self._args for x in NMS_ARGUMENTS_ORDER[NMS_ARGUMENTS_ORDER.index(NMSProperties.SCORES_TH) :]
        ):
            msg = "Both config path and config arguments were given. Please provide only a config path or some of its arguments, but not both."
            raise AllocatorScriptParserException(msg)
        # validating values' types
        for arg, value in self._args.items():
            value_type = ARG_TO_TYPE[arg.value]
            if value_type is int and isinstance(value, float) and value.is_integer():
                # the value is integer but stored as float
                self._args.update({arg: int(value)})

            if not isinstance(self._args[arg], value_type):
                raise AllocatorScriptParserException(
                    f"The value of argument {arg.value} must be from type of "
                    f"{value_type!s}. Please make sure the value is correct",
                )
        if self.meta_arch not in SUPPORTED_DFL_ON_NN_CORE_META_ARCHS and self.dfl_on_nn_core:
            raise AllocatorScriptParserException(
                f"Hybrid mode is not supported for {self.meta_arch.value}. Please remove the flag.",
            )

    def _validate_nms_engine(self):
        meta_arch = self.meta_arch
        if self.engine in [PostprocessTarget.NN_CORE, PostprocessTarget.AUTO] and meta_arch not in NN_CORE_META_ARCHS:
            raise UnsupportedMetaArchError(f"The specified meta architecture {meta_arch.value} cannot be run on chip.")

        if self.engine == PostprocessTarget.CPU:
            if meta_arch not in CPU_META_ARCHS:
                raise UnsupportedMetaArchError(
                    f"The specified meta architecture {meta_arch.value} cannot be run on host.",
                )
            if self.bbox_decoding_only:
                logger.info(
                    "Running bbox decoding only on CPU is computationally expensive, since the decoding is done over all the proposals. Please consider running full NMS."
                )

    def _validate_config_file_in_har(self):
        if self.is_config_from_har and not self._config_file:
            msg = "Can't load nms configuration from har file. The har might be corrupted."
            raise AllocatorScriptParserException(msg)

    @classmethod
    def get_default_engine_from_meta_arch(cls, meta_arch):
        if meta_arch in AUTO_SUPPORTED_META_ARCH:
            engine = PostprocessTarget.AUTO
        elif meta_arch in NN_CORE_META_ARCHS:
            engine = PostprocessTarget.NN_CORE
        elif meta_arch in CPU_META_ARCHS:
            engine = PostprocessTarget.CPU
        else:
            raise AllocatorScriptParserException(f"Unsupported meta_arch {meta_arch.value}")
        return engine

    def apply(self, hailo_nn, params, hw_consts):
        if not self.is_config_from_har:
            self._get_config_file_from_path()
        self._verify_meta_arch_configuration(hailo_nn)

        if not self.engine:
            self.engine = self.get_default_engine_from_meta_arch(self.meta_arch)

            logger.info(
                f"For NMS architecture {self.meta_arch.value} the default engine is {self.engine.value}. "
                f"For other engine please use the 'engine' flag in the nms_postprocess model script command. "
                f"If the NMS has been added during parsing, please parse the model again without confirming the "
                f"addition of the NMS, and add the command manually with the desired engine.",
            )

        if self.engine != PostprocessTarget.CPU and self.bbox_decoding_only:
            msg = "Preforming bbox decoding only is possible only when the post-process runs on CPU, please change to engine=cpu in `nms_postprocess` model script command"
            raise AllocatorScriptParserException(msg)

        if self.engine == PostprocessTarget.NN_CORE.value:
            logger.warning(
                "Currently `nn-core` performs only score threshold. In the near future it will perform score"
                " + IoU threshold. If you want to apply IoU threshold please use `cpu` or `auto` instead",
            )

        self._update_config_file(hailo_nn)
        self.validate_nms_config_json()
        pp_creator = create_nms_postprocess(
            hailo_nn,
            params,
            self._config_file,
            self.engine,
            hw_consts,
            self.meta_arch,
            self.enforce_iou_threshold,
            self.bbox_decoding_only,
            self.dfl_on_nn_core,
            self.output_original_name,
        )
        self._nms_config = pp_creator.config
        self.update_modifications_meta_data(pp_creator)

        return pp_creator.hn, pp_creator.weights

    def _verify_meta_arch_configuration(self, hailo_nn):
        if self.meta_arch not in NMSMetaArchitectures:
            raise AllocatorScriptParserException(f"Invalid NMS meta arch was given {self.meta_arch.value}")

        if self.meta_arch == NMSMetaArchitectures.YOLOV6:
            # checks whether the tag of the provided YOLOv6 version is supported

            log_message = "The version of the provided YOLOv6 is not supported. Please use YOLOv6 2.1 instead"
            number_of_bboxes = len(self._config_file["bbox_decoders"])
            expected_outputs = 2 * number_of_bboxes  # 2 outputs for each branch - cls + reg layers
            found_outputs = len(hailo_nn.get_output_layers())
            strides_for_outputs = [bbox["stride"] for bbox in self._config_file["bbox_decoders"]]

            if expected_outputs != found_outputs:
                if META_ARCH_OUTPUT_CHANNELS[NMSMetaArchitectures.YOLOX](number_of_bboxes) == found_outputs:
                    # the current arch is YOLOv6 tag 0.1.0, the postprocess is as YOLOX, changes the meta_arch and the config file accordingly
                    logger.warning(
                        "The current network architecture was detected as YOLOv6 tag 0.1.0, which has the same post-process as YOLOX."
                        "Changing the post-process meta architecture to YOLOX and uses its default configuration. "
                        "Note that the post-process will run on cpu.",
                    )
                    self.meta_arch = NMSMetaArchitectures.YOLOX
                    self.engine = PostprocessTarget.CPU
                    yolox_default_config_path = self._get_default_config_path(self.meta_arch)
                    with open(yolox_default_config_path) as f:
                        self._config_file = json.load(f)

                else:
                    raise AllocatorScriptParserException(log_message)

            for stride in strides_for_outputs:
                reg_layer = [
                    layer
                    for layer in hailo_nn.get_real_output_layers()
                    if (
                        layer.output_shape[1] == math.ceil(self._config_file["image_dims"][0] / stride)
                        and layer.output_shape[-1] == YOLOV6_FEATURES_OUT
                    )
                ]
                cls_layer = [
                    layer
                    for layer in hailo_nn.get_real_output_layers()
                    if (
                        layer.output_shape[1] == math.ceil(self._config_file["image_dims"][0] / stride)
                        and layer.output_shape[-1] == self._config_file["classes"]
                    )
                ]

                if len(reg_layer) == len(cls_layer) == 1:
                    continue
                else:
                    # the cls layer might be with 4 features, checks if it preceded by sigmoid
                    if self._config_file["classes"] == YOLOV6_FEATURES_OUT:
                        reg_layer = [layer for layer in reg_layer if layer.activation == ActivationType.linear]
                        cls_layer = [layer for layer in cls_layer if layer.activation == ActivationType.sigmoid]
                    if len(reg_layer) == len(cls_layer) == 1:
                        continue
                raise AllocatorScriptParserException(log_message)

    def update_modifications_meta_data(self, pp_creator):
        for layer in pp_creator.hn.get_real_output_layers(False):
            if layer.op in (LayerType.nms, LayerType.postprocess):
                hn_layers = self.meta_data["hn_layers"]
                del self.meta_data["hn_layers"]
                config = NMSConfig(
                    cmd_type=SupportedCommands.NMS_POSTPROCESS,
                    engine=self.engine,
                    meta_arch=self.meta_arch,
                    hn_output_layers=hn_layers,
                    sigmoid_layers=pp_creator.sigmoid_layers,
                )
                self.meta_data[layer.outputs[0]] = config

    def validate_nms_config_json(self):
        score_th = self._config_file.get(NMSProperties.SCORES_TH.value, DEFAULT_SCORES_TH)
        iou_th = self._config_file.get(NMSProperties.IOU_TH.value, DEFAULT_IOU_TH)
        max_prop = self._config_file.get(NMSProperties.MAX_PROPOSALS_PER_CLASS.value, DEFAULT_MAX_PROPOSALS_PER_CLASS)
        classes = self._config_file.get(NMSProperties.CLASSES.value, DEFAULT_SSD_CLASSES)
        if score_th < 0:
            raise NMSConfigPostprocessException(f"NMS scores_threshold={score_th} should be non-negative")
        if iou_th < 0 or iou_th > 1:
            raise NMSConfigPostprocessException(f"NMS iou_threshold={iou_th} should be in [0,1] range")
        if max_prop <= 0:
            raise NMSConfigPostprocessException(f"NMS max_proposals_per_class={max_prop} should be greater than 0")
        if classes < 1:
            raise NMSConfigPostprocessException(f"NMS classes={classes} should be greater than 0")
        for bbox_decoder in self._config_file["bbox_decoders"]:
            if BBoxDecodersInfo.H.value in bbox_decoder and (
                len(bbox_decoder[BBoxDecodersInfo.H.value]) != len(bbox_decoder[BBoxDecodersInfo.W.value])
            ):
                raise NMSConfigPostprocessException("h and w must be the same length")

        if self._is_default_or_detected_config():
            logger.info(
                f"Using the default score threshold of {score_th} (range is [0-1], where 1 performs "
                f"maximum suppression) and IoU threshold of {iou_th} (range is [0-1], "
                f"where 0 performs maximum suppression).\nChanging the values is possible using the "
                f"nms_postprocess model script command.",
            )

    def export_nms_metadata(self):
        return NMSMetaData(self._nms_config, self.meta_arch, self.engine, self._config_file)

    def _get_config_file_from_path(self):
        use_path = True
        config_path = self.config_path
        if config_path is None:
            if self._config_file:
                # config file was provided via autodetection while parsing or via load har
                use_path = False
            else:
                config_path = self._get_default_config_path(self.meta_arch)

        if use_path:
            if not os.path.exists(config_path):
                raise AllocatorScriptParserException(
                    f"Post-process config file isn't found in {config_path}. "
                    f"Please make sure the given path is relative to the .alls file "
                    f"location, or it is a global path.",
                )
            with open(config_path) as f:
                self._config_file = json.load(f)

    @staticmethod
    def _get_default_config_path(meta_arch: NMSMetaArchitectures):
        path = f"tools/core_postprocess/default_nms_config_{meta_arch.value}.json"
        path = SDKPaths().join_sdk_client(path)
        if os.path.isfile(path):
            return SDKPaths().join_sdk_client(path)
        else:
            raise AllocatorScriptParserException(f"Meta arch {meta_arch.value} doesn't have default config file.")

    @classmethod
    def get_value_from_default_config_json(cls, key, meta_arch):
        default_json_path = cls._get_default_config_path(meta_arch)
        with open(default_json_path) as f:
            config_file = json.load(f)
        return config_file[key]

    def _update_config_file(self, hailo_nn):
        self._update_config_file_by_engine()
        self._update_command_args_in_config_file()
        self._layers_scope_addition(hailo_nn)
        self._update_config_layers(hailo_nn)

    def _update_command_args_in_config_file(self):
        args_start_index = NMS_ARGUMENTS_ORDER.index(NMSProperties.SCORES_TH)
        for argument in NMS_ARGUMENTS_ORDER[args_start_index:]:
            if self._args.get(argument):
                self._config_file[argument.value] = self._args[argument]

    def _update_config_file_by_engine(self):
        if self.engine == PostprocessTarget.CPU:
            if self._is_default_or_detected_config():
                self._config_file[NMSProperties.IOU_TH.value] = CPU_DEFAULT_IOU_TH.get(self.meta_arch, DEFAULT_IOU_TH)
        elif self.meta_arch == NMSMetaArchitectures.YOLOV5 and (
            self._is_default_or_detected_config() or "input_division_factor" not in self._config_file
        ):
            # the division factor must divide the output shapes of the bbox decoder layer
            self._config_file["input_division_factor"] = self._get_default_division_factor()

    def _get_default_division_factor(self):
        max_stride = max(
            bbox_decoder[BBoxDecodersInfo.STRIDE.value] for bbox_decoder in self._config_file["bbox_decoders"]
        )
        min_height = self._config_file[NMSProperties.IMAGE_DIMS.value][0] // max_stride
        range_beginning = MIN_SUFFICIENT_DIVISION_FACTOR if min_height >= MIN_SUFFICIENT_DIVISION_FACTOR else 1
        for divider in range(range_beginning, min_height + 1):
            if min_height % divider == 0:
                return divider
        return 1

    def _layers_scope_addition(self, hailo_nn):
        fields_with_layer_name = [
            BBoxDecodersInfo.ENCODED_LAYER,
            BBoxDecodersInfo.REG_LAYER,
            BBoxDecodersInfo.CLS_LAYER,
            BBoxDecodersInfo.OBJ_LAYER,
            BBoxDecodersInfo.REG_LAYER_H,
            BBoxDecodersInfo.REG_LAYER_W,
            BBoxDecodersInfo.COMBINED_LAYER,
        ]
        for bbox_decoder in self._config_file.get("bbox_decoders", []):
            scope_addition_fields = [field for field in fields_with_layer_name if bbox_decoder.get(field.value)]
            for field in scope_addition_fields:
                bbox_decoder[field.value] = hailo_nn.get_layer_by_name(bbox_decoder[field.value]).name

    def _update_config_layers(self, hailo_nn):
        if self.meta_arch == NMSMetaArchitectures.CENTERNET:
            self.meta_data["hn_layers"] = [v for k, v in self._config_file["bbox_decoders"][0].items() if k != "name"]
        elif self.meta_arch == NMSMetaArchitectures.SSD:
            self._set_ssd_config_layers(hailo_nn)
        elif self.meta_arch in YOLO_META_ARCHS:
            self._set_yolo_config_layers(hailo_nn)

    @staticmethod
    def _get_output_preds(hailo_nn):
        output_preds = []
        for out in hailo_nn.get_output_layers():
            output_preds.extend(list(hailo_nn.predecessors(out)))

        if any(pred.op != LayerType.conv for pred in output_preds):
            diff_op = next(pred for pred in output_preds if pred.op != LayerType.conv)
            raise AllocatorScriptParserException(
                f"Error in the last layers of the model, expected conv but found {diff_op.op} layer.",
            )
        return output_preds

    def _set_yolo_config_layers(self, hailo_nn):
        meta_arch = self.meta_arch
        classes = self._config_file[NMSProperties.CLASSES.value]
        num_anchors = len(self._config_file["bbox_decoders"][0].get("w", []))
        num_proto = self._config_file["proto"][0][ProtoInfo.NUMBER.value] if "proto" in self._config_file else None
        combined_layer = BBoxDecodersInfo.COMBINED_LAYER.value in self._config_file["bbox_decoders"][0]
        reg_len = self._config_file.get(NMSProperties.REGRESSION_LENGTH.value, 16)
        f_out = get_f_out_by_meta_arch(meta_arch, classes, num_anchors, num_proto, combined_layer, reg_len)

        image_dims = self._config_file.get(
            NMSProperties.IMAGE_DIMS.value,
            hailo_nn.get_input_layers()[0].input_shapes[0],
        )
        conv_layers = self._get_output_preds(hailo_nn)
        self.meta_data["hn_layers"] = [x.name for x in conv_layers]
        for decoder in self._config_file["bbox_decoders"]:
            if (
                "encoded_layer" in decoder
                and not decoder["encoded_layer"]
                or "reg_layer" in decoder
                and not decoder["reg_layer"]
                or "cls_layer" in decoder
                and not decoder["cls_layer"]
                or "objectness_layer" in decoder
                and not decoder["objectness_layer"]
                or "proto_layer" in decoder
                and not decoder["proto_layer"]
                or "combined_layer" in decoder
                and not decoder["combined_layer"]
            ):
                # the encoded layer name was provided in the config file thus not trying to find it
                stride = decoder[BBoxDecodersInfo.STRIDE.value]
                branch_layers = [
                    layer for layer in conv_layers if layer.output_shapes[0][1] == math.ceil(image_dims[0] / stride)
                ]
                if (
                    meta_arch in [NMSMetaArchitectures.YOLOV5, NMSMetaArchitectures.YOLOV5_SEG]
                    and len(branch_layers) == 1
                ):
                    self._set_yolov5_and_5seg_config_layer(decoder, conv_layers, image_dims, stride, f_out)
                elif len(branch_layers) in YOLO_OUTPUTS_PER_BRANCH.get(meta_arch, []):
                    self._set_anchorless_yolo_config_layer(branch_layers, decoder, f_out)
                else:
                    msg = "Cannot infer bbox conv layers automatically. Please specify the bbox layer in the json configuration file"
                    raise AllocatorScriptParserException(msg)

            if meta_arch == NMSMetaArchitectures.YOLOV5_SEG:
                self._set_yolov5_seg_proto_layer(conv_layers)

    def _set_yolov5_seg_proto_layer(self, conv_layers):
        proto = self._config_file["proto"][0]
        if not proto.get(ProtoInfo.PROTO_LAYER.value):
            proto_num = proto[ProtoInfo.NUMBER.value]
            # proto layer is originally conv + sigmoid + mul
            optional_layers = [
                conv for conv in conv_layers if conv.output_shapes[0][-1] == proto_num and len(conv.original_names) == 3
            ]
            if len(optional_layers) == 1:
                proto[ProtoInfo.PROTO_LAYER.value] = optional_layers[0].name
                logger.info(f"{optional_layers[0].full_name_msg} was detected as proto layer.")
            else:
                raise AllocatorScriptParserException("Cant autodetect proto layer. Please provide a full config file.")

    def _set_anchorless_yolo_config_layer(self, branch_layers, decoder, expected_f_out):
        # extracting the names of regression layer, the class layer and the objectness layer
        for layer in branch_layers:
            if layer.op == LayerType.conv:
                layer_type = ""
                f_out = layer.output_shape[-1]
                if f_out == expected_f_out and layer.activation == ActivationType.linear:
                    layer_type = (
                        BBoxDecodersInfo.REG_LAYER.value
                        if BBoxDecodersInfo.REG_LAYER.value in decoder
                        else BBoxDecodersInfo.COMBINED_LAYER.value
                    )
                    if decoder[layer_type] == "":
                        decoder[layer_type] = layer.name
                elif (
                    self.meta_arch == NMSMetaArchitectures.YOLOX
                    and f_out == 1
                    and decoder[BBoxDecodersInfo.OBJ_LAYER.value] == ""
                ):
                    layer_type = BBoxDecodersInfo.OBJ_LAYER.value
                    decoder[layer_type] = layer.name
                elif f_out == self._config_file["classes"] and decoder[BBoxDecodersInfo.CLS_LAYER.value] == "":
                    if self._config_file["classes"] == 1:
                        msg = "Cannot infer bbox conv layers automatically. Please specify the bbox layer in the json configuration file."
                        raise AllocatorScriptParserException(msg)
                    layer_type = BBoxDecodersInfo.CLS_LAYER.value
                    decoder[layer_type] = layer.name
                if layer_type:
                    logger.info(f"The layer {layer.name} was detected as {layer_type}.")

    def _set_yolov5_and_5seg_config_layer(self, decoder, conv_layers, input_shapes, stride, f_out):
        if decoder[BBoxDecodersInfo.ENCODED_LAYER.value] == "":
            encoded_layer = [
                conv
                for conv in conv_layers
                if conv.output_shapes[0][1] == (input_shapes[0] / stride) and conv.output_shapes[0][-1] == f_out
            ]
            if len(encoded_layer) == 1:
                conv_name = encoded_layer[0].name
                decoder[BBoxDecodersInfo.ENCODED_LAYER.value] = conv_name
                logger.info(f"The layer {conv_name} was detected as encoded layer.")
                if len(re.split(r"(\d+)", conv_name)) > 1:
                    decoder["name"] += re.split(r"(\d+)", conv_name)[1]

    def _set_ssd_config_layers(self, hailo_nn):
        conv_layers = self._get_output_preds(hailo_nn)
        self.meta_data["hn_layers"] = [x.name for x in conv_layers]

        if all(
            bbox_decoder[name_field]
            for bbox_decoder in self._config_file["bbox_decoders"]
            for name_field in [BBoxDecodersInfo.REG_LAYER.value, BBoxDecodersInfo.CLS_LAYER.value]
        ):
            # no need to update the config file
            return

        if self._config_file["classes"] == 4:
            msg = "Cannot auto detect regression and class layers for 4 classes. Please provide a full config file."
            raise AllocatorScriptParserException(msg)

        conv_layers = self._get_output_preds(hailo_nn)
        self.meta_data["hn_layers"] = [x.name for x in conv_layers]
        # extracts the bbox decoders that are required names update
        bbox_decoders_to_update = [
            bbox_decoder
            for bbox_decoder in self._config_file["bbox_decoders"]
            if (
                bbox_decoder[BBoxDecodersInfo.CLS_LAYER.value] == ""
                or bbox_decoder[BBoxDecodersInfo.REG_LAYER.value] == ""
            )
        ]

        for idx, decoder in enumerate(bbox_decoders_to_update):
            num_anchors = len(decoder["w"])
            position_condition = conv_layers[idx * 2].output_shapes[0][3] == (4 * num_anchors)
            if decoder[BBoxDecodersInfo.CLS_LAYER.value] == "":
                decoder[BBoxDecodersInfo.CLS_LAYER.value] = (
                    conv_layers[(idx * 2) + 1].name if position_condition else conv_layers[idx * 2].name
                )
                logger.info(f"The layer {decoder[BBoxDecodersInfo.CLS_LAYER.value]} was detected as cls layer.")

            if decoder[BBoxDecodersInfo.REG_LAYER.value] == "":
                decoder[BBoxDecodersInfo.REG_LAYER.value] = (
                    conv_layers[idx * 2].name if position_condition else conv_layers[(idx * 2) + 1].name
                )
                logger.info(f"The layer {decoder[BBoxDecodersInfo.REG_LAYER.value]} was detected as reg layer.")
