#!/usr/bin/env python
"""
This defines SDK Backend exceptions, grouped to the various modules of the SDK
"""

import sys
from enum import Enum

from hailo_sdk_common.exceptions.exceptions import SDKException
from hailo_sdk_common.logger.logger import default_logger


class SDKBackendException(SDKException):
    pass


# Note that these exceptions allows the client an informative message about the nature of the error
class BackendRuntimeException(SDKBackendException):
    pass


class ModelRunnerException(BackendRuntimeException):
    pass


class BackendModelParsingException(SDKBackendException):
    pass


class BackendModelException(SDKBackendException):
    pass


class BackendQuantizationException(SDKBackendException):
    pass


class ProfilingException(SDKBackendException):
    pass


class BackendScriptParserException(SDKBackendException):
    pass


class AllocatorScriptParserException(BackendScriptParserException):
    pass


class QuantizationScriptParserException(BackendScriptParserException):
    pass


class ModuleNotExisting(SDKBackendException):
    pass


class HailoToolsExceptionReturnCodes(Enum):
    INPUT_ARGS_ERROR = 1
    ALLOCATOR_ERROR = 2
    COMPILER_ERROR = 3
    CAT_ERROR = 4

    @classmethod
    def has_member(cls, value):
        return value in cls._value2member_map_


def hailo_tools_exception_handler(exctype, value, traceback):
    if issubclass(exctype, BackendHailoToolsException):
        default_logger().error(f"{exctype.__name__}: {value}")
    else:
        sys.__excepthook__(exctype, value, traceback)


class HailoToolsException(SDKBackendException):
    def __init__(
        self,
        message,
        client_message=None,
        returncode=HailoToolsExceptionReturnCodes.ALLOCATOR_ERROR,
        hailo_tools_error=None,
    ):
        self._returncode = returncode
        self._hailo_tools_error = hailo_tools_error
        super(SDKBackendException, self).__init__(message=message, client_message=client_message)

    @property
    def hailo_tools_error(self):
        return self._hailo_tools_error

    def internal_exception(self, message, hailo_tools_error=None):
        switcher = {
            HailoToolsExceptionReturnCodes.INPUT_ARGS_ERROR: BackendInvalidInputArgsException(message),
            HailoToolsExceptionReturnCodes.ALLOCATOR_ERROR: BackendAllocatorException(
                message,
                hailo_tools_error=hailo_tools_error,
            ),
            HailoToolsExceptionReturnCodes.COMPILER_ERROR: BackendCompilerException(
                message,
                hailo_tools_error=hailo_tools_error,
            ),
            HailoToolsExceptionReturnCodes.CAT_ERROR: BackendCompilerException(
                message,
                hailo_tools_error=hailo_tools_error,
            ),
        }

        return (
            switcher[HailoToolsExceptionReturnCodes(self._returncode)]
            if HailoToolsExceptionReturnCodes.has_member(self._returncode)
            else BackendAllocatorException(message)
        )


# Note that these exceptions does not allow the client a specific message about the nature of the error
class BackendInternalException(SDKBackendException):
    DEFAULT_MSG = "SDK Backend internal error occurred."

    def __init__(self, message, client_message=None):
        client_message = client_message if client_message is not None else type(self).DEFAULT_MSG
        super().__init__(message=message, client_message=client_message)


class BackendHailoToolsException(BackendInternalException):
    def __init__(self, message, client_message=None, hailo_tools_error=None):
        if hailo_tools_error and hailo_tools_error.HasField("error_message"):
            message += f" {hailo_tools_error.error_message}"
        super(BackendInternalException, self).__init__(message=message, client_message=client_message)
        self._hailo_tools_error = hailo_tools_error

    @property
    def hailo_tools_error(self):
        return self._hailo_tools_error


class BackendAllocatorException(BackendHailoToolsException):
    DEFAULT_MSG = "SDK Backend allocator failed."


class BackendCompilerException(BackendHailoToolsException):
    DEFAULT_MSG = "SDK Backend could not compile configuration."


class BackendMapperException(BackendInternalException):
    DEFAULT_MSG = "SDK Backend could not map current network to HW."


class BackendInvalidInputArgsException(BackendInternalException):
    DEFAULT_MSG = "SDK Backend could not recognize hailo tools input argument."

    def __init__(self, message, client_message=None):
        super(BackendInternalException, self).__init__(message=message, client_message=client_message)


class BackendEmulatorException(BackendInternalException):
    DEFAULT_MSG = "SDK Backend emulator failed."


class BackendNotImplementedError(BackendInternalException):
    DEFAULT_MSG = "SDK Backend function not implemented."


class BackendTypeError(BackendInternalException):
    DEFAULT_MSG = "SDK Backend - TypeError."


class BackendValueError(BackendInternalException):
    DEFAULT_MSG = "SDK Backend - ValueError."
