#!/usr/bin/env python
import argparse
import sys
from abc import ABC, abstractmethod

import argcomplete
from tabulate import tabulate

import hailo_sdk_client
from hailo_sdk_client.tools.cmd_utils.cmd_definitions import ClientCommandGroups
from hailo_sdk_common.logger.logger import default_logger


class CmdUtilsBaseUtilError(Exception):
    pass


class CmdUtilsBaseUtil:
    GROUP = ClientCommandGroups.OTHER
    HELP = None

    def __init__(self, parser):
        self._logger = default_logger()
        self._parser = parser


class Helper(CmdUtilsBaseUtil):
    def __init__(self, parser, commands):
        super().__init__(parser)
        self._commands = commands
        self._description = parser.description

    def __call__(self, parser):
        parser.set_defaults(func=self.run)

    def run(self, args):
        table = tabulate([("\t", name, help_msg) for name, (help_msg, _) in self._commands.items()], tablefmt="plain")

        groups = {}
        for (_, command_class), command_row in zip(self._commands.values(), table.splitlines()):
            group = getattr(command_class, "GROUP", ClientCommandGroups.OTHER)
            if group in groups:
                groups[group].append(command_row)
            else:
                groups[group] = [command_row]

        print(f"{self._description}\n")
        for group in groups:
            group_commands = "\n".join(groups[group])
            print(f"{group.value}:\n{group_commands}\n")


class BaseCommands(ABC):
    INVALID_COMMAND_EXIT_CODE = 1

    def __init__(self):
        self.parser = argparse.ArgumentParser(description=self._get_description())
        self.parser.add_argument(
            "--version",
            action="version",
            version=f"Hailo Dataflow Compiler v{hailo_sdk_client.__version__}",
        )
        self.subparsers = self.parser.add_subparsers(help="Hailo utilities aimed to help with everything you need")
        self.COMMANDS = {}

    def run(self):
        argv = sys.argv[1:]
        return self._run(argv)

    def _run(self, argv):
        self.COMMANDS["help"] = "Show the list of commands", Helper(self.parser, self.COMMANDS)
        commands = {}
        for command_name, (help_, command_class) in self.COMMANDS.items():
            commands[command_name] = command_class(self.subparsers.add_parser(command_name, help=help_))

        argcomplete.autocomplete(self.parser)

        if len(argv) == 0:
            self.parser.print_help()
            return self.INVALID_COMMAND_EXIT_CODE

        args = self.parser.parse_args(argv)
        # Due to a bug in Python's handling of subparsers, we cannot require a sub-command using argparse. A manual
        # check is performed here, so we can print out a proper message. (https://bugs.python.org/issue9253)
        if not vars(args):
            self.parser.print_help()
            return self.INVALID_COMMAND_EXIT_CODE

        return args.func(args)

    @abstractmethod
    def _get_description(self):
        pass


class AllsAppendAction(argparse.Action):
    def __call__(self, parser, namespace, values, option_string=None):
        items = getattr(namespace, self.dest) or []
        items.append(values.replace("\\n", "\n"))
        setattr(namespace, self.dest, items)
