#!/usr/bin/env python
import os.path
import platform
import subprocess
import sys
import time
from importlib.metadata import version

import psutil

from hailo_sdk_client.paths_manager.platform_importer import get_base_commands
from hailo_sdk_client.tools.cmd_utils.cmd_definitions import Commands
from hailo_sdk_client.tools.cmd_utils.command_importer import get_command_class
from hailo_sdk_common.logger.logger import default_logger

logger = default_logger()


class ClientCommands(get_base_commands()):
    SDK_COMMANDS = {}
    for command in Commands:
        command_class = get_command_class(command)
        SDK_COMMANDS[command.value] = (command_class.HELP, command_class)

    def __init__(self):
        super().__init__()
        self.COMMANDS.update(type(self).SDK_COMMANDS)

    def _get_description(self):
        return "Hailo Command Line Utility"


def print_system_information(logger):
    logger.info(f"Current Time: {time.strftime('%H:%M:%S, %D', time.localtime())}")
    # extracts the CPU model from lscpu
    cpu_model = (
        subprocess.run(
            ["lscpu | grep 'Model name: ' | cut -d ':' -f 2 | xargs"],
            shell=True,
            check=True,
            capture_output=True,
        )
        .stdout.decode("utf-8")
        .strip()
    )
    logger.info(
        f"CPU: Architecture: {platform.processor()}, Model: {cpu_model}, Number Of Cores: {os.cpu_count()}, "
        f"Utilization: {psutil.cpu_percent(interval=1)}%",
    )
    logger.info(
        f"Memory: Total: {int(psutil.virtual_memory().total / (1024.0 ** 3))}GB, "
        f"Available: {int(psutil.virtual_memory().available / (1024.0 ** 3))}GB",
    )
    logger.info(f"System info: OS: {platform.system()}, Kernel: {platform.uname().release}")
    logger.info(f"Hailo DFC Version: {version('hailo-dataflow-compiler')}")
    try:
        hailort_version = version("hailort")
    except Exception:
        hailort_version = "Not Installed"
    logger.info(f"HailoRT Version: {hailort_version}")

    # prints pcie information using bash subprocess
    # extracts the BDF (bus, device, function) of the PCIE device from lspci
    bus_device_functions = (
        subprocess.run(
            ["lspci -D -d 1e60: | cut -d ' ' -f 1"],
            shell=True,
            check=True,
            capture_output=True,
        )
        .stdout.decode("utf-8")
        .strip()
    )

    if not bus_device_functions:
        logger.info("PCIe: No Hailo PCIe device was found")
        return

    # accesses the current link width and speed of the PCIE device using the BDF
    for bus_device_function in bus_device_functions.split("\n"):
        num_of_lanes = (
            subprocess.run(
                [f"cat /sys/bus/pci/devices/{bus_device_function}/current_link_width"],
                shell=True,
                check=True,
                capture_output=True,
            )
            .stdout.decode("utf-8")
            .strip()
        )
        speed = (
            subprocess.run(
                [f"cat /sys/bus/pci/devices/{bus_device_function}/current_link_speed"],
                shell=True,
                check=True,
                capture_output=True,
            )
            .stdout.decode("utf-8")
            .strip()
        )
        logger.info(f"PCIe: {bus_device_function}: Number Of Lanes: {num_of_lanes}, Speed: {speed}")


def main():
    client_command_runner = ClientCommands()
    print_system_information(logger)
    argv = sys.argv
    if argv:
        logger.info(f"Running `{' '.join([os.path.basename(argv[0]), *argv[1:]])}`")

    ret_val = client_command_runner.run()

    if isinstance(ret_val, int):
        return ret_val
    if isinstance(ret_val, str):
        default_logger().debug(ret_val)
    return 0


if __name__ == "__main__":
    main()
