import os.path
import subprocess

import hailo_sdk_client
from hailo_sdk_client.tools.cmd_utils.base_utils import CmdUtilsBaseUtil
from hailo_sdk_client.tools.cmd_utils.cmd_definitions import ClientCommandGroups
from hailo_sdk_common.logger.logger import default_logger


class DfcStudio(CmdUtilsBaseUtil):
    HELP = "Start DFC Studio"
    GROUP = ClientCommandGroups.ANALYSIS_AND_VISUALIZATION
    MIN_NODE_REQUIRED_VERSION = (20, 9, 0)

    def __init__(self, parser):
        super().__init__(parser)
        parser.add_argument("--port", type=int, default=3000, help="Port to start DFC Studio on. Defaults to 3000.")
        parser.set_defaults(func=self.run)

    def run(self, args):
        dfc_studio_path = self._get_dfc_studio_path()
        default_logger().info("Checking requirements")
        self._check_node_installed()
        default_logger().info("Checking and installing DFC Studio dependencies")
        self._install_dfc_studio_node_dependencies()
        default_logger().info("Starting DFC Studio")
        return subprocess.run(["node", dfc_studio_path, f"--port={args.port}"])

    @staticmethod
    def _get_dfc_studio_path():
        expected_dfc_studio_server_path = os.path.join(
            hailo_sdk_client.tools.__path__[0], "dfc_studio", "dist", "src", "server.js"
        )
        if not os.path.exists(expected_dfc_studio_server_path):
            raise ValueError(f"DFC Studio server.js file wasn't found on path {expected_dfc_studio_server_path}")
        return expected_dfc_studio_server_path

    @staticmethod
    def _parse_node_version(version_string):
        """Parse a version string like 'v20.9.0' into a tuple of integers (20, 9, 0)."""
        return tuple(map(int, version_string.lstrip("v").split(".")))

    def _check_node_installed(self):
        try:
            result = subprocess.run(["node", "-v"], capture_output=True, text=True, check=True)
            version_str = result.stdout.strip()
            node_version = self._parse_node_version(version_str)
            if node_version >= self.MIN_NODE_REQUIRED_VERSION:
                default_logger().info(
                    f"Node.js is installed. Version: {version_str} meets the minimum requirement of v{'.'.join(map(str, self.MIN_NODE_REQUIRED_VERSION))}"
                )
            else:
                raise ValueError(
                    f"Node.js is installed. Version: {version_str} does not meet the minimum requirement of v{'.'.join(map(str, self.MIN_NODE_REQUIRED_VERSION))}."
                )
        except (subprocess.CalledProcessError, FileNotFoundError) as e:
            default_logger().error("Node.js is not installed.")
            raise e

    def _install_dfc_studio_node_dependencies(self):
        expected_dfc_studio_package_json_path = os.path.join(hailo_sdk_client.tools.__path__[0], "dfc_studio", "dist")
        try:
            subprocess.run(
                ["npm", "install", "--prefix", expected_dfc_studio_package_json_path],
                capture_output=True,
                text=True,
                check=True,
            )
        except subprocess.CalledProcessError as e:
            default_logger().error("Node.js cannot install dependencies.")
            raise e
