import argparse

from hailo_model_optimization.acceleras.utils.acceleras_exceptions import AccelerasUnsupportedError
from hailo_model_optimization.algorithms.lat_utils.lat_utils import AnalysisMode
from hailo_sdk_client import ClientRunner
from hailo_sdk_client.exposed_definitions import States
from hailo_sdk_client.tools.cmd_utils.base_utils import CmdUtilsBaseUtil, CmdUtilsBaseUtilError
from hailo_sdk_client.tools.cmd_utils.cmd_definitions import ClientCommandGroups
from hailo_sdk_common.logger.logger import default_logger

JSON_PATH_SUFFIX = "analysis_data.json"


class HailoQuantAnalyzerCLI(CmdUtilsBaseUtil):
    """
    This class is used for the CLI of LAT.

    Args:
       model_path - Path for the quantized har file of the model.
       data-path - Path to the data set for analysis.
       batch-size - The size of the batch for inference.
       data-count - The amount of samples to use for the analysis.
       analyze-mode - The mode that will be executed.

    """

    GROUP = ClientCommandGroups.ANALYSIS_AND_VISUALIZATION
    HELP = "Analyze network quantization noise"

    def __init__(self, parser):
        super().__init__(parser)
        self._logger = default_logger()
        self._parser.add_argument("har_path", type=str, help="Path to the quantized HAR file")
        self._parser.add_argument("--data-path", help="Path to the data set for analysis", type=str, required=True)
        self._parser.add_argument("--batch-size", help="Batch size, the default is 1", type=int, default=1)
        self._parser.add_argument(
            "--data-count",
            type=int,
            help="Maximum number of images to use for the analysis. "
            "By default all the examples in the dataset are used.",
        )
        self._parser.add_argument(
            "--analyze-mode",
            type=str,
            choices=[str(am.value) for am in AnalysisMode],
            help="Choose the analysis mode (simple/advanced).",
        )
        self._parser.add_argument("--use-optimize-model", action="store_true", help=argparse.SUPPRESS)
        self._parser.set_defaults(func=self.run)

    def run(self, args):
        runner = ClientRunner(har=args.har_path)

        # Verify HAR file contains quantize weights
        if runner.state not in [States.QUANTIZED_MODEL, States.COMPILED_MODEL]:
            raise AccelerasUnsupportedError(
                f"Layer noise analysis CLI expects quantize HAR, but got state={runner.state} instead.",
            )

        runner.analyze_noise(
            args.data_path,
            batch_size=args.batch_size,
            data_count=args.data_count,
            analyze_mode=args.analyze_mode,
            use_optimize_model=args.use_optimize_model,
        )

        runner.save_har(args.har_path)

        self._logger.info(
            f"Hailo analyze-noise is completed. Quantized HAR was updated with the Results "
            f"and can be visualized in the Hailo Model profiler using the CLI command:\n"
            f"hailo profiler {args.har_path}",
        )


class QuantAnalyzerException(CmdUtilsBaseUtilError):
    pass
