#!/usr/bin/env python
import argparse
import csv
import json
import os.path
import subprocess
import sys
from contextlib import suppress
from json import JSONDecodeError
from tarfile import ReadError

from hailo_sdk_client.exposed_definitions import States
from hailo_sdk_client.paths_manager.platform_importer import get_platform
from hailo_sdk_client.runner.client_runner import ClientRunner
from hailo_sdk_client.sdk_backend.sdk_backend_exceptions import hailo_tools_exception_handler
from hailo_sdk_client.tools.cmd_utils.base_utils import CmdUtilsBaseUtil, CmdUtilsBaseUtilError
from hailo_sdk_client.tools.cmd_utils.cmd_definitions import ClientCommandGroups
from hailo_sdk_client.tools.profiler.react_report_generator import ReactReportGenerator
from hailo_sdk_common.logger.logger import default_logger


class ProfilerException(Exception):
    pass


def profile(
    out_path=None,
    should_open_web_browser=True,
    should_use_logical_layers=True,
    runner=None,
    hef_filename=None,
    csv_path=None,
    runtime_data=None,
    collect_runtime_data=False,
    stream_fps=None,
    stats_path=None,
):
    if hef_filename is None and runner.state not in [States.COMPILED_MODEL, States.COMPILED_SLIM_MODEL]:
        invalid_args = []
        if runtime_data is not None:
            invalid_args.append("--runtime-data")
        if collect_runtime_data:
            invalid_args.append("--collect-runtime-data")
        if stream_fps is not None:
            invalid_args.append("--stream-fps")

        if invalid_args:
            invalid_args_str = ", ".join(invalid_args)
            raise ProfilerException(
                f"When using {invalid_args_str} runner state must be {States.COMPILED_MODEL} but it is in state {runner.state}",
            )

    if collect_runtime_data:
        if runtime_data is not None:
            default_logger().info(
                "--collect-runtime-data was given but runtime data file already supplied. The flag will be ignored",
            )
        else:
            hef_filename, runtime_data = get_runtime_data(runner, hef_filename)

    export = runner.profile(
        should_use_logical_layers=should_use_logical_layers,
        hef_filename=hef_filename,
        runtime_data=runtime_data,
        stream_fps=stream_fps,
    )
    state = runner.state if hef_filename is None else States.COMPILED_MODEL
    out_path = out_path if out_path else f"{runner.model_name}_{state.value}.html"
    report_generator = ReactReportGenerator(export, out_path)
    csv_data = report_generator.create_report(should_open_web_browser)
    default_logger().info(f"Saved Profiler HTML Report to: {os.path.abspath(out_path)}")

    if csv_path is not None:
        with open(csv_path, "w") as csv_file:
            writer = csv.writer(csv_file)
            for line in csv_data:
                writer.writerow(line)
        default_logger().info(f"Saved Profiler CSV to: {os.path.abspath(csv_path)}")

    if stats_path is not None:
        with open(stats_path, "w") as f:
            f.write(json.dumps(export["stats"]))
        default_logger().info(f"Saved Profiler stats JSON to: {os.path.abspath(stats_path)}")

    return csv_data


def get_runtime_data(runner, hef_filename):
    if hef_filename is None:
        hef_filename = f"{runner.model_name}.hef"
        with open(hef_filename, "wb") as hef_file:
            hef_file.write(runner.hef)

    hailo_platform = get_platform()
    if hailo_platform is None:
        raise ProfilerException(
            "The HailoRT Python API (hailo_platform package) must be installed in order"
            " to use --collect-runtime-data",
        )
    hef_base_name = os.path.splitext(os.path.basename(hef_filename))[0]
    runtime_data = f"runtime_data_{hef_base_name}.json"
    command = f"hailortcli run2 -m raw measure-fw-actions --output-path {runtime_data} set-net {hef_filename}"
    default_logger().info(f"To collect runtime data running:\n{command}")
    result = subprocess.run(command.split(), check=False)
    if result.returncode != 0:
        raise ProfilerException("Command execution failed")

    return hef_filename, runtime_data


class ProfilerCLIException(CmdUtilsBaseUtilError):
    pass


class Profiler(CmdUtilsBaseUtil):
    GROUP = ClientCommandGroups.ANALYSIS_AND_VISUALIZATION
    HELP = "Hailo models Profiler"

    def __init__(self, parser):
        super().__init__(parser)
        parser.add_argument("har_path", type=str, help="Path to a HAR which contains the model description")
        parser.add_argument("--hef", type=str, help="HEF path for profiling")
        runtime_data = parser.add_mutually_exclusive_group(required=False)
        runtime_data.add_argument(
            "--runtime-data",
            type=str,
            help="Path to the runtime data from the hailort for the Runtime tab of the Model profiler",
        )
        runtime_data.add_argument(
            "--collect-runtime-data",
            action="store_true",
            default=False,
            help="Whether to run hailort locally to collect runtime data for the Runtime tab of the Model profiler",
        )
        parser.add_argument("--stream-fps", type=float, help="FPS used for power and bandwidth calculation")
        parser.add_argument("--out-path", type=str, help="Output HTML path (default: <net_name>_<state>.html)")
        parser.add_argument("--csv", type=str, help="Path to the CSV file that would be saved")
        parser.add_argument("--stats-path", type=str, help=argparse.SUPPRESS)
        parser.add_argument("--no-browser", action="store_true", default=False, help=argparse.SUPPRESS)

        parser.set_defaults(func=self.run)

    def run(self, args):
        runner = self._initialize_runner(args.har_path)

        sys.excepthook = hailo_tools_exception_handler  # hack to hide the python API traceback here
        profile(
            out_path=args.out_path,
            runner=runner,
            hef_filename=args.hef,
            csv_path=args.csv,
            runtime_data=args.runtime_data,
            collect_runtime_data=args.collect_runtime_data,
            stream_fps=args.stream_fps,
            should_open_web_browser=not args.no_browser,
            stats_path=args.stats_path,
        )

    @staticmethod
    def _initialize_runner(model_path):
        runner = None
        if model_path.endswith(".har"):
            with suppress(ReadError):
                runner = ClientRunner(har=model_path)

        elif model_path.endswith(".hn"):
            with suppress(JSONDecodeError), open(model_path) as fp:
                runner = ClientRunner(hn=json.load(fp))

        if runner is None:
            raise ProfilerCLIException("The given model must be a valid HAR or HN file")

        return runner
