#!/usr/bin/env python
import csv
import json
import os
import webbrowser
from io import StringIO

import hailo_sdk_client


class ReactReportGenerator:
    INITIAL_DATA_TOKEN = '"%INITIAL_DATA%"'
    INITIAL_LATENCY_TOKEN = '"%LATENCY_DATA%"'
    INITIAL_CONTEXTS_LATENCY_TOKEN = '"%CONTEXTS_LATENCY_DATA%"'
    INITIAL_ACCURACY_TOKEN = '"%INITIAL_ACCURACY%"'

    def __init__(self, export, out_path):
        self._stats = export["stats"]
        self._csv_data = export["csv_data"]
        self._latency_data = export["latency_data"]
        self._runtime_data = export["latency_data"]["runtime_data"]
        self._accuracy_data = export["accuracy_data"]
        self._extended_hn_model_dict = export["hn_model"]
        self._out_path = out_path

    def _generate_html_report(self, serialized_csv_data):
        with open(self._get_report_template_path()) as template_fp, open(self._out_path, "w") as output_fp:
            output_fp.write(self._put_json_data_in_html(template_fp.read(), serialized_csv_data))

    def _put_json_data_in_html(self, html, serialized_csv_data):
        json_data = {
            "csv_data": serialized_csv_data,
            "stats": self._stats,
            "hn_model": self._extended_hn_model_dict,
        }
        html = type(self)._replace_with_json_by_delim(html, json_data, type(self).INITIAL_DATA_TOKEN)
        html = type(self)._replace_with_json_by_delim(html, self._latency_data, type(self).INITIAL_LATENCY_TOKEN)
        html = type(self)._replace_with_json_by_delim(
            html,
            self._runtime_data,
            type(self).INITIAL_CONTEXTS_LATENCY_TOKEN,
        )
        return type(self)._replace_with_json_by_delim(html, self._accuracy_data, type(self).INITIAL_ACCURACY_TOKEN)

    def create_report(self, should_open_web_browser=True):
        layers_begin_row = None
        for idx, line in enumerate(self._csv_data):
            if len(line) == 0:
                continue
            if line[0] == "layer_name":
                layers_begin_row = idx
                break

        csv_data_for_html = self._csv_data[0:2]
        csv_data_for_html.extend(self._csv_data[layers_begin_row:])
        mem_file = StringIO()
        try:
            writer = csv.writer(mem_file, lineterminator=";")
            for line in csv_data_for_html:
                writer.writerow(line)
            mem_file.seek(0)
            self._generate_html_report(mem_file.read())
        finally:
            mem_file.close()
        if should_open_web_browser:
            webbrowser.open(self._out_path)
        return self._csv_data

    @staticmethod
    def _replace_with_json_by_delim(html_data, new_data, delim, dump=False):
        txt = list(html_data.partition(delim))
        txt[1] = json.dumps(new_data)
        if dump:
            with open("file_{}.json".format(delim.replace("%", "").replace('"', "")), "w") as f:
                f.write(json.dumps(new_data))
        return "".join(txt)

    @staticmethod
    def _get_report_template_path():
        return os.path.join(hailo_sdk_client.tools.__path__[0], "profiler", "report_template.html")
