#!/usr/bin/env python
import json
import os
import webbrowser
from contextlib import suppress
from tarfile import ReadError

from hailo_sdk_client.hailo_archive.hailo_archive import HailoArchiveLoader
from hailo_sdk_client.tools.cmd_utils.base_utils import CmdUtilsBaseUtil, CmdUtilsBaseUtilError
from hailo_sdk_client.tools.cmd_utils.cmd_definitions import ClientCommandGroups
from hailo_sdk_common.hailo_nn.hailo_nn import HailoNN
from hailo_sdk_common.logger.logger import default_logger


class VisualizerException(CmdUtilsBaseUtilError):
    pass


class Visualizer(CmdUtilsBaseUtil):
    GROUP = ClientCommandGroups.ANALYSIS_AND_VISUALIZATION
    HELP = "HAR visualization tool"

    def __init__(self, parser):
        super().__init__(parser)
        parser.add_argument("har_path", type=str, help="HAR path. This file contains the network description")
        parser.add_argument("--out-path", type=str, default=None, help="Output SVG path (default: <net_name>.svg)")
        parser.add_argument(
            "-v",
            "--verbose",
            action="store_true",
            default=False,
            help="Add more details to the created SVG",
        )
        parser.add_argument(
            "--no-browser",
            action="store_true",
            default=False,
            help="Use this flag if the preference is to not open the browser",
        )
        parser.set_defaults(func=self.run)

    @staticmethod
    def run(args):
        visualizer_args = vars(args)
        visualizer_args.pop("func")
        VisualizerFromObj(**visualizer_args).run()


class VisualizerFromObj:
    """Model visualize wrapper. Can be used via the Visualizer (as a cmd tool) or directly (via import)."""

    def __init__(self, har_path, out_path=None, verbose=False, no_browser=False, har=False):
        self._model_path = har_path
        self._out_path = out_path
        self._verbose = verbose
        self._no_browser = no_browser
        self._har = har

    def _fix_hn(self):
        if self._model_path.endswith(".har") or self._har:
            with suppress(ReadError), HailoArchiveLoader(self._model_path) as har_loader:
                hn = json.loads(har_loader.get_hn())
                hn.pop("direct_control", None)
                return hn

        if self._model_path.endswith(".hn"):
            with suppress(json.JSONDecodeError), open(self._model_path) as hn_file:
                return json.load(hn_file)

        raise VisualizerException("The given model must be a valid HAR file")

    def _get_hailo_nn(self):
        hn = self._fix_hn()
        return HailoNN.from_parsed_hn(hn)

    def _load(self):
        hailo_nn = self._get_hailo_nn()
        if self._out_path is None:
            self._out_path = f"{hailo_nn.name}.svg"

        return hailo_nn

    def run(self):
        """Runs the visualize functions. Opens result in browser unless "no_browser" == True."""
        hn = self._load()
        hn.visualize(self._out_path, self._verbose)
        default_logger().info(f"Visualized network saved to: {os.path.abspath(self._out_path)}")
        if not self._no_browser:
            webbrowser.open(self._out_path)
