#!/usr/bin/env python
import copy
import json
import os
import re
from collections import OrderedDict
from functools import cmp_to_key, reduce
from operator import attrgetter

import jsonref
import jsonschema
import networkx as nx
import tensorflow as tf
from networkx.algorithms.traversal.breadth_first_search import bfs_tree
from past.utils import old_div
from tabulate import tabulate

from hailo_model_optimization.acceleras.hailo_layers.op_factories import get_layer_type_from_hn_element
from hailo_model_optimization.acceleras.model_optimization_config.mo_config import update_nested
from hailo_model_optimization.acceleras.model_optimization_config.mo_config_layer import (
    LayerPrecisionConfig,
    LayerTranslationConfig,
)
from hailo_model_optimization.acceleras.utils.acceleras_definitions import (
    DEFAULT_OPTIMIZATION_TARGET,
    BiasMode,
    ConcatAxis,
    PostprocessTarget,
    PrecisionMode,
)
from hailo_model_optimization.acceleras.utils.acceleras_exceptions import AccelerasImplementationError
from hailo_sdk_common.compatibility import ensure_str
from hailo_sdk_common.hailo_nn.exceptions import (
    HailoNNException,
    InvalidHNError,
    InvalidHNStage,
    ShapeMismatchError,
    UnsupportedModelError,
)
from hailo_sdk_common.hailo_nn.hn_definitions import (
    DefuseType,
    FeatureMultiplierType,
    HnStage,
    HnVersion,
    InputConversions,
    LayerType,
    ResizeMethod,
)
from hailo_sdk_common.hailo_nn.hn_layers import (
    INDEX_NOT_SET,
    ActivationLayer,
    ArgmaxLayer,
    BatchNormLayer,
    BboxDecoderLayer,
    BiasAddLayer,
    ConcatLayer,
    ConstInputLayer,
    Conv2DLayer,
    DemuxLayer,
    DenseLayer,
    DepthToSpaceLayer,
    EWAddLayer,
    EWDivLayer,
    EWMaxLayer,
    EWMinLayer,
    EWMultLayer,
    EWSubLayer,
    ExternalInputLayer,
    ExternalOutputLayer,
    ExternalPadLayer,
    FeatureInterleaveLayer,
    FeatureMultiplierLayer,
    FeatureShuffleLayer,
    FeatureSplitterLayer,
    FormatConversionLayer,
    FusedBatchNormLayer,
    FusedBboxDecoderLayer,
    FusedConv2DLayer,
    FusedDenseLayer,
    FusedSliceLayer,
    FusedStandaloneActivationLayer,
    FusedStandaloneEWAddLayer,
    FusedStandaloneEWSubLayer,
    GlobalAvgPoolingLayer,
    InputLayer,
    LayerNormalizationLayer,
    LoopbackLayer,
    MatmulLayer,
    NMSLayer,
    NormalizationLayer,
    OutputLayer,
    OutputMuxLayer,
    PoolingLayer,
    PortalLayer,
    PostprocessLayer,
    PPInputLayer,
    PPOutputLayer,
    PrecisionSplitterLayer,
    PrecisionSplitterSignedLayer,
    ProposalGeneratorLayer,
    ReduceMaxLayer,
    ReduceMeanLayer,
    ReduceMinLayer,
    ReduceSumLayer,
    ResizeLayer,
    RowSplitterLayer,
    ShortcutLayer,
    SliceLayer,
    SoftmaxLayer,
    SpaceToDepthLayer,
    SpatialSplitterLayer,
    WidthSplitterLayer,
    input_to_output_height_width,
)
from hailo_sdk_common.hailo_nn.hn_layers.layer_with_activation import ACTIVATION_ATTRS
from hailo_sdk_common.hailo_nn.layer_equiv_set import LayersEquivSet
from hailo_sdk_common.logger.logger import default_logger
from hailo_sdk_common.paths_manager.paths import SDKPaths
from hailo_sdk_common.serialization.numpy_serialization import hailo_np_savez
from hailo_sdk_common.tools.models_translator_helper import valid_orig_name


def prettify_json(json_dump):
    """Prettify JSON by removing extra whitespace inside [] brackets."""
    brackets_count = 0
    inside_string = False
    inside_escaping = False
    result = ""
    for ch in json_dump:
        if inside_string:
            # inside/outside brackets, inside string
            if inside_escaping:
                inside_escaping = False
            elif ch == '"':
                inside_string = False
            elif ch == "\\":
                inside_escaping = True
            result += ch
        elif brackets_count == 0:
            # outside brackets, outside string
            if ch == "[":
                brackets_count += 1
            elif ch == '"':
                inside_string = True
            result += ch
        elif ch == '"':
            inside_string = True
            result += ch
        elif ch == "[":
            brackets_count += 1
            result += ch
        elif ch == "]":
            brackets_count -= 1
            result += ch
        elif ch in [" ", "\n"]:
            # remove whitespace from result string
            pass
        elif ch == ",":
            result += ", "
        else:
            result += ch
    return result


def hn_to_npz_key(layer_name, key_name):
    suffix = ""
    if ":0" not in key_name:
        suffix = ":0"
    return f"{layer_name}/{key_name}{suffix}"


class NetParams:
    def __init__(self, net_params):
        self.clusters_placement = net_params.get("clusters_placement", [[]])
        # If stage was not set, give it the default value of HN
        self.stage = HnStage[net_params["stage"]] if "stage" in net_params else HnStage.HN
        self.clusters_to_skip = net_params.get("clusters_to_skip", [])
        self.version = HnVersion(net_params["version"]) if "version" in net_params else HnVersion.V1_0
        self.output_layers_order = (
            list(net_params["output_layers_order"]) if "output_layers_order" in net_params else []
        )
        self.is_transformer = net_params.get("is_transformer", False)
        self.prefill_size = net_params.get("prefill_size")
        self.cache_size = net_params.get("cache_size")
        self.transposed_net = net_params.get("transposed_net", False)
        self.net_scopes = net_params.get("net_scopes", [])
        self.lora_adapters = net_params.get("lora_adapters", [])

        # backward compatibility fix, to be removed as soon as possible
        if self.lora_adapters == 0:
            self.lora_adapters = []

    def to_json(self):
        result = OrderedDict()
        result["version"] = self.version.value
        result["stage"] = self.stage.value
        result["clusters_placement"] = self.clusters_placement
        result["clusters_to_skip"] = self.clusters_to_skip
        result["output_layers_order"] = self.output_layers_order
        result["is_transformer"] = self.is_transformer
        result["transposed_net"] = self.transposed_net
        if self.prefill_size:
            result["prefill_size"] = self.prefill_size
        if self.cache_size:
            result["cache_size"] = self.cache_size
        result["net_scopes"] = self.net_scopes
        result["lora_adapters"] = self.lora_adapters
        return result


class HailoNN(nx.DiGraph):
    """Hailo NN representation. This is the Python class that corresponds to HN files."""

    TYPE_TO_HN_NAME = {LayerType.dense: "fc", LayerType.bias_add: "bias_add"}

    def __init__(self, network_name=None, stage=None, **kwargs):
        """Initialize an HailoNN object."""
        super().__init__(**kwargs)
        self._logger = default_logger()
        if network_name is None:
            network_name = "HailoNN"
        self.name = network_name
        net_params = {"stage": stage} if stage is not None else {}

        self.net_params = NetParams(net_params)
        self._equivalence = OrderedDict()
        self._all_components = None
        self._detected_anchors = {}
        self._blocks = {}

    def build_equiv_sets(self, equiv_set_algo):
        equiv_sets = OrderedDict()
        handled_sources = set()
        for layer in self.stable_toposort():
            equiv_set = LayersEquivSet.build_layer_equiv_set(self, layer, equiv_set_algo, handled_sources)
            if equiv_set is None:
                continue
            handled_sources |= set(equiv_set.source_layers)
            equiv_sets[layer] = equiv_set
        if equiv_set_algo not in self._equivalence:
            self._equivalence[equiv_set_algo] = []
        key = len(self._equivalence[equiv_set_algo])
        self._equivalence[equiv_set_algo].append(equiv_sets)
        return self._equivalence[equiv_set_algo][key], key

    def iter_equiv_sets(self, equiv_set_algo, start_layers=None, key=None):
        valid_subgraph = self._get_componenet_sub_graph(start_layers)

        algo_equivalence, _ = self.get_equiv_sets(equiv_set_algo, key)
        for layer in valid_subgraph.stable_toposort():
            if layer not in algo_equivalence:
                continue
            yield algo_equivalence[layer]

    def _get_componenet_sub_graph(self, start_layers):
        if start_layers is None:
            start_layers = self.get_input_layers()
        start_layer_successors = self._get_all_layers_successors(start_layers)
        return self.subgraph(start_layer_successors)

    def _get_all_layers_successors(self, given_layers):
        given_layers_successors = set()
        if given_layers is None:
            return given_layers_successors
        for layer in given_layers:
            subgraph = bfs_tree(self, layer)
            given_layers_successors |= subgraph.nodes

        return given_layers_successors

    def get_equiv_sets(self, equiv_set_algo, key=None):
        existing_key = (key is not None) and (len(self._equivalence[equiv_set_algo]) > key)
        if (equiv_set_algo in self._equivalence) and existing_key:
            return self._equivalence[equiv_set_algo][key], key

        return self.build_equiv_sets(equiv_set_algo)

    def _set_layer_scope_name(self, layer, scope):
        if not scope:
            return

        if layer.scope:
            self._logger.debug(f"Replacing layer {layer.name} scope name to {scope}")

        layer_name_without_scope = layer.name_without_scope
        old_layer_name = layer.name
        layer.name = f"{scope}/{layer_name_without_scope}"
        if scope not in self.net_params.net_scopes:
            self.net_params.net_scopes.append(scope)
        if old_layer_name in self.net_params.output_layers_order:
            layer_index = self.net_params.output_layers_order.index(old_layer_name)
            self.net_params.output_layers_order[layer_index] = layer.name

        for pred in self.predecessors(layer):
            layer_index = pred.outputs.index(old_layer_name)
            pred.outputs[layer_index] = layer.name
        for succ in self.successors(layer):
            layer_index = succ.inputs.index(old_layer_name)
            succ.inputs[layer_index] = layer.name

        if layer.defuse_name:
            layer.defuse_name = f"{scope}/{layer.defuse_name_without_scope}"

        if layer.op == LayerType.merged_layer:
            for sub_layer in layer.sub_layers:
                if sub_layer.defuse_name:
                    sub_layer.defuse_name = f"{scope}/{sub_layer.defuse_name_without_scope}"

    def get_start_layers_of_component(self, component):
        input_layers = set(self.get_input_layers())

        intersect = list(component.nodes & input_layers)
        intersect.sort()  # for stable iteration
        return intersect

    def count_layers_naively(self):
        layers = set()
        for layer in self:
            if not layer.is_real_layer():
                continue
            # Defused layers - Use the Defused name so they will be counted only once
            if layer.is_defused():
                if self.net_params.stage not in [HnStage.DEFUSED, HnStage.EXPANDED, HnStage.SPLIT]:
                    self._logger.warning(f"Layer is Defused but HN stage is {self.net_params.stage}")
                layers.add(layer.defuse_name)
                continue
            # Concat layers are not marked as defused so we have an alternative check
            if layer.op == LayerType.concat and any(pred.is_defused() for pred in self.predecessors(layer)):
                continue
            # Non defused layers - count normally
            if layer.name in layers:
                self._logger.warning("Duplicate layer names while counting them")
            layers.add(layer.name)
        return len(layers)

    def _relabel_nodes_mapping(self, graph, verbose):
        """
        build a map for relabeling the graph.
        map from original label to new label
        """
        res = {}
        # Not changing names for pre-fused HNs (0 scopes) or multi-scope HNs (>1 scopes).
        if len(self.net_params.net_scopes) != 1:
            return None
        if verbose:
            # when verbose is true the graph is self.
            for node in graph.nodes():
                new_node = copy.deepcopy(node)
                new_node.name = node.name_without_scope
                res[node] = new_node
        else:
            # when verbose is false the graph is descriptions (strings)
            for orig_desc in graph.nodes():
                split_desc = orig_desc.split("/", maxsplit=1)
                if len(split_desc) > 1:
                    res[orig_desc] = split_desc[1]
        return res

    def visualize(self, filename_prefix, verbose=True):
        graph = self if verbose else self._get_simplified_graph()
        if not SDKPaths().has_graphviz:
            self._logger.warning("Cannot visualize Hailo NN because graphviz in unavailable")
            return

        node_mapping = self._relabel_nodes_mapping(graph, verbose)
        relabeled_graph = nx.relabel_nodes(graph, node_mapping) if node_mapping else graph
        svg_path = filename_prefix if filename_prefix.endswith(".svg") else f"{filename_prefix}.svg"
        dot_path = svg_path.replace(".svg", ".dot")
        nx.drawing.nx_agraph.write_dot(relabeled_graph, dot_path)
        os.system(f'dot -Tsvg "{dot_path}" -o "{svg_path}"')

    def save_hn(self, path):
        if not path.endswith(".hn"):
            net_name = path
            path += ".hn"
        else:
            net_name = path.rsplit(".", 1)[0]

        with open(path, "w") as f:
            f.write(self.to_hn(net_name))

    def _get_simplified_graph(self):
        new_graph = nx.DiGraph()
        descriptions = []
        for node in self:
            desc = node.short_description
            assert (
                desc not in descriptions
            ), f"Cannot visualize because two layers have this identical description: {desc}"
            new_graph.add_node(desc)
            descriptions.append(desc)
        for node1, node2, attributes in self.edges(data=True):
            if "in_shape" not in attributes and "out_shape" not in attributes:
                edge_description = "(no shape yet)"
            elif attributes["in_shape"] == attributes["out_shape"]:
                edge_description = attributes["in_shape"]
            else:
                edge_description = "{} -> {}\n(implicit reshape)".format(
                    attributes["out_shape"],
                    attributes["in_shape"],
                )
            new_graph.add_edge(node1.short_description, node2.short_description, label=edge_description)
        return new_graph

    def stable_toposort(self, key=None):
        """
        Get a generator over the model's layers, topologically sorted.

        Example:
            >>> example_hn = '''{
            ...     "name": "Example",
            ...     "layers": {
            ...         "in": {"type": "input_layer", "input": [], "output": ["out"], "input_shape": [-1, 10]},
            ...         "out": {"type": "output_layer", "input": ["in"], "output": [], "input_shape": [-1, 10]}
            ...     }
            ... }'''
            >>> hailo_nn = HailoNN.from_hn(example_hn)
            >>> for layer in hailo_nn.stable_toposort():
            ...     print('The layer name is "{}"'.format(layer.name))
            The layer name is "in"
            The layer name is "out"

        """
        if key is not None:
            return nx.lexicographical_topological_sort(self, key=attrgetter(key))
        return nx.lexicographical_topological_sort(self)

    def successors(self, layer, sort=True):
        base_iter = super().successors
        iter_to_use = base_iter(layer) if not sort else sorted(base_iter(layer), key=cmp_to_key(layer.sort_outputs()))
        yield from iter_to_use

    def predecessors(self, layer, sort=True):
        base_iter = super().predecessors
        iter_to_use = base_iter(layer) if not sort else sorted(base_iter(layer), key=cmp_to_key(layer.sort_inputs()))
        yield from iter_to_use

    def get_layer_dtype(self, layer):
        return (
            tf.uint16
            if layer.precision_config.precision_mode
            in [PrecisionMode.a16_w16, PrecisionMode.a16_w16_a16, PrecisionMode.a16_w16_a8]
            else tf.uint8
        )

    def get_output_layers_by_recipe(self, recipe_nodes):
        return [node for node in recipe_nodes if not list(self.successors(node))]

    def get_output_dtypes_by_recipe(self, recipe):
        recipe_nodes = list(recipe)
        dtypes = []
        for output_layer in self.get_output_layers_by_recipe(recipe_nodes):
            dtype = self.get_layer_dtype(output_layer)
            real_output_layers = self.find_real_output_layers(output_layer)
            dtypes.extend([dtype] * max(1, len(real_output_layers)))
        return dtypes

    def get_output_dtypes(self):
        dtypes = []
        for output_layer in self.get_output_layers():
            dtype = self.get_layer_dtype(output_layer)
            real_output_layers = self.find_real_output_layers(output_layer)
            # Support NMS layers - 16 bit dtype
            if len(real_output_layers) > 0:
                dtypes.extend([tf.uint16 if isinstance(layer, NMSLayer) else dtype for layer in real_output_layers])
            else:
                dtypes.extend([dtype])
        return dtypes

    def get_real_output_layers_by_recipe(self, recipe):
        real_output_layers = []
        recipe_nodes = list(recipe)
        output_layers = self.get_output_layers_by_recipe(recipe_nodes)
        for output_layer in output_layers:
            real_output_layers += self.get_real_output_layer(output_layer)
        return real_output_layers

    def get_real_output_layers(self, remove_non_neural_core_layers=True):
        real_output_layers = []
        output_layers = self.get_output_layers(remove_non_neural_core_layers)
        for output_layer in output_layers:
            real_output_layers += self.get_real_output_layer(output_layer)
        return real_output_layers

    def get_real_input_layers(self):
        real_input_layers = []
        for input_layer in self.get_non_const_input_layers():
            real_input_layers += self.successors(input_layer)
        return real_input_layers

    def get_real_output_layer(self, layer):
        # this situation happens when layers from ppu are connected directly to output layer (like in hailo15h)
        if layer.op in [LayerType.format_conversion, LayerType.nms, LayerType.softmax, LayerType.resize]:
            return [layer]

        if layer.op not in [LayerType.output_layer, LayerType.pp_output_layer, LayerType.external_output_layer]:
            return []

        real_out_layers = []
        output_layer_predecessors = self.predecessors(layer)
        for output_layer_predecessor in output_layer_predecessors:
            if output_layer_predecessor.op in [
                LayerType.output_mux,
                LayerType.portal,
                LayerType.input_layer,
                LayerType.pp_input_layer,
            ]:
                real_out_layers += self.find_real_output_layers(output_layer_predecessor)
            else:
                real_out_layers.append(output_layer_predecessor)
        return real_out_layers

    def find_real_output_layers(self, output_layer, real_output_layers=None):
        if real_output_layers is None:
            real_output_layers = []
        edge_layer_types = [
            LayerType.portal,
            LayerType.output_layer,
            LayerType.input_layer,
            LayerType.pp_output_layer,
            LayerType.pp_input_layer,
            LayerType.output_mux,
        ]

        for node in self.predecessors(output_layer):
            if node.op in edge_layer_types and len(list(self.predecessors(node))) > 0:
                self.find_real_output_layers(node, real_output_layers)
            else:
                real_output_layers.append(node)
        return real_output_layers

    def get_layer_by_name(self, layer_name):
        potential_matches = []
        for layer in self:
            if layer.name == layer_name:
                return layer

            if layer.name_without_scope == layer_name:
                potential_matches.append(layer)

        if len(potential_matches) == 1:
            return potential_matches[0]
        if len(potential_matches) > 1:
            raise HailoNNException(
                f"The layer named {layer_name} exist under multiple scopes in the HN {[layer.scope for layer in potential_matches]}",
            )

        raise HailoNNException(f"The layer named {layer_name} doesn't exist in the HN")

    def _has_hw_external_io_layers(self):
        return self.net_params.stage in [HnStage.EXPANDED, HnStage.SPLIT]

    def _has_hw_io_layers(self):
        return self.net_params.stage == HnStage.SPLIT

    def get_all_input_layers(self):
        all_start_nodes = [layer for layer in self if self.in_degree(layer) == 0]
        ops = [LayerType.external_input_layer] if self._has_hw_external_io_layers() else [LayerType.input_layer]
        ops.append(LayerType.const_input)
        for layer in all_start_nodes:
            assert layer.op in ops, f"Layer {layer} is not an input layer but its in_degree is 0"

        return all_start_nodes

    def get_non_const_input_layers(self):
        return [layer for layer in self.get_all_input_layers() if layer.op != LayerType.const_input]

    def get_input_layers(self, edge_layers_only=True):
        non_const_start_nodes = self.get_non_const_input_layers()
        if not self._has_hw_io_layers():
            return non_const_start_nodes

        start_nodes = [succ for non_const_start in non_const_start_nodes for succ in self.successors(non_const_start)]
        for layer in start_nodes:
            assert layer.op in [
                LayerType.input_layer,
                LayerType.pp_input_layer,
                LayerType.external_output_layer,
            ], f"Layer {layer} is not an input layer but its predecessor is an external input layer"

        input_layers = [layer for layer in self if layer.op in [LayerType.input_layer, LayerType.pp_input_layer]]
        if edge_layers_only:
            # Filter out inner input layers
            input_layers = [layer for layer in input_layers if layer in start_nodes]

        return input_layers

    def get_external_output_layers(self, remove_non_neural_core_layers=True):
        # if remove_non_neural_core_layers=True the function will return all the output layers
        # except those their predecessor is postprocess layer
        op = LayerType.external_output_layer if self._has_hw_external_io_layers() else LayerType.output_layer
        external_end_nodes = [
            layer
            for layer in self
            if (
                (not remove_non_neural_core_layers and self.out_degree(layer) == 0)
                or (
                    remove_non_neural_core_layers
                    and layer.op == op
                    and self.out_degree(layer) == 0
                    and next(iter(self.predecessors(layer))).op != LayerType.postprocess
                )
            )
        ]
        for layer in external_end_nodes:
            assert layer.op == op, f"Layer {layer} is not an output layer but its out_degree is 0"

        return external_end_nodes

    def get_output_layers(self, remove_non_neural_core_layers=True, edge_layers_only=True):
        # if remove_non_neural_core_layers=True the function will return all the output layers
        # except those their predecessor is postprocess layer
        external_end_nodes = self.get_external_output_layers(remove_non_neural_core_layers)
        if not self._has_hw_io_layers():
            output_layers = external_end_nodes
        else:
            valid_output_types = [
                LayerType.output_layer,
                LayerType.pp_output_layer,
                LayerType.external_input_layer,
                LayerType.format_conversion,
                LayerType.nms,
                LayerType.softmax,
                LayerType.resize,
            ]
            end_nodes = [
                pred for external_end_node in external_end_nodes for pred in self.predecessors(external_end_node)
            ]
            for layer in end_nodes:
                assert (
                    layer.op in valid_output_types
                ), f"Layer {layer} is not an output or external input but its successor is an external output layer"

            output_layers = [layer for layer in self if layer.op in valid_output_types]
            if edge_layers_only:
                # Filter out inner output layers
                output_layers = [layer for layer in output_layers if layer in end_nodes]

        if not self.net_params.output_layers_order or len(output_layers) == 1:
            return output_layers

        ordered_out_layers = []
        for real_output_layer_name in self.net_params.output_layers_order:
            for out_layer in output_layers:
                real_outputs_names = [
                    layer.name if not layer.defuse_name else layer.defuse_name
                    for layer in self.get_real_output_layer(out_layer)
                ]
                if real_output_layer_name in real_outputs_names and out_layer not in ordered_out_layers:
                    ordered_out_layers.append(out_layer)
        if len(output_layers) == len(ordered_out_layers):
            return ordered_out_layers
        self._logger.warning("output_layers_order in net_params don't match actual output layers in HN.")
        return output_layers

    def get_input_shapes(self, ignore_conversion=False, specific_lname=None):
        input_shapes = []
        for input_layer in self.get_input_layers():
            if specific_lname and specific_lname != input_layer.name:
                continue
            shape = input_layer.output_shape
            if ignore_conversion:
                succ = next(iter(self.successors(input_layer)))
                if succ.op == LayerType.format_conversion and succ.conversion_type in InputConversions:
                    shapes = succ.output_shapes
                    if len(shapes) != 1:
                        raise UnsupportedModelError(f"Can't ignore input conversion {succ.name}")
                    shape = shapes[0]

            input_shapes.append(shape)

        return input_shapes

    def set_input_shapes(self):
        """Set input shapes according to predecessors' output shapes."""
        for layer in self.stable_toposort():
            preds = list(self.predecessors(layer))
            if layer.op in [LayerType.input_layer, LayerType.pp_input_layer]:
                layer.input_shape = layer.output_shape
            elif len(preds) > 0:
                input_shapes = []
                for pred in preds:
                    if pred.op == LayerType.concat:
                        # edge case: concat output shape validates its input shape, which can be unknown at this stage
                        # of model creation. we allow unsafe output shape for this case, and rely on later stage that
                        # calculates and validates the correct shapes.
                        input_shapes.append(pred.unsafe_output_shape)
                    elif (
                        pred.op
                        in [
                            LayerType.feature_splitter,
                            LayerType.row_splitter,
                            LayerType.width_splitter,
                            LayerType.demux,
                            LayerType.feature_multiplier,
                            LayerType.precision_splitter,
                        ]
                        or pred.op == LayerType.format_conversion
                        and pred.is_nv_converter()
                    ):
                        if pred.op == LayerType.feature_splitter and pred.split_indices:
                            input_shapes.append(pred.output_shapes[pred.split_indices[pred.outputs.index(layer.name)]])
                        else:
                            input_shapes.append(pred.output_shapes[pred.outputs.index(layer.name)])
                    else:
                        input_shapes.append(pred.output_shape)

                layer.input_shapes = input_shapes
            else:
                raise UnsupportedModelError(f"Can't find layer input shape at {layer.full_name_msg}")

    @staticmethod
    def _reduce_shape(shape):
        return [shape[0], 1, 1, reduce(lambda x, y: x * y, shape[1:], 1)]

    def _compare_shapes(self, layer1, layer2, layer1_output_shape, layer2_input_shape):
        """
        Compare layer1 output shape with layer2 input shape, while considering conv to dense
        reshaping.
        """
        shape1 = copy.deepcopy(layer1_output_shape)
        shape2 = copy.deepcopy(layer2_input_shape)
        if layer2.op == LayerType.conv and layer2.ew_add_enabled and layer1 in layer2.ew_add_connections:
            # in case of conv'n'add, input_shape is the conv input shape, so we should handle the
            # add input shape separately
            if layer2.defuse_type != DefuseType.super_conv:
                if layer2.defuse_type != DefuseType.none:
                    batch, height, width, _ = shape1
                    shape1 = [batch, height, width, layer2.kernel_shape[3]]
                layer2_out_h, layer2_out_w = input_to_output_height_width(
                    layer2.input_shape,
                    layer2.kernel_shape,
                    layer2.strides,
                    layer2.padding,
                )
                shape2 = [shape2[0], layer2_out_h, layer2_out_w, layer2.kernel_shape[3]]
        if shape1[1:-1] != [1, 1] and shape2[1:-1] == [1, 1]:
            # dense after conv and similar cases
            shape1 = self._reduce_shape(shape1)
        if shape1[1:-1] == [1, 1] and shape2[1:-1] != [1, 1]:
            # this does not seem to make sense because there is never conv after dense, but apparently
            # it does happen when parsing activation after dense from a TF protobuf
            shape2 = self._reduce_shape(shape2)
        if shape1 != shape2:
            raise ShapeMismatchError(
                f"{layer1.full_name_msg} has an expected output shape {layer1_output_shape}, while"
                f" its successor {layer2.full_name_msg} has a non-matching input shape "
                f"{layer2_input_shape}\nLayer = {layer1.name} = {layer1}\n"
                f"Successor = {layer2.name} = {layer2}",
            )

    def get_number_of_successors(self, layer):
        return len(list(self.successors(layer)))

    def calculate_shapes(self, meta_edges_graph=None, validate_shapes=True):
        # handle input list for multi-input layers (elementwise, concat, bbox, etc.)
        self.update_input_lists()

        graph = nx.lexicographical_topological_sort(meta_edges_graph) if meta_edges_graph else self.stable_toposort()
        for layer in graph:
            self.update_input_shapes_from_predecessors(layer)
            if (
                layer.op
                in [
                    LayerType.feature_splitter,
                    LayerType.row_splitter,
                    LayerType.width_splitter,
                    LayerType.demux,
                    LayerType.feature_multiplier,
                    LayerType.precision_splitter,
                ]
                or layer.op == LayerType.format_conversion
                and layer.is_nv_converter()
            ):
                layer.output_copies = 1
            else:
                layer.output_copies = self.get_number_of_successors(layer)

            layer.update_output_shapes(hn_stage=self.net_params.stage, validate_shapes=validate_shapes)

    def update_input_shapes_from_predecessors(self, layer):
        predecessors = list(self.predecessors(layer, sort=True))
        if predecessors:
            input_shapes = [[] for _ in predecessors]
            for pred in predecessors:
                index = 0
                if (
                    pred.op
                    in [
                        LayerType.feature_splitter,
                        LayerType.row_splitter,
                        LayerType.width_splitter,
                        LayerType.demux,
                        LayerType.feature_multiplier,
                        LayerType.precision_splitter,
                    ]
                    or pred.op == LayerType.format_conversion
                    and pred.is_nv_converter()
                ):
                    # need to use first index in case of pred layer that already updated output copies
                    if (
                        pred.op == LayerType.feature_multiplier
                        and pred.feature_multiplier_type == FeatureMultiplierType.square
                        and pred.output_copies == 1
                    ):
                        index = 0
                    # need to use successors index field instead of outputs names (order not fixed after from/to_pb)
                    elif pred.output_indices and layer.index in pred.output_indices:
                        index = pred.output_indices.index(layer.index)
                    else:
                        # fallback, use previous heuristic, should be fixed in SDK-19749
                        index = pred.outputs.index(layer.name)

                    # handling splitter layer with multiple outputs at each split
                    if (
                        pred.op == LayerType.feature_splitter
                        and pred.split_sizes
                        and len(pred.split_sizes) < len(pred.outputs)
                    ):
                        index = pred.split_indices[index]

                pred_output_shape = pred.output_shapes[index]
                input_shapes[layer.inputs.index(pred.name)] = pred_output_shape
                self.add_edge(pred, layer, out_shape=pred_output_shape, in_shape=layer.reshape_input(pred_output_shape))
            assert all(shape != [] for shape in input_shapes)
            layer.input_shapes = input_shapes

    def validate_shapes(self):
        for layer in self:
            for succ in self.successors(layer):
                layer_output_shape = self.get_edge_data(layer, succ)["out_shape"]
                succ_input_shape = self.get_edge_data(layer, succ)["in_shape"]
                self._compare_shapes(layer, succ, layer_output_shape, succ_input_shape)

    def get_next_index(self):
        return max(x.index for x in self) + 1

    def set_names_and_indices(self, force=False):
        type_counters = {}
        if (not force) and all(layer.index != INDEX_NOT_SET and layer.name is not None for layer in self.nodes()):
            return

        old_index_to_layer = {}
        old_index_to_old_name = {}
        old_index_to_old_outputs = {}
        # we can't have a stable order easily without relying on indices. We try to rely on names
        # and on insertion order
        sort_key = "name" if all(layer.name is not None for layer in self.nodes()) else "insertion_order"
        for index, layer in enumerate(self.stable_toposort(key=sort_key)):
            if layer.index != INDEX_NOT_SET:
                old_index_to_layer[layer.index] = layer
                old_index_to_old_name[layer.index] = layer.name
                old_index_to_old_outputs[layer.index] = layer.outputs.copy()
            layer.index = index
            self._set_layer_name(type_counters, layer, force=force)

        if len(old_index_to_layer) > 0:
            self._update_connections(old_index_to_layer, old_index_to_old_name, old_index_to_old_outputs)

    def add_scopes(self):
        if len(self.components) == 1:
            names_mapping = self.add_scope_name_to_component(self.components[0], self.name)
            scope_names = [self.name]
        else:
            scope_names = []
            names_mapping = {}
            for scope_index, component in enumerate(self.components, start=1):
                scope_name = f"{self.name}_scope{scope_index}"
                component_names_mapping = self.add_scope_name_to_component(component, scope_name)
                names_mapping.update(component_names_mapping)
                scope_names.append(scope_name)

        self.net_params.net_scopes = scope_names

        return names_mapping

    def add_scope_name_to_component(self, component, scope_name):
        names_mapping = {}
        for layer in component.stable_toposort(key="name"):
            old_name = layer.name
            self._set_layer_scope_name(layer, scope_name)
            names_mapping[old_name] = layer.name

        return names_mapping

    def update_output_indices(self):
        for layer in self.stable_toposort():
            output_indices = [0] * self.get_number_of_successors(layer)
            for succ in self.successors(layer):
                if succ.name in layer.outputs:
                    idx = layer.outputs.index(succ.name)
                    output_indices[idx] = succ.index
            layer.output_indices = output_indices

    def has_postprocess_layer(self):
        return any(layer.op == LayerType.postprocess for layer in self)

    def update_output_layers_order(self, original_end_nodes_order=None):
        has_postprocess_layer = False
        # preparing recipe and looking for post-process node
        recipe = []
        for node in self.stable_toposort(key="name"):
            if node.op == LayerType.postprocess:
                has_postprocess_layer = True
            recipe.append(node)

        output_layers = self.get_real_output_layers_by_recipe(recipe)
        # In case we have only one output we still update this field
        if len(self.net_params.output_layers_order) <= 1:
            if original_end_nodes_order:
                valid_original_end_nodes = []
                for name in original_end_nodes_order:
                    # orig name might contain unsupported chars, like ';', replaced with default
                    valid_name = valid_orig_name(name)
                    valid_original_end_nodes.append(valid_name)

                out_layers_order = []
                handled_nodes = set()
                for node_name in valid_original_end_nodes:
                    for output_layer in output_layers:
                        if node_name in output_layer.original_names and node_name not in handled_nodes:
                            handled_nodes.add(node_name)
                            out_layers_order.append(output_layer.name)

                # Validate we handled all the nodes in end nodes order
                for node_name in valid_original_end_nodes:
                    if node_name not in handled_nodes:
                        raise InvalidHNError(
                            f"The original node name {node_name} in end_node_names is missing in the HN.",
                        )

                self.net_params.output_layers_order = out_layers_order
            else:
                if self.net_params.output_layers_order and len(output_layers) > 1:
                    raise InvalidHNError(
                        "There are multiple outputs found in the HN, but only one layer in output_layers_order.",
                    )
                self.net_params.output_layers_order = [layer.name for layer in output_layers]
        elif not has_postprocess_layer:
            if len(self.net_params.output_layers_order) != len(output_layers):
                raise InvalidHNError(
                    f"There are {len(output_layers)} output layers in the HN, but "
                    f"{len(self.net_params.output_layers_order)} layers in output_layers_order.",
                )
            output_layers_names = [layer.name for layer in output_layers]
            for output_layer_name in output_layers_names:
                if output_layer_name not in self.net_params.output_layers_order:
                    raise InvalidHNError(
                        f"Output layer {output_layer_name} found in the HN is not in output_layers_order.",
                    )

    @property
    def components(self):
        self._all_components = [self.subgraph(comp) for comp in nx.weakly_connected_components(self)]
        return self._all_components

    def get_component_by_scope(self, scope):
        possible_components_by_scope = []
        for component in self.components:
            if any(node.scope == scope for node in component.nodes):
                possible_components_by_scope.append(component)
        if len(possible_components_by_scope) > 1:
            raise HailoNNException(f"Found multiple components for scope {scope}.")
        if len(possible_components_by_scope) == 1:
            return possible_components_by_scope[0]

        return None

    def is_output_scale_per_channel(self):
        return any(layer.output_scale_per_channel for layer in self.get_output_layers())

    def is_transposed(self):
        return all(layer.transposed for layer in self.get_input_layers() + self.get_output_layers())

    def update_io_layers_transposed(self):
        if self.net_params.transposed_net:
            for layer in self.get_input_layers() + self.get_output_layers():
                layer.transposed = True

    def transpose_layers_height_width(self, input_layers=None):
        if not input_layers:
            self.net_params.transposed_net = True
            components = self.components
        else:
            components = []
            for component in self.components:
                component_inputs = [node for node in component if component.in_degree(node) == 0]
                inputs_to_transpose = [input_node for input_node in component_inputs if input_node.name in input_layers]
                if not inputs_to_transpose:
                    continue
                if inputs_to_transpose != component_inputs:
                    raise HailoNNException(
                        f"{component_inputs} are in the same connected component, you can either transpose all or none",
                    )
                components.append(component)

        for component in components:
            for layer in component:
                preds = list(self.predecessors(layer))
                if layer.op == LayerType.nms or (layer.op == LayerType.output_layer and preds[0].op == LayerType.nms):
                    continue
                if layer.op == LayerType.concat:
                    concat_preds = preds
                    while concat_preds[0].op == LayerType.concat:
                        concat_preds = list(self.predecessors(concat_preds[0]))
                    if concat_preds[0].op == LayerType.proposal_generator:
                        continue

                if layer.op in [LayerType.space_to_depth, LayerType.depth_to_space]:
                    raise UnsupportedModelError(f"Layer {layer.full_name_msg} cannot be transposed")
                if layer.op == LayerType.proposal_generator:
                    layer.input_shapes = [shape[:1] + [shape[2], shape[1]] + [shape[3]] for shape in layer.input_shapes]
                    continue
                if hasattr(layer, "spatial_flatten_output") and layer.spatial_flatten_output:
                    raise UnsupportedModelError(f"Layer {layer.name} with spatial flatten output cannot be transposed")
                if hasattr(layer, "transpose_output_width_features") and layer.transpose_output_width_features:
                    raise UnsupportedModelError(
                        f"Layer {layer.name} with transpose output width features cannot be transposed",
                    )

                if layer.op == LayerType.concat and hasattr(layer, "axis") and layer.axis != ConcatAxis.features:
                    if layer.axis == ConcatAxis.spatial_w:
                        layer.axis = ConcatAxis.spatial_h
                    else:
                        layer.axis = ConcatAxis.spatial_w
                if layer.op == LayerType.slice and hasattr(layer, "height_slice") and hasattr(layer, "width_slice"):
                    layer.height_slice, layer.width_slice = layer.width_slice, layer.height_slice

                layer.output_shapes = [
                    shape[:1] + [shape[2], shape[1]] + [shape[3]] if len(shape) == 4 else shape
                    for shape in layer.output_shapes
                ]
                layer.input_shapes = [
                    shape[:1] + [shape[2], shape[1]] + [shape[3]] if len(shape) == 4 else shape
                    for shape in layer.input_shapes
                ]
                if layer.op != LayerType.normalization and hasattr(layer, "kernel") and len(layer.kernel_shape) == 4:
                    layer.kernel_shape = [layer.kernel_shape[1], layer.kernel_shape[0]] + layer.kernel_shape[2:]
                if layer.op == LayerType.avgpool and not layer.is_global_avg_pool() and len(layer.kernel_shape) == 4:
                    layer.kernel_shape = [
                        layer.kernel_shape[0],
                        layer.kernel_shape[2],
                        layer.kernel_shape[1],
                        layer.kernel_shape[3],
                    ]

                if hasattr(layer, "dilations") and layer.dilations is not None and layer.dilations[1:3] != [1, 1]:
                    layer.dilations = [1, layer.dilations[2], layer.dilations[1], 1]
                if hasattr(layer, "strides") and layer.strides is not None and layer.strides[1:3] != [1, 1]:
                    layer.strides = [1, layer.strides[2], layer.strides[1], 1]
                if hasattr(layer, "h_ratios") and hasattr(layer, "w_ratios"):
                    layer.h_ratios, layer.w_ratios = layer.w_ratios, layer.h_ratios
                if hasattr(layer, "block_sizes") and layer.block_sizes is not None:
                    layer.block_sizes = layer.block_sizes[::-1]
                if hasattr(layer, "spatial_reshape_sizes") and layer.spatial_reshape_sizes is not None:
                    layer.spatial_reshape_sizes = layer.spatial_reshape_sizes[::-1]
                if layer.transposed:
                    raise UnsupportedModelError(f"Cant transposed already transposed layer {layer.name}")
                layer.transposed = True

    def _update_connections(self, old_index_to_layer, old_index_to_old_name, old_index_to_old_outputs):
        for old_index, layer in old_index_to_layer.items():
            old_name = old_index_to_old_name[old_index]
            # create a copy of the old input indices so, they won't change mid-loop
            old_input_indices = layer.input_indices[:]
            layer.output_indices = [old_index_to_layer[idx].index for idx in layer.output_indices]
            for old_input_index in old_input_indices:
                input_layer = old_index_to_layer[old_input_index]
                self._update_connection(
                    layer,
                    input_layer,
                    old_name,
                    old_input_index,
                    old_input_indices,
                    old_index_to_old_outputs,
                )

    def _update_connection(
        self,
        layer,
        new_pred,
        old_layer_name,
        old_pred_index,
        old_input_indices,
        old_index_to_old_outputs,
    ):
        self._update_layer_inputs(layer, new_pred, old_pred_index, old_input_indices)
        new_pred.outputs[old_index_to_old_outputs[old_pred_index].index(old_layer_name)] = layer.name

    @staticmethod
    def _update_layer_inputs(layer, new_pred, old_pred_index, old_input_indices):
        input_placement = old_input_indices.index(old_pred_index)
        layer.input_indices[input_placement] = new_pred.index
        layer.inputs[input_placement] = new_pred.name

    def insert_layers(self, new_layers_to_succs):
        for new_layer in new_layers_to_succs:
            self.add_node(new_layer)

        self.set_names_and_indices()
        self.update_output_indices()

        for new_layer, succ in new_layers_to_succs.items():
            for pred in list(self.predecessors(succ)):
                self.remove_edge(pred, succ)
                self.add_edge(pred, new_layer)
                self._update_layer_inputs(
                    succ,
                    new_pred=new_layer,
                    old_pred_index=pred.index,
                    old_input_indices=succ.input_indices,
                )
                new_layer.inputs.extend([pred.name])
                new_layer.input_indices.extend([pred.index])
                pred.outputs = [new_layer.name if output == succ.name else output for output in pred.outputs]
                pred.output_indices = [new_layer.index if idx == succ.index else idx for idx in pred.output_indices]

            new_layer.outputs = [succ.name]
            self.add_edge(new_layer, succ)
            new_layer.input_shapes = succ.input_shapes
            new_layer.update_output_shapes()
            succ.input_shapes = new_layer.output_shapes

            if not new_layer.scope:
                self._set_layer_scope_name(new_layer, succ.scope)

        for layer in self:
            layer.input_indices = [self.get_layer_by_name(inp_layer).index for inp_layer in layer.inputs]

    def push_layer(self, layer, preds, calc_shapes=True, connect_succs=True):
        self.add_node(layer)
        self.set_names_and_indices()
        self.update_output_indices()

        for pred in preds:
            if connect_succs:
                succs = list(self.successors(pred))
                for successor in succs:
                    self.remove_edge(pred, successor)
                    self.add_edge(layer, successor)
                    self._update_layer_inputs(
                        successor,
                        new_pred=layer,
                        old_pred_index=pred.index,
                        old_input_indices=successor.input_indices,
                    )

                self.update_input_lists(layers=succs)

            self.add_edge(pred, layer)
            layer.inputs.extend([pred.name])
            layer.input_indices.extend([pred.index])
            if connect_succs:
                layer.outputs.extend(pred.outputs)
                pred.outputs = [layer.name]
            else:
                pred.outputs.append(layer.name)

        if calc_shapes:
            self.calculate_shapes()

        if not layer.scope:
            self._set_layer_scope_name(layer, preds[0].scope)

    def remove_null_layers(self, null_layers=None, validate_shapes=True):
        if not null_layers:
            null_layers = [layer for layer in self if layer.op == LayerType.null]

        for null_layer in null_layers:
            preds = list(self.predecessors(null_layer))
            if len(preds) != 1:
                raise UnsupportedModelError(f"{null_layer.full_name_msg} should have exactly one predecessor")

            pred = preds[0]
            succs = list(self.successors(null_layer))

            if pred.op in [LayerType.feature_splitter, LayerType.spatial_splitter, LayerType.width_splitter] and (
                len(succs) > 1 or set(null_layer.outputs).intersection(set(pred.outputs))
            ):
                continue

            self.remove_edge(pred, null_layer)
            for original_name in null_layer.original_names:
                pred.add_original_name(original_name)

            for succ in succs:
                self.add_edge(pred, succ)
                self.remove_edge(null_layer, succ)
                succ.replace_input_layer(null_layer.name, pred.name)
                succ.replace_input_index(null_layer.index, pred.index)
                if null_layer.name in pred.outputs:
                    pred.replace_output_layer(null_layer.name, succ.name)
                else:
                    pred.append_output_layer(succ.name)
                    pred.append_output_shape(null_layer.output_shape)
                if null_layer.index in pred.output_indices:
                    pred.replace_output_index(null_layer.index, succ.index)
                else:
                    pred.append_output_index(succ.index)

            self.remove_node(null_layer)

        self.calculate_shapes(validate_shapes=validate_shapes)

    def remove_layer(self, layer):
        for input_name in layer.inputs:
            input_layer = self.get_layer_by_name(input_name)
            if layer.index in input_layer.output_indices:
                index = input_layer.output_indices.index(layer.index)
                del input_layer.outputs[index]
                del input_layer.output_indices[index]
                del input_layer.output_shapes[index]
            if self.has_edge(input_layer, layer):
                self.remove_edge(input_layer, layer)

        for output_name in layer.outputs:
            output_layer = self.get_layer_by_name(output_name)
            if layer.index in output_layer.input_indices:
                index = output_layer.input_indices.index(layer.index)
                del output_layer.inputs[index]
                del output_layer.input_indices[index]
                del output_layer.input_shapes[index]
            if self.has_edge(layer, output_layer):
                self.remove_edge(layer, output_layer)

        self.remove_node(layer)
        if layer.scope is not None and all(layer.scope != other.scope for other in self):
            self.net_params.net_scopes.remove(layer.scope)

    def replace_layer(self, old_layer, new_layer):
        preds = list(self.predecessors(old_layer))
        succs = list(self.successors(old_layer))
        self.remove_layer(old_layer)
        self.push_layer(new_layer, preds, False)

        for succ in succs:
            self.add_edge(new_layer, succ)
            succ.inputs.append(new_layer.name)
            new_layer.output_indices.append(succ.index)
            succ.input_indices.append(new_layer.index)

        for pred in preds:
            pred.update_output_shapes()
            pred.output_indices.append(new_layer.index)

        new_layer.inputs = old_layer.inputs
        new_layer.outputs = old_layer.outputs

    def adjust_new_layer_input_output(self, old_layer, new_layer):
        for input in old_layer.inputs:
            input_layer = self.get_layer_by_name(input)
            input_layer.outputs.append(new_layer.name)
            input_layer.output_shapes.extend(new_layer.input_shapes)

            self.add_edge(input_layer, new_layer)
            if self.has_edge(input_layer, old_layer):
                self.remove_edge(input_layer, old_layer)

        for output in old_layer.outputs:
            output_layer = self.get_layer_by_name(output)
            output_layer.input_indices.append(new_layer.index)
            output_layer.inputs.append(new_layer.name)
            output_layer.input_shapes.extend(new_layer.output_shapes)

            self.add_edge(new_layer, output_layer)
            if self.has_edge(old_layer, output_layer):
                self.remove_edge(old_layer, output_layer)

    def relax_new_layer_into_graph(self, new_layer, successors_meta_data=None):
        preds = list(self.predecessors(new_layer))
        succs = list(self.successors(new_layer))

        for pred in preds:
            if pred.index not in new_layer.input_indices:
                new_layer.append_input_index(pred.index)
            if pred.name not in new_layer.inputs:
                new_layer.append_input_layer(pred.name)
            if new_layer.name not in pred.outputs:
                pred.outputs.append(new_layer.name)
                pred_index = new_layer.input_indices.index(pred.index)
                pred.output_shapes.append(new_layer.input_shapes[pred_index])
            if new_layer.index not in pred.output_indices:
                pred.output_indices.append(new_layer.index)

        for succ in succs:
            if successors_meta_data and succ.name in successors_meta_data:
                succ.inputs = successors_meta_data[succ.name]["inputs"]
                succ.input_indices = successors_meta_data[succ.name]["input_indices"]
                if successors_meta_data[succ.name]["input_shapes"]:
                    succ.input_shapes = successors_meta_data[succ.name]["input_shapes"]
            if new_layer.index not in succ.input_indices:
                succ.input_indices.append(new_layer.index)
            if succ.index not in new_layer.output_indices:
                new_layer.append_output_index(succ.index)
            if succ.name not in new_layer.outputs:
                new_layer.append_output_layer(succ.name)
            if new_layer.index not in succ.input_indices:
                succ.input_indices.append(new_layer.index)
            if new_layer.name not in succ.inputs:
                succ.inputs.append(new_layer.name)
                index = new_layer.output_indices.index(succ.index)
                succ.input_shapes.append(new_layer.output_shapes[index])

    def _set_layer_name(self, type_counters, layer, force=False):
        hn_name = layer.hn_name
        if hn_name not in type_counters:
            type_counters[hn_name] = 1
        if force or (layer.name is None):
            # if op is missing in TYPE_TO_HN_NAME dict, the default value is the op itself
            new_name_base = hn_name
            new_name = None
            # if we force, we don't check for duplicates because we will reset all names anyway
            while (new_name is None) or (
                (not force)
                and (
                    (any(layer.name == f"{layer.scope}/{new_name}" for layer in self.nodes()))
                    or (any(layer.name == new_name for layer in self.nodes()))
                )
            ):
                new_name = f"{new_name_base}{type_counters[hn_name]}"
                type_counters[hn_name] += 1

            layer.name = new_name

    def validate_stage(self, expected_stage):
        if self.net_params.stage != expected_stage:
            raise InvalidHNStage(f"Unexpected HN stage. Expected {expected_stage} but received {self.net_params.stage}")

    def to_hn(self, network_name, npz_path=None, json_dump=True, should_get_default_params=False):
        """
        Export Hailo model to JSON format (HN) and params NPZ file. The NPZ is saved to a file.

        Args:
            network_name (str): Name of the network.
            npz_path (str, optional): Path to save the parameters in NPZ format. If it is None, no
                file is saved. Defaults to None.
            json_dump (bool, optional): Indicates whether to dump the HN to a formatted JSON, or
                leave it as a dictionary. Defaults to True, which means to dump.
            should_get_default_params (bool, optional): Indicates whether the HN should include
                fields with default values. Defaults to False, which means they will not be
                included.

        Returns:
            The HN, as a string or a dictionary, depending on the ``json_dump`` argument.

        """
        exporter = HNExporter(self)
        hn_data = exporter.get_hn(network_name, json_dump, should_get_default_params)
        if npz_path is not None:
            exporter.save_params_npz(network_name, npz_path)
        return hn_data

    def to_hn_npz(self, network_name, json_dump=True, should_get_default_params=False):
        """
        Export Hailo model to JSON format (HN) and params NPZ file. The NPZ is returned to the
        caller.

        Args:
            network_name (str): Name of the network.
            json_dump (bool, optional): Indicates whether to dump the HN into a formatted JSON, or
                leave it as a dictionary. Defaults to True, which means to dump.
            should_get_default_params (bool, optional): Indicates whether the HN should include
                fields with default values. Defaults to False, which means they will not be
                included.

        Returns:
            tuple: The first item is the HN, as a string or a dictionary, depending on the
            ``json_dump`` argument. The second item contains the model's parameters as a dictionary.

        """
        exporter = HNExporter(self)
        hn_data = exporter.get_hn(network_name, json_dump, should_get_default_params)
        npz_data = exporter.get_params_npz(network_name)
        return hn_data, npz_data

    def to_pb(self, pb_wrapper, pb_path=None):
        """Export Hailo model to a Protobuf file."""
        if pb_path is None:
            return pb_wrapper.protobuf_exporter(self).to_pb()

        return pb_wrapper.protobuf_exporter(self).save_pb(pb_path)

    def set_compilation_params(self, compilation_params):
        for layer in self:
            layer.set_compilation_params(**compilation_params)

    # NOTE: doesn't seem to ever be used, maybe remove?!
    def summary(self):
        headers = [
            "Layer",
            "Type",
            "Input",
            "Kernel",
            "Features",
            "Groups",
            "Dilation",
            "Weights",
            "Ops",
            "Connected To",
            "Activation",
            "Padding",
            "Batch\nNorm",
        ]
        lines = []

        for layer in self.stable_toposort():
            op = layer.op.value
            if layer.op == LayerType.resize:
                op += " (NN)" if layer.resize_method == ResizeMethod.nearest_neighbor else " (BL)"

            if layer.op in [LayerType.concat, LayerType.output_mux, LayerType.proposal_generator, LayerType.matmul]:
                input_shapes = ", ".join(f"{input_shape[2]} x {input_shape[1]}" for input_shape in layer.input_shapes)
                features = (
                    " | ".join(f"{input_shape[-1]}" for input_shape in layer.input_shapes)
                    + f" -> {layer.output_features}"
                )
            elif len(layer.input_shape) >= 3 and len(layer.output_shape) >= 3:
                input_shapes = f"{layer.input_width} x {layer.input_height}"
                features = f"{layer.input_features} -> {layer.output_features}"
            else:
                input_shapes = ""
                features = ""

            kernel = (
                f"{layer.kernel_width} x {layer.kernel_height} / {layer.stride_height}"
                if all(hasattr(layer, attr) for attr in ["kernel_width", "kernel_height", "stride_height"])
                else ""
            )

            lines += [
                [
                    layer.name,
                    op,
                    input_shapes,
                    kernel,
                    features,
                    layer.groups if hasattr(layer, "groups") else "",
                    layer.dilations[1]
                    if hasattr(layer, "dilations") and layer.dilations and len(layer.dilations) > 1
                    else "",
                    int(layer.weights // 1000),
                    old_div(layer.ops, 1.0e9),
                    ", ".join(layer.outputs),
                    layer.activation.value if hasattr(layer, "activation") else "",
                    (
                        layer.padding.value.lower().replace("tensorflow", "tf")
                        if hasattr(layer, "padding") and layer.padding
                        else ""
                    ),
                    layer.bn_enabled,
                ],
            ]

        self._logger.info(
            "\n" + tabulate(lines, headers, showindex="always", floatfmt=".2f", numalign="left", tablefmt="psql"),
        )

    @property
    def layers_by_index(self):
        return {n.index: n for n in self.nodes}

    @property
    def requires_native_weights(self):
        return any(layer.requires_native_weights for layer in self.nodes)

    @property
    def requires_quantized_weights(self):
        return any(layer.requires_quantized_weights for layer in self.nodes)

    def get_layer_name_by_index(self, index):
        return self.layers_by_index[index].name

    def _get_layer_by_original_name(self, vertex_name, scope_name=None, input_only=True):
        possible_layers_by_scope = {}
        layers = self if not input_only else self.get_input_layers()
        for layer in layers:
            # orig name might contain unsupported chars, like ';', replace with default
            vertex_name = valid_orig_name(vertex_name)
            if vertex_name in layer.original_names:
                if (scope_name and scope_name == layer.scope) or not scope_name:
                    if layer.scope in possible_layers_by_scope:
                        possible_layers_by_scope[layer.scope].append(layer)
                    else:
                        possible_layers_by_scope[layer.scope] = [layer]
        if len(possible_layers_by_scope) > 1:
            raise HailoNNException(
                f"Found multiple layers for original vertex {vertex_name}. Please specify scope to "
                f"find the relevant match",
            )
        if len(possible_layers_by_scope) == 1:
            return next(iter(possible_layers_by_scope.values()))[-1]

        return None

    def get_layers_by_type(self, layer_type):
        return [layer for layer in self if layer.op == layer_type]

    def get_layer_by_original_name(self, vertex_name, scope_name=None):
        return self._get_layer_by_original_name(vertex_name, scope_name=scope_name, input_only=False)

    def get_input_layer_by_original_name(self, vertex_name, scope_name=None):
        return self._get_layer_by_original_name(vertex_name, scope_name=scope_name, input_only=True)

    def update_signed_output(self):
        # Should be used only for legacy quantized hars
        # or in the beginning of legacy quantization flow
        for layer in self:
            if layer.precision_config.signed_output is not None:
                continue
            signed_output = False
            if layer.outputs:
                succ = self.get_layer_by_name(layer.outputs[0])
                if succ.dynamic_weights and succ.inputs[1] == layer.name:
                    if layer.op not in [
                        LayerType.conv,
                        LayerType.dw,
                        LayerType.slice,
                        LayerType.feature_splitter,
                        LayerType.normalization,
                        LayerType.ew_add,
                        LayerType.ew_sub,
                        LayerType.activation,
                    ]:
                        raise UnsupportedModelError(
                            f"Data driven weighted layers support only convolution layers "
                            f"as inputs, but {layer.full_name_msg} is of type {layer.op}",
                        )
                    if layer.op != LayerType.feature_splitter and len(list(self.successors(layer))) != 1:
                        raise UnsupportedModelError(
                            f"The layer {layer.full_name_msg}, has multiple outputs, despite being matmul layer input",
                        )
                    signed_output = True
            layer.precision_config.signed_output = signed_output

    def set_input_tensors_shapes(self, inputs_shapes):
        """
        Set the tensor shape (resolution) for each input layer.

        Args:
            inputs_shapes (dict): Each key is a name of an input layer, and each value is the new
                shape to assign to it. Currently doesn't support changing number of features.

        """
        input_layers = self.get_input_layers()

        for layer_name in inputs_shapes:
            layer = self.get_layer_by_name(layer_name)
            if layer not in input_layers:
                raise HailoNNException(
                    f"set_input_tensors_shapes expects input layer shapes and {layer.full_name_msg} "
                    f"is not an input layer",
                )

            inputs_shapes[layer_name][0] = -1
            if inputs_shapes[layer_name][-1] != layer.input_shape[-1]:
                raise HailoNNException(
                    f"Currently, set_input_tensors_shapes doesnt support changing number of "
                    f"features. In {layer.full_name_msg}, existing number of features is "
                    f"{layer.input_shape[-1]} and got {inputs_shapes[layer_name][-1]}",
                )
            layer.input_shape = inputs_shapes[layer_name]

        self.calculate_shapes()

    def _convert_old_precision_mode_to_new(self):
        """
        Converts old precision modes to new precision modes for each layer in the model.
        This method iterates over all layers in the model and updates their precision modes
        from the format aX_wY to aX_wY_aZ, which includes the output precision as well.
        If a layer has no successors, it updates its precision mode directly. If a layer has
        successors, it updates its precision mode based on the input precision mode of the
        first successor.
        Raises:
            AssertionError: If any layer has no precision mode or if any layer retains an old precision mode.
        """

        for layer in self:
            successors = list(self.successors(layer))
            if not successors:
                if layer.precision_config.precision_mode == PrecisionMode.a8_w8:
                    layer.precision_config.precision_mode = PrecisionMode.a8_w8_a8
                elif layer.precision_config.precision_mode == PrecisionMode.a16_w16:
                    layer.precision_config.precision_mode = PrecisionMode.a16_w16_a16
            else:
                successor = successors[0]
                successor_precision_mode = successor.precision_config.precision_mode
                if successor_precision_mode.input_precision_mode() == PrecisionMode.a8_w8_a8:
                    if layer.precision_config.precision_mode == PrecisionMode.a8_w8:
                        layer.precision_config.precision_mode = PrecisionMode.a8_w8_a8
                    elif layer.precision_config.precision_mode == PrecisionMode.a16_w16:
                        layer.precision_config.precision_mode = PrecisionMode.a16_w16_a8
                    elif layer.precision_config.precision_mode == PrecisionMode.a8_w4:
                        layer.precision_config.precision_mode = PrecisionMode.a8_w4_a8
                elif successor_precision_mode.input_precision_mode() == PrecisionMode.a16_w16_a16:
                    if layer.precision_config.precision_mode == PrecisionMode.a8_w8:
                        layer.precision_config.precision_mode = PrecisionMode.a8_w8_a16
                    elif layer.precision_config.precision_mode == PrecisionMode.a16_w16:
                        layer.precision_config.precision_mode = PrecisionMode.a16_w16_a16
                    elif layer.precision_config.precision_mode == PrecisionMode.a8_w4:
                        layer.precision_config.precision_mode = PrecisionMode.a8_w4_a16

        for layer in self:
            if layer.precision_config.precision_mode is None:
                assert False, f"Layer {layer.name} has no precision mode"
            if layer.precision_config.precision_mode in [
                PrecisionMode.a8_w8,
                PrecisionMode.a8_w4,
                PrecisionMode.a16_w16,
                PrecisionMode.a16_w8,
                PrecisionMode.a16_w4,
                PrecisionMode.a8_w4_exp,
            ]:
                assert False, f"Layer {layer.name} has old precision mode {layer.precision_config.precision_mode}"

    def fill_default_quantization_params(self, disable_warning=False, logger=None):
        if logger is None:
            logger = default_logger()
        for layer in self:
            if (
                layer.precision_config.precision_mode is None
                or layer.precision_config.bias_mode is None
                or layer.precision_config.quantization_groups is None
            ):
                if not disable_warning and layer.engine != PostprocessTarget.CPU:
                    logger.deprecation_warning(
                        f"Layer {layer.name} has implicit precision config during compilation, filling default values",
                    )
                if layer.precision_config.bias_mode is None and layer.op.value in {"conv", "deconv", "dense"}:
                    layer.precision_config.bias_mode = BiasMode.single_scale_decomposition
                hn_element = layer.to_hn()
                layer_cfg = layer.precision_config
                try:
                    layer_class = get_layer_type_from_hn_element(hn_element, DEFAULT_OPTIMIZATION_TARGET)
                except AccelerasImplementationError:
                    if layer_cfg.precision_mode is None:
                        layer_cfg.precision_mode = "a8_w8"
                    if layer_cfg.bias_mode is None:
                        layer_cfg.bias_mode = "single_scale_decomposition"
                    if layer_cfg.quantization_groups is None:
                        layer_cfg.quantization_groups = 1
                else:
                    # The precision_config is a property which copies the original values.
                    # Therefore we need to set the values explicitly
                    layer_cfg.fill_default_config(layer_class)

        self._convert_old_precision_mode_to_new()

    def get_per_layer_precision_config(self):
        mo_config_precision = {}
        precision_config = {
            layer.name: layer.precision_config.raw_dict() for layer in self if layer.precision_config.raw_dict()
        }
        if precision_config:
            mo_config_precision["precision_config"] = {"layers": precision_config}
        return mo_config_precision

    def get_per_layer_translation_config(self):
        mo_config_translation = {}
        translation_config = {
            layer.name: layer.translation_config.raw_dict() for layer in self if layer.translation_config.raw_dict()
        }
        if translation_config:
            mo_config_translation["translation_config"] = {"layers": translation_config}
        return mo_config_translation

    @staticmethod
    def from_fp(fp):
        """Get Hailo model from a file."""
        hn_json = fp.read()
        return HNImporter().from_hn(hn_json)

    @staticmethod
    def from_hn(hn_json):
        """Get Hailo model from HN raw JSON data."""
        return HNImporter().from_hn(hn_json)

    @staticmethod
    def from_parsed_hn(hn_json, validate=True):
        """Get Hailo model from HN dictionary."""
        return HNImporter().from_parsed_hn(hn_json, validate)

    @staticmethod
    def from_integrated_pb_file(pb_path, pb_wrapper):
        """Get Hailo model from Protobuf serialization."""
        return pb_wrapper.protobuf_importer().from_integrated_pb_file(pb_path)

    @staticmethod
    def from_integrated_pb_data(pb_data, pb_wrapper):
        """Get Hailo model from Protobuf serialization."""
        return pb_wrapper.protobuf_importer().from_integrated_pb_data(pb_data)

    @staticmethod
    def from_pb_graph(pb_graph, pb_wrapper):
        return pb_wrapper.protobuf_importer().from_pb_graph(pb_graph)

    @staticmethod
    def from_pb(pb_path, pb_wrapper):
        """Get Hailo model from Protobuf serialization."""
        return pb_wrapper.protobuf_importer().from_pb(pb_path)

    @staticmethod
    def from_pb_string(pb_string, pb_wrapper, hailo_nn_before_fusing=None):
        """Get Hailo model from serialized Protobuf."""
        return pb_wrapper.protobuf_importer().from_pb_string(pb_string, hailo_nn_before_fusing)

    @staticmethod
    def get_valid_input_identifier(input_identifier, arg_name, warn_user=True):
        """Get valid net/scope name adhering to the alphanums/hyphen/underscore limitation."""
        if input_identifier is None:
            return input_identifier

        allowed_chars_pattern = "^[A-Za-z0-9/_-]+$"
        new_val = copy.copy(input_identifier).replace(".", "_")
        if new_val != input_identifier and warn_user:
            default_logger().info(
                f"Found a '.' character in {arg_name}, which isn't supported. New {arg_name} is {new_val}",
            )

        if len(new_val) > 0 and new_val[0].isdigit():
            new_val = f"net_{new_val}"
            default_logger().info(
                f"Found a {arg_name} that starts with a digit, added a prefix. New {arg_name} is {new_val}",
            )

        validated = bool(re.match(allowed_chars_pattern, new_val))
        if not validated:
            raise UnsupportedModelError(
                f"Provided {arg_name} {new_val} has illegal characters. Supported identifiers "
                f"consist of alphanumeric chars, hyphens and underscores.",
            )
        return new_val

    @property
    def is_multi_scope(self):
        return len(self.net_params.net_scopes) > 1

    @property
    def is_single_scope(self):
        return len(self.net_params.net_scopes) == 1

    def update_input_lists(self, layers=None):
        layers = layers if layers else list(self)
        for layer in layers:
            if layer.op in [
                LayerType.ew_add,
                LayerType.ew_sub,
                LayerType.ew_mult,
                LayerType.ew_div,
                LayerType.concat,
                LayerType.output_mux,
                LayerType.bbox_decoder,
                LayerType.proposal_generator,
                LayerType.fused_bbox_decoder,
                LayerType.ew_max,
                LayerType.ew_min,
            ]:
                layer.input_list = []
                for input_index in layer.input_indices:
                    layer.append_to_input_list(self.layers_by_index[input_index])

    @property
    def detected_anchors(self):
        return self._detected_anchors

    @detected_anchors.setter
    def detected_anchors(self, anchors):
        self._detected_anchors = anchors

    def update_detected_anchors_info(self):
        if self.detected_anchors:
            meta_arch = self.detected_anchors["meta_arch"]
            if meta_arch.value in ["yolov5", "yolov5_seg"]:  # has anchors
                for orig_name in self.detected_anchors["info"]:
                    hn_name = self.get_layer_by_original_name(orig_name).name
                    hn_name = hn_name.split("/")[1]
                    self.detected_anchors["info"][orig_name].update({"encoded_layer": hn_name})

    @staticmethod
    def update_successors_meta_data(succ, successors_meta_data):
        successors_meta_data.update(
            {
                succ.name: {
                    "inputs": succ.inputs.copy(),
                    "input_indices": succ.input_indices.copy(),
                    "input_shapes": succ.input_shapes.copy(),
                },
            },
        )

    @property
    def blocks(self):
        return self._blocks

    @blocks.setter
    def blocks(self, blocks):
        self._blocks = blocks

    def extract_parsing_report_blocks(self):
        for layer in self.stable_toposort():
            if layer.block_info:
                block_type, block_name = layer.block_info
                if block_type not in self.blocks:
                    self.blocks[block_type] = {}

                if block_name in self.blocks[block_type]:
                    self.blocks[block_type][block_name]["block_layers"].append(layer.name)
                else:
                    self.blocks[block_type][block_name] = {
                        "block_layers": [layer.name],
                        "original_names": layer.original_names,
                    }


class HailoNNImporter:
    def __init__(self):
        self._hailo_nn = None


class HNImporter(HailoNNImporter):
    TYPE_TO_CLASS = {
        LayerType.activation.value: FusedStandaloneActivationLayer,
        LayerType.base_activation.value: ActivationLayer,
        LayerType.batch_norm.value: FusedBatchNormLayer,
        LayerType.base_batch_norm.value: BatchNormLayer,
        LayerType.base_conv.value: Conv2DLayer,
        LayerType.base_dw.value: Conv2DLayer,
        LayerType.base_deconv.value: Conv2DLayer,
        LayerType.base_dense.value: DenseLayer,
        LayerType.conv.value: FusedConv2DLayer,
        LayerType.dw.value: FusedConv2DLayer,
        LayerType.deconv.value: FusedConv2DLayer,
        LayerType.dense.value: FusedDenseLayer,
        LayerType.global_avg_pool.value: GlobalAvgPoolingLayer,
        LayerType.input_layer.value: InputLayer,
        LayerType.output_layer.value: OutputLayer,
        LayerType.avgpool.value: PoolingLayer,
        LayerType.maxpool.value: PoolingLayer,
        LayerType.concat.value: ConcatLayer,
        LayerType.output_mux.value: OutputMuxLayer,
        LayerType.proposal_generator.value: ProposalGeneratorLayer,
        LayerType.loopback.value: LoopbackLayer,
        LayerType.shortcut.value: ShortcutLayer,
        LayerType.external_pad.value: ExternalPadLayer,
        LayerType.resize.value: ResizeLayer,
        LayerType.portal.value: PortalLayer,
        LayerType.feature_interleave.value: FeatureInterleaveLayer,
        LayerType.format_conversion.value: FormatConversionLayer,
        LayerType.external_input_layer.value: ExternalInputLayer,
        LayerType.external_output_layer.value: ExternalOutputLayer,
        LayerType.pp_input_layer.value: PPInputLayer,
        LayerType.pp_output_layer.value: PPOutputLayer,
        LayerType.depth_to_space.value: DepthToSpaceLayer,
        LayerType.normalization.value: NormalizationLayer,
        LayerType.argmax.value: ArgmaxLayer,
        LayerType.feature_shuffle.value: FeatureShuffleLayer,
        LayerType.softmax.value: SoftmaxLayer,
        LayerType.nms.value: NMSLayer,
        LayerType.bbox_decoder.value: BboxDecoderLayer,
        LayerType.feature_splitter.value: FeatureSplitterLayer,
        LayerType.spatial_splitter.value: SpatialSplitterLayer,
        LayerType.slice.value: FusedSliceLayer,
        LayerType.bias_add.value: BiasAddLayer,
        LayerType.row_splitter.value: RowSplitterLayer,
        LayerType.width_splitter.value: WidthSplitterLayer,
        LayerType.base_slice.value: SliceLayer,
        LayerType.reduce_max.value: ReduceMaxLayer,
        LayerType.reduce_min.value: ReduceMinLayer,
        LayerType.space_to_depth.value: SpaceToDepthLayer,
        LayerType.demux.value: DemuxLayer,
        LayerType.matmul.value: MatmulLayer,
        LayerType.reduce_sum.value: ReduceSumLayer,
        LayerType.feature_multiplier.value: FeatureMultiplierLayer,
        LayerType.base_ew_add.value: EWAddLayer,
        LayerType.ew_add.value: FusedStandaloneEWAddLayer,
        LayerType.base_ew_sub.value: EWSubLayer,
        LayerType.ew_sub.value: FusedStandaloneEWSubLayer,
        LayerType.ew_mult.value: EWMultLayer,
        LayerType.ew_div.value: EWDivLayer,
        LayerType.ew_max.value: EWMaxLayer,
        LayerType.ew_min.value: EWMinLayer,
        LayerType.const_input.value: ConstInputLayer,
        LayerType.postprocess.value: PostprocessLayer,
        LayerType.reduce_mean.value: ReduceMeanLayer,
        LayerType.precision_splitter.value: PrecisionSplitterLayer,
        LayerType.precision_splitter_signed.value: PrecisionSplitterSignedLayer,
        LayerType.layer_normalization.value: LayerNormalizationLayer,
        LayerType.fused_bbox_decoder.value: FusedBboxDecoderLayer,
    }

    def __init__(self):
        super().__init__()
        self._layers = {}
        self._hn = None

    @staticmethod
    def _load_schema():
        schema_path = SDKPaths().join_sdk_common("hailo_nn/json_schemas/hn.schema.json")
        with open(schema_path) as hn_schema:
            return jsonref.load(hn_schema, base_uri=f"file:{schema_path}", jsonschema=True)

    def from_parsed_hn(self, hn_json, validate=True):
        hn_json = copy.deepcopy(hn_json)
        hn_json.pop("direct_control", None)
        if validate:
            hn_schema = self._load_schema()
            jsonschema.validate(hn_json, hn_schema)

        hn_json, net_name = self._validate_input_identifiers(hn_json)

        self._hailo_nn = HailoNN(net_name)
        self._hailo_nn.net_params = NetParams(hn_json.get("net_params", {}))
        self._layers = {}
        self._hn = hn_json
        self._validate_hn()
        self._add_layers()
        self._add_connections()
        self._hailo_nn.set_names_and_indices()
        self._update_input_indices()
        self._hailo_nn.update_output_indices()
        self._hailo_nn.calculate_shapes()
        self._hailo_nn.validate_shapes()
        self._hailo_nn.update_output_layers_order()
        self._hailo_nn.update_io_layers_transposed()
        self._validate_net_params()

        if not self._hailo_nn.net_params.net_scopes:
            self._hailo_nn.add_scopes()

        return self._hailo_nn

    def from_hn(self, hn_json):
        return self.from_parsed_hn(json.loads(ensure_str(hn_json)))

    def _validate_input_identifiers(self, hn_json):
        # validate net name
        old_net_name = hn_json.get("name", None)
        old_net_num_unique_layers = len(hn_json["layers"])
        net_name = HailoNN.get_valid_input_identifier(old_net_name, "net_name")

        # validate scope names
        net_params = hn_json.get("net_params", None)
        old_net_scopes = net_params.get("net_scopes", None) if net_params else None
        new_net_scopes = {}
        if old_net_scopes:
            for scope in old_net_scopes:
                validated_scope = HailoNN.get_valid_input_identifier(scope, "scope")
                if scope != validated_scope:
                    new_net_scopes[scope] = validated_scope

        hn = {}
        validated_hn_json_str = json.dumps(update_nested(hn, hn_json))
        net_name_modified = old_net_name != net_name
        scope_name_modified = len(new_net_scopes) > 0

        if net_name_modified:
            validated_hn_json_str = validated_hn_json_str.replace(old_net_name, net_name)

        if scope_name_modified:
            for old_scope, new_scope in new_net_scopes.items():
                validated_hn_json_str = validated_hn_json_str.replace(old_scope, new_scope)

        # validate same number of unique layers after modification
        if net_name_modified or scope_name_modified:
            hn_json = json.loads(ensure_str(validated_hn_json_str))
            new_net_num_unique_layers = len(hn_json["layers"])
            if new_net_num_unique_layers != old_net_num_unique_layers:
                raise UnsupportedModelError(f"New net_name {net_name} contradicts existing layers/scope names")

        return hn_json, net_name

    def _add_layers(self):
        for layer_name, layer_hn in self._hn["layers"].items():
            layer_parsed = self.create_layer(layer_hn, layer_name)
            self._layers[layer_name] = (layer_hn, layer_parsed)
            self._hailo_nn.add_node(layer_parsed)

    @classmethod
    def create_layer(cls, layer_hn, layer_name):
        layer_hn["name"] = layer_name
        if layer_hn["type"] not in cls.TYPE_TO_CLASS:
            raise UnsupportedModelError(
                f"'{layer_hn['name']}' has an unexpected layer type '{layer_hn['type']}'. "
                f"Supported layers: {cls.TYPE_TO_CLASS.keys()}",
            )
        layer_parsed = cls.TYPE_TO_CLASS[layer_hn["type"]].from_hn(layer_hn)
        if "defuse_params" in layer_hn:
            layer_parsed.set_defuse_params(**layer_hn["defuse_params"])
        if "compilation_params" in layer_hn:
            layer_parsed.set_compilation_params(**layer_hn["compilation_params"])
        if "quantization_params" in layer_hn:
            ignored_quantization_params = {}
            for key, value in layer_hn["quantization_params"].items():
                if key in LayerPrecisionConfig.keys():
                    setattr(layer_parsed.precision_config, key, value)
                elif key in LayerTranslationConfig.keys():
                    # TODO: remove translation config
                    setattr(layer_parsed.translation_config, key, value)
                else:
                    ignored_quantization_params[key] = value
            layer_parsed.set_legacy_precision_config(ignored_quantization_params)
        return layer_parsed

    def _add_connections(self):
        for _, layer_parsed in self._layers.values():
            number_of_inputs = len(layer_parsed.inputs)
            input_layers_parsed = [self._layers[layer_parsed.inputs[i]][1] for i in range(number_of_inputs)]
            for i in range(number_of_inputs):
                self._hailo_nn.add_edge(self._layers[layer_parsed.inputs[i]][1], layer_parsed)
                if layer_parsed.op in [
                    LayerType.concat,
                    LayerType.output_mux,
                    LayerType.bbox_decoder,
                    LayerType.proposal_generator,
                    LayerType.ew_add,
                    LayerType.ew_sub,
                    LayerType.ew_mult,
                    LayerType.ew_div,
                    LayerType.fused_bbox_decoder,
                    LayerType.ew_max,
                    LayerType.ew_min,
                ]:
                    layer_parsed.append_to_input_list(self._layers[layer_parsed.inputs[i]][1])
            if number_of_inputs >= 2:
                if (
                    number_of_inputs == 2
                    and not layer_parsed.dynamic_weights
                    and layer_parsed.op in [LayerType.conv, LayerType.dw, LayerType.normalization, LayerType.batch_norm]
                ):
                    layer_parsed.add_ew_connection(input_layers_parsed[1])
                elif not (
                    (
                        number_of_inputs == 2
                        and layer_parsed.op
                        in [
                            LayerType.ew_add,
                            LayerType.ew_sub,
                            LayerType.ew_mult,
                            LayerType.ew_div,
                            LayerType.output_layer,
                            LayerType.bbox_decoder,
                            LayerType.dw,
                            LayerType.proposal_generator,
                            LayerType.matmul,
                            LayerType.fused_bbox_decoder,
                            LayerType.ew_max,
                            LayerType.ew_min,
                            LayerType.softmax,
                        ]
                    )
                    or (
                        number_of_inputs >= 2
                        and layer_parsed.op in [LayerType.concat, LayerType.output_mux, LayerType.postprocess]
                    )
                ):
                    raise UnsupportedModelError(f"Unexpected inputs at {layer_parsed.full_name_msg}")

    def _validate_input_output(self, layer, layer_name, field, other_field):
        for other_layer_name in layer[field]:
            if other_layer_name not in self._hn["layers"]:
                raise InvalidHNError(f"{field} named {other_layer_name} of layer {layer_name} is not found")
            if layer_name not in self._hn["layers"][other_layer_name][other_field]:
                raise InvalidHNError(
                    f"Layer {other_layer_name} is {field} of layer {layer_name}, but layer "
                    f"{layer_name} is not {other_field} of layer {other_layer_name}",
                )

    def _validate_hn(self):
        # TODO: Remove input shape validation, it's checked in the jsonschema (SDK-9764)
        for layer_name, layer in self._hn["layers"].items():
            if "input_shapes" in layer and "input_shape" in layer:
                raise InvalidHNError(
                    f"Layer {layer_name} has both 'input_shapes' and 'input_shape', which can't be used together",
                )
            if "input_shapes" not in layer and "input_shape" not in layer:
                raise InvalidHNError(f"Missing required field 'input_shapes'/'input_shape' in layer {layer_name}")
        for layer_name, layer in self._hn["layers"].items():
            self._validate_input_output(layer, layer_name, "input", "output")
            self._validate_input_output(layer, layer_name, "output", "input")

    def _update_input_indices(self):
        for _, layer in self._layers.values():
            for input_layer_name in layer.inputs:
                input_layer = self._layers[input_layer_name][1]
                input_index = input_layer.index
                if input_index not in layer.input_indices:
                    layer.input_indices.append(input_index)

    def _validate_net_params(self):
        # Validate that all the output name exists and are indeed outputs
        for output_layer_name in self._hailo_nn.net_params.output_layers_order:
            if output_layer_name not in [layer.name for layer in self._hailo_nn]:
                raise InvalidHNError(
                    f"Output layer {output_layer_name} in net_params.output_layers_order could not be "
                    f"found in the graph!",
                )
            layer = self._hailo_nn.get_layer_by_name(output_layer_name)
            if not any(
                output_layer.op in [LayerType.output_mux, LayerType.output_layer, LayerType.external_output_layer]
                for output_layer in self._hailo_nn.successors(layer)
            ):
                raise InvalidHNError(
                    f"Output layer {output_layer_name} in net_params.output_layers_order is not an output!",
                )

        # Validate that if net scopes exist, then all appear in the HN, and no other scopes exist in it
        if self._hailo_nn.net_params.net_scopes:
            found_scopes = set()
            for old_layer_name in self._layers:
                layer_name = self._layers[old_layer_name][1].name
                split_name = layer_name.split("/")
                if len(split_name) != 2:
                    raise InvalidHNError(f"Layer name {old_layer_name} is illegal. Each layer must have one scope")
                found_scopes.add(split_name[0])
            if found_scopes != set(self._hailo_nn.net_params.net_scopes):
                raise InvalidHNError(
                    f"Mismatch between network scopes in net_params and the layers in the HN. These "
                    f"scopes were found in the HN layers: {found_scopes}, but the scopes specified in "
                    f"the net_params are: {self._hailo_nn.net_params.net_scopes}",
                )


class HailoNNExporter:
    def __init__(self, hailo_nn):
        self._hailo_nn = hailo_nn


class HNExporter(HailoNNExporter):
    def __init__(self, hailo_nn):
        super().__init__(hailo_nn)
        self._hn = OrderedDict()

    def get_hn(self, network_name, json_dump=True, should_get_default_params=False):
        self._reset_hn(network_name, self._hailo_nn.net_params.to_json())
        self._hailo_nn.set_names_and_indices()
        self._hailo_nn.update_output_indices()
        layers = list(self._hailo_nn)
        layers.sort(key=attrgetter("index"))
        for current_layer in layers:
            self._hn["layers"][str(current_layer.name)] = current_layer.to_hn(should_get_default_params)
        for current_layer in layers:
            self._set_layer_io(current_layer)
        if json_dump:
            return prettify_json(json.dumps(self._hn, indent=4))
        return self._hn

    def get_params_npz(self, network_name):
        result = {}
        if len(self._hn) == 0:
            self.get_hn(network_name)

        for curr_layer in self._hailo_nn:
            layer_name = curr_layer.name

            if curr_layer.op in [
                LayerType.conv,
                LayerType.deconv,
                LayerType.dw,
                LayerType.maxpool,
                LayerType.avgpool,
                LayerType.external_pad,
            ]:
                result[hn_to_npz_key(layer_name, "padding_const_value")] = curr_layer.padding_const_value

            if curr_layer.op == LayerType.layer_normalization:
                result[hn_to_npz_key(layer_name, "epsilon")] = curr_layer.epsilon

            if (
                curr_layer.op == LayerType.feature_multiplier
                and curr_layer.feature_multiplier_type != FeatureMultiplierType.square
            ):
                result[hn_to_npz_key(layer_name, "power_table")] = curr_layer.power_table

            if curr_layer.op == LayerType.const_input:
                result[hn_to_npz_key(layer_name, "const_data")] = curr_layer.const_values

            if (
                curr_layer.op
                in [LayerType.conv, LayerType.dw, LayerType.deconv, LayerType.dense, LayerType.normalization]
                and not curr_layer.dynamic_weights
            ):
                result[hn_to_npz_key(layer_name, "kernel")] = curr_layer.kernel
                result[hn_to_npz_key(layer_name, "bias")] = curr_layer.bias
            if curr_layer.op == LayerType.softmax and curr_layer.additive_mask is not None:
                result[hn_to_npz_key(layer_name, "additive_mask")] = curr_layer.additive_mask
            if curr_layer.op in [
                LayerType.conv,
                LayerType.dw,
                LayerType.deconv,
                LayerType.dense,
                LayerType.batch_norm,
                LayerType.normalization,
                LayerType.activation,
                LayerType.ew_add,
                LayerType.ew_sub,
                LayerType.ew_mult,
                LayerType.avgpool,
                LayerType.reduce_sum,
            ]:
                # storing activation parameters in npz, if they exist in curr_layer
                for attr in ACTIVATION_ATTRS:
                    if hasattr(curr_layer, attr) and getattr(curr_layer, attr) is not None:
                        result[hn_to_npz_key(layer_name, attr)] = getattr(curr_layer, attr)

            if (
                curr_layer.op in [LayerType.conv, LayerType.dw, LayerType.deconv, LayerType.dense, LayerType.batch_norm]
                and curr_layer.bn_info is not None
            ):
                bn_info = curr_layer.bn_info
                result[hn_to_npz_key(layer_name, "gamma")] = bn_info.gamma
                result[hn_to_npz_key(layer_name, "beta")] = bn_info.beta
                result[hn_to_npz_key(layer_name, "moving_mean")] = bn_info.moving_mean
                result[hn_to_npz_key(layer_name, "moving_variance")] = bn_info.moving_variance
                result[hn_to_npz_key(layer_name, "epsilon")] = bn_info.epsilon

        if len(result) > 0:  # Params are not empty
            result["params_kind"] = [0]  # ParamsKinds.NATIVE

        return result

    def save_params_npz(self, network_name, path):
        result = self.get_params_npz(network_name)
        hailo_np_savez(path, **result)

    def _reset_hn(self, network_name, net_params):
        self._hn = OrderedDict()
        self._hn["name"] = network_name
        self._hn["net_params"] = net_params
        self._hn["layers"] = OrderedDict()

    def _set_layer_io(self, layer):
        layer_hn = self._hn["layers"][str(layer.name)]
        layer_hn["input"] = self._get_hn_io(self._hailo_nn.predecessors(layer))
        layer_hn["output"] = self._get_hn_io(self._hailo_nn.successors(layer))

    def _get_hn_io(self, layer_io):
        return [layer_io_item.name for layer_io_item in layer_io]
