#!/usr/bin/env python
from enum import Enum

from hailo_model_optimization.acceleras.utils.acceleras_definitions import (  # noqa: F401
    ColorConversionType,
    ColorConversionTypes,
    EmulationSupportedConversions,
    FormatConversionType,
    HybridConversionType,
    HybridConversionTypes,
    InputConversions,
)


# Note: Enums decorated with `@auto_generate_schema` are used in HN schemas!
# To allow for HN backwards compatibility:
# * Do not remove keys from the enums, rather append new keys.
# * Do not change the string values.
def auto_generate_schema(enumeration):
    """
    Enums decorated with this decorator will be added to the autogenerated
    hn_definitions.schema.auto.json file.
    """
    enumeration.auto_generate_schema = True
    return enumeration


@auto_generate_schema
class HnVersion(Enum):
    V1_0 = "1.0"


class InputDataType(Enum):
    uint8 = "uint8"


@auto_generate_schema
class HnStage(Enum):
    """The current build stage of this HN"""

    HN = "HN"
    PRE_FUSED = "PRE_FUSED"
    DEFUSED = "DEFUSED"
    EXPANDED = "EXPANDED"
    SPLIT = "SPLIT"


@auto_generate_schema
class ActivationType(Enum):
    """Enum-like class for layers activations"""

    linear = "linear"
    relu = "relu"
    relu6 = "relu6"
    leaky = "leaky"
    elu = "elu"
    sigmoid = "sigmoid"
    exp = "exp"
    tanh = "tanh"
    threshold = "threshold"
    biased_delta = "biased_delta"
    prelu = "prelu"
    softplus = "softplus"
    silu = "silu"
    gelu = "gelu"
    mish = "mish"
    inv_pos = "inv_pos"
    minus_inv_pos = "minus_inv_pos"
    hardswish = "hardswish"
    swish = "swish"
    sqrt = "sqrt"
    relu1 = "relu1"
    less = "less"
    log = "log"
    hardsigmoid = "hardsigmoid"
    clip = "clip"
    inv_sqrt = "inv_sqrt"
    softsign = "softsign"
    delta = "delta"
    greater = "greater"
    pow = "pow"
    hdr_compression = "hdr_compression"
    relu_positive_square = "relu_positive_square"
    pwl = "pwl"
    exp_decompose = "exp_decompose"
    shift = "shift"


ActivationTypes = {x.value: x for x in ActivationType}


@auto_generate_schema
class LayerType(str, Enum):
    """Enum-like class for layers types"""

    # Cluster
    input_layer = "input_layer"
    output_layer = "output_layer"
    external_input_layer = "external_input_layer"
    external_output_layer = "external_output_layer"
    const_input = "const_input"
    base_conv = "base_conv"
    conv = "conv"
    CONV_DECOMPOSE = "conv_decompose"
    base_dw = "base_dw"
    dw = "dw"
    base_dense = "base_dense"
    dense = "dense"
    l2_normalization = "l2_normalization"
    maxpool = "maxpool"
    avgpool = "avgpool"
    concat = "concat"
    output_mux = "output_mux"
    global_avg_pool = "global_avg_pool"
    base_activation = "base_activation"
    activation = "activation"
    base_batch_norm = "base_batch_norm"
    batch_norm = "batch_norm"
    loopback = "loopback"
    shortcut = "shortcut"
    external_pad = "external_pad"
    resize = "resize"
    nms = "nms"
    portal = "portal"
    feature_interleave = "feature_interleave"
    base_deconv = "base_deconv"
    deconv = "deconv"
    depth_to_space = "depth_to_space"
    normalization = "normalization"
    argmax = "argmax"
    feature_shuffle = "feature_shuffle"
    bbox_decoder = "bbox_decoder"
    format_conversion = "format_conversion"
    softmax = "softmax"
    proposal_generator = "proposal_generator"
    feature_splitter = "feature_splitter"
    base_slice = "base_slice"
    slice = "slice"
    merged_layer = "merged_layer"
    bias_add = "bias_add"
    row_splitter = "row_splitter"
    width_splitter = "width_splitter"
    reduce_max = "reduce_max"
    reduce_min = "reduce_min"
    space_to_depth = "space_to_depth"
    demux = "demux"
    matmul = "matmul"
    reduce_sum = "reduce_sum"
    feature_multiplier = "feature_multiplier"
    null = "null"
    reduce_l2 = "reduce_l2"
    instance_normalization = "instance_normalization"
    reduce_sum_square = "reduce_sum_square"
    transpose = "transpose"
    postprocess = "postprocess"
    fused_bbox_decoder = "fused_bbox_decoder"
    reduce_mean = "reduce_mean"
    equal = "equal"
    layer_normalization = "layer_normalization"
    precision_splitter = "precision_splitter"
    precision_splitter_signed = "precision_splitter_signed"
    rnn = "rnn"
    lstm = "lstm"
    gru = "gru"
    one_hot = "one_hot"
    log_softmax = "log_softmax"
    spatial_splitter = "spatial_splitter"
    tile = "tile"
    scatter_nd = "scatter_nd"
    einsum = "einsum"
    # EW Operators
    base_ew_add = "base_ew_add"
    base_ew_add_n = "base_ew_add_n"
    base_ew_mean = "base_ew_mean"
    ew_add = "ew_add"
    base_ew_sub = "base_ew_sub"
    ew_sub = "ew_sub"
    ew_mult = "ew_mult"
    ew_div = "ew_div"
    ew_max = "ew_max"
    ew_min = "ew_min"
    # PrePost
    pp_input_layer = "pp_input_layer"
    pp_output_layer = "pp_output_layer"


LayerTypes = {x.value: x for x in LayerType}


@auto_generate_schema
class PaddingType(Enum):
    """Enum-like class for padding LayerTypes"""

    same = "SAME"
    valid = "VALID"
    same_tensorflow = "SAME_TENSORFLOW"
    deconv = "DECONV"


PaddingTypes = {x.value: x for x in PaddingType}


class TemporaryPaddingType(Enum):
    """Enum-like class for temporary padding LayerTypes"""

    external_undecided = "EXTERNAL_UNDECIDED"
    same_lower = "SAME_LOWER"
    conv3d = "CONV3D"


TemporaryPaddingTypes = {x.value: x for x in TemporaryPaddingType}


@auto_generate_schema
class DefuseType(Enum):
    """Enum-like class for defused LayerTypes"""

    none = "NONE"
    normal = "NORMAL"
    deconv = "DECONV"
    super_deconv = "SUPER_DECONV"
    maxpool = "MAXPOOL"
    spatial_w = "SPATIAL_W"
    compute_lanes = "COMPUTE_LANES"
    proposal_generator = "PROPOSAL_GENERATOR_ANCHORS"
    nms = "NMS"
    portal_l4 = "PORTAL_L4"
    portal_ddr = "PORTAL_DDR"
    resize = "RESIZE"
    input_features = "INPUT_FEATURES"
    super_conv = "SUPER_CONV"
    arch_required = "ARCH_REQUIRED"
    depth_to_space = "DEPTH_TO_SPACE"
    space_to_depth = "SPACE_TO_DEPTH"
    resize_transpose = "RESIZE_TRANSPOSE"
    super_dw = "SUPER_DW"
    const_input = "CONST_INPUT"
    global_avgpool_transposed_input = "GLOBAL_AVGPOOL_TRANSPOSED_INPUT"
    spatial_reshape = "SPATIAL_RESHAPE"
    double_precision_conv = "DOUBLE_PRECISION_CONV"
    nv = "NV"
    i420 = "I420"
    dynamic_weights = "DYNAMIC_WEIGHTS"
    l3_portal = "L3_PORTAL"
    ew_mult_on_mac = "EW_MULT_ON_MAC"
    conv_weight_groups = "CONV_WEIGHT_GROUPS"


DefuseTypes = {x.value: x for x in DefuseType}


@auto_generate_schema
class ResourcesAllocationStrategy(Enum):
    """Enum-like class for resources search method during allocation"""

    min_scs_add_mem = "min_scs_add_mem"
    min_scs_match_fps = "min_scs_match_fps"
    min_l3_mem_match_fps = "min_l3_mem_match_fps"
    manual_scs_selection = "manual_scs_selection"
    automatic_scs_selection = "automatic_scs_selection"


ResourcesAllocationStrategies = {x.value: x for x in ResourcesAllocationStrategy}


@auto_generate_schema
class ResizeMethod(Enum):
    """Enum-like class for methods supported by 'resize' LayerType"""

    bilinear = "bilinear"
    nearest_neighbor = "nearest_neighbor"


ResizeMethods = {x.value: x for x in ResizeMethod}


@auto_generate_schema
class SpaceToDepthType(Enum):
    """Enum-like class for Types of space_to_depth by 'SpaceToDepth' LayerType"""

    classic_dcr = "classic_dcr"
    classic_crd = "classic_crd"
    focus = "focus"
    serial = "serial"


SpaceToDepthTypes = {x.value: x for x in SpaceToDepthType}


@auto_generate_schema
class DepthToSpaceType(Enum):
    """Enum-like class for Types of depth_to_space by 'DepthToSpace' LayerType"""

    dcr = "dcr"
    crd = "crd"


DepthToSpaceTypes = {x.value: x for x in DepthToSpaceType}


@auto_generate_schema
class HWLayerType(Enum):
    """Enum-like class for hw layer associated with a given Layer (hn_layers)"""

    lcu = "lcu"
    ppu = "ppu"


HWLayerTypes = {x.value: x for x in HWLayerType}


@auto_generate_schema
class Subclusters16x4Policy(Enum):
    """Enum-like class for subclusters ordering policy during allocation"""

    allowed = "allowed"
    enabled = "enabled"
    disabled = "disabled"


Subclusters16x4Policies = {x.value: x for x in Subclusters16x4Policy}


@auto_generate_schema
class UseL2WeightsPolicy(Enum):
    """Enum-like class for using L2 for weights"""

    allowed = "allowed"
    enabled = "enabled"
    disabled = "disabled"


UseL2WeightsPolicies = {x.value: x for x in UseL2WeightsPolicy}


@auto_generate_schema
class PrecisionSplitMode(Enum):
    """Enum-like class for using precision split mode"""

    NORMAL = "normal"
    PIXELS = "pixels"


PrecisionSplitModes = {x.value: x for x in PrecisionSplitMode}


@auto_generate_schema
class CompressL3WeightsPolicy(Enum):
    """Enum-like class for using compression in L3 weights"""

    allowed = "allowed"
    enabled = "enabled"
    disabled = "disabled"


CompressL3WeightsPolicies = {x.value: x for x in CompressL3WeightsPolicy}


@auto_generate_schema
class EnableEwAddOptimizationPolicy(Enum):
    """Enum-like class for ew_add optimization policy for conv kernels"""

    allowed = "allowed"
    enabled = "enabled"
    disabled = "disabled"


EnableEwAddOptimizationPolicies = {x.value: x for x in EnableEwAddOptimizationPolicy}


@auto_generate_schema
class ParallelActivationPolicy(Enum):
    """Enum-like class for parallel activation policy for conv kernels"""

    allowed = "allowed"
    enabled = "enabled"
    disabled = "disabled"


ParallelActivationPolicies = {x.value: x for x in ParallelActivationPolicy}


@auto_generate_schema
class BalanceOutputMultisplitPolicy(Enum):
    """Enum-like class for balancing buffers of layer with multiple outputs"""

    allowed = "allowed"
    enabled = "enabled"
    disabled = "disabled"


BalanceOutputMultisplitPolicies = {x.value: x for x in BalanceOutputMultisplitPolicy}


@auto_generate_schema
class SubclustersNoContextsPolicy(Enum):
    """Enum-like class for subclusters ordering policy during allocation"""

    allowed = "allowed"
    enabled = "enabled"
    disabled = "disabled"


SubclustersNoContextsPolicies = {x.value: x for x in SubclustersNoContextsPolicy}


@auto_generate_schema
class SC4bitWeightsPolicy(Enum):
    """Enum-like class for subclusters 4-bit weights"""

    enabled = "enabled"
    disabled = "disabled"


SC4bitWeightsPolicies = {x.value: x for x in SC4bitWeightsPolicy}


@auto_generate_schema
class SCExponentialMode4bitWeightsPolicy(Enum):
    """Enum-like class for subclusters exp mode 4-bit weights"""

    enabled = "enabled"
    disabled = "disabled"


SCExponentialMode4bitWeightsPolicies = {x.value: x for x in SCExponentialMode4bitWeightsPolicy}


@auto_generate_schema
class ResizeBilinearStreamingPolicy(Enum):
    """Enum-like class for resize bilinear streaming hw new feature in b0 ordering policy during allocation"""

    allowed = "allowed"
    enabled = "enabled"
    disabled = "disabled"


ResizeBilinearStreamingPolicies = {x.value: x for x in ResizeBilinearStreamingPolicy}


@auto_generate_schema
class TwoLineBufferModePolicy(Enum):
    """Enum-like class for resize bilinear streaming hw new feature in b0 ordering policy during allocation"""

    allowed = "allowed"
    enabled = "enabled"
    disabled = "disabled"


TwoLineBufferModePolicies = {x.value: x for x in TwoLineBufferModePolicy}


@auto_generate_schema
class ResizeBilinearPixelsMode(Enum):
    """Enum-like class for resize bilinear pixel coordinates transformation mode. Currently supporting basic PyTorch and TF variants."""

    disabled = "disabled"
    align_corners = "align_corners"
    half_pixels = "half_pixels"


ResizeBilinearPixelsModes = {x.value: x for x in ResizeBilinearPixelsMode}


@auto_generate_schema
class FeatureMultiplierType(Enum):
    """Enum-like class for feature multiplier working mode."""

    user_specified = "user_specified"
    square = "square"
    yolov5 = "yolov5"


FeatureMultiplierTypes = {x.value: x for x in FeatureMultiplierType}


@auto_generate_schema
class NormalizationType(Enum):
    """Enum-like class for normalization layer type: real normalization or standalone mul, sub, or add by const nodes."""

    normalization = "normalization"
    mul_and_add = "mul_and_add"


class KoPolicy(Enum):
    ALLOWED = "allowed"
    ENABLED = "enabled"
    DISABLED = "disabled"


class LayerBuffersFormat(Enum):
    """Policy for layer's I/O buffers format"""

    consecutive = "consecutive"
    row_per_cut = "row_per_cut"
    automatic = "automatic"
    divide_by_rows = "divide_by_rows"
    divide_by_words = "divide_by_words"
    offset = "offset"


class UseSequencerPolicy(Enum):
    """Enum-like class for the different modes for using sequencer in the CAT"""

    allowed = "allowed"
    enabled = "enabled"
    disabled = "disabled"


class ShouldAlignCcwsSection(Enum):
    allowed = "allowed"
    enabled = "enabled"
    disabled = "disabled"


class EnableLcuFromSequencerPolicy(Enum):
    """Enum-like class for the different modes for enabling lcu sequencer in the CAT"""

    enabled = "enabled"
    disabled = "disabled"


class ParamsLoadTimeCompressionPolicy(Enum):
    """Enum-like class for the different modes for compressing params in the CAT"""

    allowed = "allowed"
    enabled = "enabled"
    disabled = "disabled"


def policies_dict(policy):
    return {x.value: x for x in policy}


class NMSMetaArchitectures(str, Enum):
    """Network meta architectures to which on-chip/ on-host post-processing can be added."""

    #: Single Shot Detection meta architecture.
    SSD = "ssd"

    #: Centernet meta architecture
    CENTERNET = "centernet"

    #: Yolov5 meta architecture
    YOLOV5 = "yolov5"

    #: Yolox meta architecture
    YOLOX = "yolox"

    #: Yolov5 seg meta architecture
    YOLOV5_SEG = "yolov5_seg"

    #: Yolov6 meta architecture
    YOLOV6 = "yolov6"

    #: Yolov8 meta architecture
    YOLOV8 = "yolov8"

    #: Damoyolo meta architecture
    DAMOYOLO = "damoyolo"


class BlockType(str, Enum):
    """Enum-like class for block types"""

    RNN = "rnn"
    LSTM = "lstm"
    GRU = "gru"
    LOG_SOFTMAX = "log_softmax"
    L2_NORMALIZATION = "l2_normalization"
    EW_MEAN = "ew_mean"
    ONE_HOT = "one_hot"
    INSTANCE_NORMALIZATION = "instance_normalization"
    EW_MAX = "ew_max"
    REDUCE_MIN = "reduce_min"
    PRELU = "prelu"


class ShapeSplitterType(Enum):
    """Enum-like class for shape splitter type"""

    SPLIT_HEIGHT = "SPLIT_HEIGHT"
    SPLIT_WIDTH = "SPLIT_WIDTH"
    SPLIT_FEATURES = "SPLIT_FEATURES"
