import copy

import numpy as np

from hailo_model_optimization.acceleras.utils.acceleras_definitions import (
    EquivClassification,
    LayerHandlerType,
    LayerSupportStatus,
)
from hailo_sdk_common.hailo_nn.exceptions import UnsupportedModelError
from hailo_sdk_common.hailo_nn.hn_definitions import LayerType
from hailo_sdk_common.hailo_nn.hn_layers.bbox_decoder import BboxDecoderLayer
from hailo_sdk_common.hailo_nn.hn_layers.nms import NMSLayer


class FusedBboxDecoderLayer(BboxDecoderLayer):
    def __init__(self):
        super().__init__()
        self._op = LayerType.fused_bbox_decoder
        self._number_of_inputs_supported = 2
        self._proposals_per_output = NMSLayer.BBOX_PER_CHUNK
        self._number_of_coordinates_per_proposal = NMSLayer.BBOX_PARAMETERS - 1
        self._values_per_proposal = NMSLayer.BBOX_PARAMETERS
        self._input_division_factor = 1

    @property
    def input_division_factor(self):
        return self._input_division_factor

    @input_division_factor.setter
    def input_division_factor(self, input_division_factor):
        self._input_division_factor = input_division_factor

    @property
    def proposals_per_output(self):
        return self._proposals_per_output

    @property
    def input_features(self):
        return self._get_shape_single_dim(self._input_shapes, 3, validate=False)

    @property
    def number_of_coordinates_per_proposal(self):
        return self._number_of_coordinates_per_proposal

    @property
    def values_per_proposal(self):
        return self._values_per_proposal

    def to_pb(self, pb_wrapper, is_multi_scope):
        node = super(BboxDecoderLayer, self).to_pb(pb_wrapper, is_multi_scope)
        node.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_NETWORK_FUSED_BBOX_DECODER
        node.input_division_factor = self._input_division_factor
        return node

    @classmethod
    def from_pb(cls, pb, pb_wrapper):
        layer = super(BboxDecoderLayer, cls).from_pb(pb, pb_wrapper)
        layer._input_division_factor = pb.input_division_factor
        return layer

    @classmethod
    def from_hn(cls, hn):
        layer = super(BboxDecoderLayer, cls).from_hn(hn)
        if "params" in hn and "input_division_factor" in hn["params"]:
            layer._input_division_factor = hn["params"]["input_division_factor"]
        return layer

    def to_hn(self, should_get_default_params=False):
        result = copy.deepcopy(super(BboxDecoderLayer, self).to_hn(should_get_default_params))
        if self._input_division_factor:
            result["params"]["input_division_factor"] = self._input_division_factor
        return result

    def _calc_output_shape(self):
        if len(self._input_shapes) != 2:
            raise UnsupportedModelError(f"Invalid number of input shapes for {self.full_name_msg}.")
        boxes_shape = self._input_shapes[0]
        classes_shape = self._input_shapes[1]
        classes = int(classes_shape[3])
        if boxes_shape[1] != classes_shape[1] or boxes_shape[2] != classes_shape[2]:
            raise UnsupportedModelError(f"Invalid input shapes for {self.full_name_msg}.")
        if boxes_shape[1] % self._input_division_factor != 0:
            raise UnsupportedModelError(f"Invalid input_division_factor for {self.full_name_msg}.")
        if "defuse_input_width" in self.defuse_params and self.defuse_input_width != 0:
            input_width = self.defuse_input_width
        else:
            input_width = boxes_shape[2]

        padded_width = int(np.ceil(input_width / self._proposals_per_output) * self._proposals_per_output)
        total_proposals = boxes_shape[1] * padded_width
        prop_gen_outputs = int(total_proposals / self._proposals_per_output)
        return [
            boxes_shape[0],
            classes * self._input_division_factor,
            self._number_of_coordinates_per_proposal,
            (prop_gen_outputs * self._values_per_proposal) // self._input_division_factor,
        ]

    def get_equalization_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.unsupported, is_source=False)

    def get_params_sorter_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.unsupported, is_source=False)

    def get_dead_channels_removal_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.unsupported, is_source=False)

    def ibc_supported(self):
        return LayerSupportStatus.unsupported

    @property
    def finetune_supported(self):
        return False
