import copy

from hailo_model_optimization.acceleras.utils.acceleras_definitions import LayerHandlerType, LayerSupportStatus
from hailo_sdk_common.hailo_nn.exceptions import UnsupportedModelError
from hailo_sdk_common.hailo_nn.hn_definitions import LayerType
from hailo_sdk_common.hailo_nn.hn_layers.layer_with_params import LayerWithParams
from hailo_sdk_common.hailo_nn.layer_equiv_set import EquivClassification


class PrecisionSplitterSignedLayer(LayerWithParams):
    _REQUIRES_NATIVE_WEIGHTS = False
    _REQUIRES_QUANTIZED_WEIGHTS = True
    _IS_REAL_LAYER = False

    def __init__(self):
        super().__init__()
        self._op = LayerType.precision_splitter_signed
        self._groups = 1

    @classmethod
    def create(cls, original_name, input_vertex_order, output_shapes=None, groups=1):
        layer = super().create(original_name, input_vertex_order, output_shapes=output_shapes)
        layer.groups = groups

        return layer

    @classmethod
    def from_pb(cls, pb, pb_wrapper):
        layer = super().from_pb(pb, pb_wrapper)
        layer.groups = pb.groups
        return layer

    @classmethod
    def from_layer(cls, old_layer):
        layer = super().from_layer(old_layer)
        layer.groups = old_layer.groups
        return layer

    def to_pb(self, pb_wrapper, is_multi_scope):
        node = super().to_pb(pb_wrapper, is_multi_scope)
        node.groups = self._groups
        return node

    @classmethod
    def from_hn(cls, hn):
        layer = super().from_hn(hn, validate_params_exist=False)
        if "params" in hn:
            if "groups" in hn["params"]:
                layer._groups = hn["params"]["groups"]
                if layer.input_shape[-1] % layer._groups != 0:
                    raise UnsupportedModelError(
                        f"input features must be a multiply of groups for {layer.full_name_msg}",
                    )
        return layer

    def to_hn(self, should_get_default_params=False):
        result = copy.deepcopy(super().to_hn(should_get_default_params))
        result["params"]["groups"] = self._groups
        return result

    def _calc_output_shape(self):
        output_shape = [self.input_shape[0], self.input_shape[1], self.input_shape[2], 2 * self.input_shape[-1]]
        return output_shape

    def move_params(self, layer):
        super().move_params(layer)
        if hasattr(layer, "groups"):
            self.groups = layer.groups

    @property
    def groups(self):
        return self._groups

    @groups.setter
    def groups(self, groups):
        self._groups = groups

    def get_equalization_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.unsupported, is_source=False)

    def get_params_sorter_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.unsupported, is_source=False)

    def get_dead_channels_removal_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.unsupported, is_source=False)

    def ibc_supported(self):
        return LayerSupportStatus.unsupported
