import copy

from hailo_model_optimization.acceleras.utils.acceleras_definitions import LayerHandlerType, LayerSupportStatus
from hailo_sdk_common.hailo_nn.exceptions import UnsupportedModelError
from hailo_sdk_common.hailo_nn.hn_definitions import DefuseType, LayerType
from hailo_sdk_common.hailo_nn.hn_layers.layer_with_params import LayerWithParams
from hailo_sdk_common.hailo_nn.layer_equiv_set import EquivClassification


class ReduceMaxLayer(LayerWithParams):
    _REQUIRES_NATIVE_WEIGHTS = False
    _REQUIRES_QUANTIZED_WEIGHTS = False
    _IS_REAL_LAYER = True

    def __init__(self):
        super().__init__()
        self._op = LayerType.reduce_max
        self._groups = 1
        self._reduce_axes = [3]

    @classmethod
    def create(cls, original_name, input_vertex_order, output_shapes=None, groups=1):
        layer = super().create(original_name, input_vertex_order, output_shapes=output_shapes)
        layer.groups = groups

        return layer

    @classmethod
    def from_pb(cls, pb, pb_wrapper):
        layer = super().from_pb(pb, pb_wrapper)
        layer.groups = pb.groups
        layer.reduce_axes = pb.reduce_axes
        return layer

    @classmethod
    def from_layer(cls, old_layer):
        layer = super().from_layer(old_layer)
        layer.groups = old_layer.groups
        layer.reduce_axes = old_layer.reduce_axes
        return layer

    def to_pb(self, pb_wrapper, is_multi_scope):
        node = super().to_pb(pb_wrapper, is_multi_scope)
        node.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_NETWORK_REDUCE_MAX
        node.groups = self._groups
        node.reduce_axes.extend(self._reduce_axes)
        return node

    @classmethod
    def from_hn(cls, hn):
        layer = super().from_hn(hn, validate_params_exist=False)
        if "params" in hn:
            if "groups" in hn["params"]:
                layer._groups = hn["params"]["groups"]
                if layer.input_shape[-1] % layer._groups != 0:
                    raise UnsupportedModelError(
                        f"input features must be a multiply of groups for {layer.full_name_msg}",
                    )
            reduce_axes = hn["params"].get("reduce_axes", [3])
            layer.reduce_axes = reduce_axes
        return layer

    def to_hn(self, should_get_default_params=False):
        result = copy.deepcopy(super().to_hn(should_get_default_params))
        result["params"]["groups"] = self._groups
        result["params"]["reduce_axes"] = self._reduce_axes
        return result

    def _calc_output_shape(self):
        output_shape = [dim if i not in self.reduce_axes else self._groups for i, dim in enumerate(self.input_shape)]
        if "defuse_input_width" in self.defuse_params and self.defuse_input_width != 0:
            output_shape[2] = self.defuse_input_width
        if self.defuse_params.get("defuse_input_features", 0) != 0 and self.defuse_params.get("defuse_groups", 0) != 0:
            output_shape[3] = self.defuse_params["defuse_groups"]
        return output_shape

    def move_params(self, layer):
        super().move_params(layer)
        if hasattr(layer, "groups"):
            self.groups = layer.groups

    @property
    def input_width(self):
        if self.defuse_type == DefuseType.spatial_w:
            return self.defuse_input_width
        return super().input_width

    @property
    def groups(self):
        return self._groups

    @groups.setter
    def groups(self, groups):
        self._groups = groups

    @property
    def reduce_axes(self):
        return self._reduce_axes

    @reduce_axes.setter
    def reduce_axes(self, reduce_axes):
        self._reduce_axes = [axis if axis > 0 else axis + len(self.input_shape) for axis in reduce_axes]

    def get_equalization_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.unsupported, is_source=False)

    def get_params_sorter_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.unsupported, is_source=False)

    def get_dead_channels_removal_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.unsupported, is_source=False)

    def ibc_supported(self):
        return LayerSupportStatus.unsupported
