import numpy as np

from hailo_model_optimization.acceleras.utils.acceleras_definitions import LayerHandlerType, LayerSupportStatus
from hailo_sdk_common.hailo_nn.hn_definitions import LayerType, PaddingType
from hailo_sdk_common.hailo_nn.hn_layers.layer_with_params import LayerWithParams
from hailo_sdk_common.hailo_nn.layer_equiv_set import EquivClassification


class ReduceSumSquareLayer(LayerWithParams):
    _REQUIRES_NATIVE_WEIGHTS = False
    _REQUIRES_QUANTIZED_WEIGHTS = True
    _IS_REAL_LAYER = True

    def __init__(self):
        super().__init__()
        self._op = LayerType.reduce_sum_square
        self._reduce_axes = [3]
        self._groups = 1

    @classmethod
    def create(cls, original_name, input_vertex_order, reduce_axes, groups=1, output_shapes=None):
        layer = super().create(original_name, input_vertex_order, output_shapes)
        layer.reduce_axes = reduce_axes
        layer.groups = groups
        return layer

    def get_width_kernel(self):
        width_dim = len(self.input_shape) - 2
        return self.input_shape[2] if width_dim in self.reduce_axes else 1

    @property
    def strides(self):
        return [1, 1, self.get_width_kernel(), 1]

    @property
    def macs(self):
        # The /2 is because we don't do multiply
        return self.ops / 2

    @property
    def ops(self):
        # A little trick that simplifies it all
        total_input_size = float(np.abs(np.prod(np.array(self.input_shape))))
        # Each input is being summed one time, and squared one time
        return total_input_size * 2

    @property
    def kernel_shape(self):
        features_dim = len(self.input_shape) - 1
        features_output_dim = self.groups if features_dim in self.reduce_axes else self.input_shape[features_dim]

        return [1, self.get_width_kernel(), int(self.input_shape[features_dim] / self.groups), features_output_dim]

    @property
    def padding(self):
        return PaddingType.same

    @property
    def groups(self):
        return self._groups

    @groups.setter
    def groups(self, groups):
        self._groups = groups

    def _calc_output_shape(self):
        if len(self.input_shape) == 2:
            return [self.input_shape[0], self._groups]

        output_shape = self.input_shape
        for axis in self.reduce_axes:
            output_shape[axis] = 1

        if output_shape[-1] == 1:
            output_shape[-1] = self._groups

        return output_shape

    def set_input_shapes(self, input_shapes, validate=True):
        super().set_input_shapes(input_shapes, validate)
        if len(self.input_shape) == 2:
            self._reduce_axes = [1]

    @property
    def reduce_axes(self):
        return self._reduce_axes

    @reduce_axes.setter
    def reduce_axes(self, reduce_axes):
        self._reduce_axes = reduce_axes

    def get_equalization_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.unsupported, is_source=False)

    def get_params_sorter_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.unsupported, is_source=False)

    def get_dead_channels_removal_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.unsupported, is_source=False)

    def ibc_supported(self):
        return LayerSupportStatus.unsupported
