import copy

import numpy as np

from hailo_model_optimization.acceleras.utils.acceleras_definitions import LayerHandlerType, LayerSupportStatus
from hailo_sdk_common.hailo_nn.exceptions import UnsupportedModelError
from hailo_sdk_common.hailo_nn.hn_definitions import LayerType
from hailo_sdk_common.hailo_nn.hn_layers.layer_with_params import LayerWithParams
from hailo_sdk_common.hailo_nn.layer_equiv_set import EquivClassification


class RowSplitterLayer(LayerWithParams):
    _REQUIRES_NATIVE_WEIGHTS = False
    _REQUIRES_QUANTIZED_WEIGHTS = False
    _IS_REAL_LAYER = True

    def __init__(self):
        super().__init__()
        self._op = LayerType.row_splitter
        self._sequential_row_split = False

    @classmethod
    def create(cls, original_name, input_vertex_order, split_sizes, output_shapes=None, sequential_row_split=False):
        layer = super().create(original_name, input_vertex_order, output_shapes)
        layer._sequential_row_split = sequential_row_split
        return layer

    @classmethod
    def from_pb(cls, pb, pb_wrapper):
        if not cls.output_shapes:
            raise UnsupportedModelError("layer Row Splitter requires output shapes")
        layer = super().from_pb(pb, pb_wrapper)
        layer._sequential_row_split = pb.sequential_row_split
        return layer

    @classmethod
    def from_hn(cls, hn):
        layer = super().from_hn(hn, validate_params_exist=False)
        if not cls.output_shapes:
            raise UnsupportedModelError(f"{layer.full_name_msg} requires output shapes")
        if "params" in hn and "sequential_row_split" in hn["params"]:
            layer._sequential_row_split = hn["params"]["sequential_row_split"]

        return layer

    def to_hn(self, should_get_default_params=False):
        result = copy.deepcopy(super().to_hn(should_get_default_params))
        result["params"]["sequential_row_split"] = self._sequential_row_split
        return result

    def to_pb(self, pb_wrapper, is_multi_scope):
        node = super().to_pb(pb_wrapper, is_multi_scope)
        node.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_NETWORK_ROW_SPLITTER
        node.sequential_row_split = self._sequential_row_split
        return node

    @property
    def sequential_row_split(self):
        return self._sequential_row_split

    def update_output_shapes(self, **kwargs):
        if not self._validate_output_shapes():
            raise UnsupportedModelError(
                f"Unexpected split shapes at {self.full_name_msg}, "
                f"output_shapes={self.output_shapes}, input_shapes={self.input_shapes})",
            )
        # Overrided because len(output_shapes)>1 but output_copies == 1
        self.output_shapes = self._calc_output_shape()

    def _validate_output_shapes(self):
        if sum([shape[1] for shape in self.output_shapes]) != self.input_shapes[0][1]:
            return False
        return True

    def _calc_output_shape(self):
        output_shapes = []
        splits_num = len(self.output_shapes)
        input_height = self.input_shape[1]
        remainder = input_height % splits_num
        split_base_height = int(np.floor(input_height / splits_num))
        for _ in self.output_shapes:
            output_height = split_base_height + (1 if remainder else 0)
            output_shapes.append([-1, output_height, self.input_shape[2], self.input_shape[3]])
            remainder = max([remainder - 1, 0])
        return output_shapes

    def _get_output_shape(self, validate=False, layer_name=None, layer_index=None):
        if layer_name is None:
            raise UnsupportedModelError(f"{self.full_name_msg} successor name is missing, output shape is ambiguous")
        if len(self._output_indices) > 0:
            if layer_index is None:
                raise UnsupportedModelError(
                    f"{self.full_name_msg} successor index is missing, output shape is ambiguous",
                )
            return self._output_shapes[self._output_indices.index(layer_index)]
        return self._output_shapes[self.outputs.index(layer_name)]

    def sort_outputs(self):
        return lambda layer1, layer2: 1 if self.outputs.index(layer1.name) > self.outputs.index(layer2.name) else -1

    def get_equalization_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.transparent, is_source=False)

    def get_params_sorter_handler_type(self, predecessor=None):
        # TODO: maybe it is not unsupported
        return EquivClassification(LayerHandlerType.transparent, is_source=False)

    def get_dead_channels_removal_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.transparent, is_source=False)

    def ibc_supported(self):
        return LayerSupportStatus.unsupported
