import copy

import numpy as np

from hailo_model_optimization.acceleras.utils.acceleras_definitions import LayerHandlerType, LayerSupportStatus
from hailo_sdk_common.hailo_nn.exceptions import UnsupportedModelError
from hailo_sdk_common.hailo_nn.hn_definitions import LayerType
from hailo_sdk_common.hailo_nn.hn_layers import FeatureSplitterLayer
from hailo_sdk_common.hailo_nn.layer_equiv_set import EquivClassification


class WidthSplitterLayer(FeatureSplitterLayer):
    _REQUIRES_NATIVE_WEIGHTS = False
    _REQUIRES_QUANTIZED_WEIGHTS = False
    _IS_REAL_LAYER = True

    def __init__(self):
        super().__init__()
        self._op = LayerType.width_splitter
        self._sequential_row_split = True

    @classmethod
    def create(cls, original_name, input_vertex_order, split_sizes, output_shapes=None):
        return super().create(original_name, input_vertex_order, split_sizes, output_shapes=output_shapes, groups=1)

    @classmethod
    def from_pb(cls, pb, pb_wrapper):
        if not cls.output_shapes:
            raise UnsupportedModelError("layer Width Splitter requires output shapes")
        return super().from_pb(pb, pb_wrapper)

    @classmethod
    def from_hn(cls, hn):
        layer = super().from_hn(hn)
        if not cls.output_shapes:
            raise UnsupportedModelError(f"{layer.full_name_msg} requires output shapes")
        return layer

    def to_hn(self, should_get_default_params=False):
        return copy.deepcopy(super().to_hn(should_get_default_params))

    def to_pb(self, pb_wrapper, is_multi_scope):
        node = super().to_pb(pb_wrapper, is_multi_scope)
        node.type = pb_wrapper.integrated_hw_graph_base_pb2.PROTO_NETWORK_WIDTH_SPLITTER
        return node

    def update_output_shapes(self, **kwargs):
        if not self._validate_output_shapes():
            raise UnsupportedModelError(
                f"Unexpected split shapes at {self.full_name_msg}, "
                f"output_shapes={self.output_shapes}, input_shapes={self.input_shapes})",
            )
        # Overrided because len(output_shapes)>1 but output_copies == 1
        self.output_shapes = self._calc_output_shape()

    def _validate_output_shapes(self):
        return sum([shape[2] for shape in self.output_shapes]) == self.input_shapes[0][2]

    def _calc_output_shape(self):
        output_shapes = []
        splits_num = len(self.output_shapes)
        input_width = self.input_shape[2]
        split_width = int(np.ceil(input_width / splits_num / 8) * 8)
        remainder = input_width - split_width * (splits_num - 1)
        last_split_width = remainder if remainder else split_width
        for _ in range(splits_num - 1):
            output_shapes.append([-1, self.input_shape[1], split_width, self.input_shape[3]])
        output_shapes.append([-1, self.input_shape[1], last_split_width, self.input_shape[3]])
        return output_shapes

    def _get_output_shape(self, validate=False, layer_name=None, layer_index=None):
        if layer_name is None:
            raise UnsupportedModelError(f"{self.full_name_msg} successor name is missing, output shape is ambiguous")
        if len(self._output_indices) > 0:
            if layer_index is None:
                raise UnsupportedModelError(
                    f"{self.full_name_msg} successor index is missing, output shape is ambiguous",
                )
            return self._output_shapes[self._output_indices.index(layer_index)]
        return self._output_shapes[self.outputs.index(layer_name)]

    def sort_outputs(self):
        return lambda layer1, layer2: 1 if self.outputs.index(layer1.name) > self.outputs.index(layer2.name) else -1

    def get_equalization_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.transparent, is_source=False)

    def get_params_sorter_handler_type(self, predecessor=None):
        # TODO: maybe it is not unsupported
        return EquivClassification(LayerHandlerType.transparent, is_source=False)

    def get_dead_channels_removal_handler_type(self, predecessor=None):
        return EquivClassification(LayerHandlerType.transparent, is_source=False)

    def ibc_supported(self):
        return LayerSupportStatus.unsupported

    def set_input_shapes(self, input_shapes, validate=True):
        super().set_input_shapes(input_shapes, validate)
        if self._split_sizes:
            inp = self.input_shapes[0]
            output_shapes = []
            if self._split_indices and len(self._split_indices) >= len(self._split_sizes):
                for index in self._split_indices:
                    output_shapes.append([-1, inp[1], self._split_sizes[index], inp[3]])
            else:
                for split_size in self._split_sizes:
                    output_shapes.append([-1, inp[1], split_size, inp[3]])
            self.output_shapes = output_shapes
