#!/usr/bin/env python
__all__ = ["default_logger", "set_default_logger_level", "create_custom_logger", "DeprecationVersion", "DeprecatedAPI"]

import logging
import os
import sys
from datetime import date
from enum import Enum

import verboselogs

from hailo_model_optimization.acceleras.utils.logger import (
    DEFAULT_LEVEL,
    DEFAULT_LOG_DIR,
    DFC_FOLDER_PATH,
    LOG_LEVEL_ENV_VAR,
    SDK_CLIENT_LOG_DIR_ENV_VAR,
    SDK_CLIENT_LOG_EN_ENV_VAR,
    has_append_permission,
)

SDK_CLIENT_LOG_NAME = "hailo_sdk.client"
DFC_LOG_FILE_PATH = os.path.join(DFC_FOLDER_PATH, f"{SDK_CLIENT_LOG_NAME}.log")

IMPORTANT = logging.INFO + 1
logging.addLevelName(IMPORTANT, "IMPORTANT")
DEPRECATION_WARNING = logging.WARNING - 1
logging.addLevelName(DEPRECATION_WARNING, "DEPRECATION_WARNING")
COMMAND = logging.INFO - 1
logging.addLevelName(COMMAND, "COMMAND")


class DeprecatedAPI(Exception):
    pass


class DeprecationVersion(Enum):
    JAN2022 = date(2022, 1, 1)
    APR2022 = date(2022, 4, 1)
    JUN2022 = date(2022, 6, 1)
    AUG2022 = date(2022, 8, 1)
    JAN2023 = date(2023, 1, 1)
    APR2023 = date(2023, 4, 1)
    JUL2023 = date(2023, 7, 1)
    OCT2023 = date(2023, 10, 1)
    JAN2024 = date(2024, 1, 1)
    APR2024 = date(2024, 4, 1)
    JUL2024 = date(2024, 7, 1)
    OCT2024 = date(2024, 10, 1)
    JAN2025 = date(2025, 1, 1)
    APR2025 = date(2025, 4, 1)
    JUL2025 = date(2025, 7, 1)
    FUTURE = date(9999, 12, 31)


def important(self, message, *args, **kws):
    if self.isEnabledFor(IMPORTANT):
        self._log(IMPORTANT, message, args, **kws)


def deprecation_warning(self, message, dep_version: DeprecationVersion = DeprecationVersion.FUTURE, *args, **kws):
    if self.isEnabledFor(DEPRECATION_WARNING):
        self._log(DEPRECATION_WARNING, message, args, **kws)
        # if dep_version != DeprecationVersion.FUTURE:
        #     dep_date = dep_version.value
        #     dep_date = f"{dep_date.year:04}-{dep_date.month:02}"
        #     self._log(DEPRECATION_WARNING, f"Planned to be deprecated in {dep_date}", args)
    if dep_version.value <= date(2022, 1, 31):
        raise DeprecatedAPI("You are using deprecated API, please review the preceding message")


def command(self, message, *args, **kws):
    if self.isEnabledFor(COMMAND):
        self._log(COMMAND, message, args, **kws)


logging.Logger.important = important
logging.Logger.deprecation_warning = deprecation_warning
logging.Logger.command = command


class ConsoleFormatter(logging.Formatter):
    """
    Logging Formatter for the console.
    """

    COLOR_PREFIX = "\x1b["
    COLOR_SUFFIX = "\x1b[0m"
    COLORS = {
        logging.DEBUG: "36m",  # blue
        logging.INFO: "32m",  # green
        logging.WARNING: "33;1m",  # bold yellow
        logging.ERROR: "31;1m",  # bold red
        logging.CRITICAL: "41;1m",  # bold white on red
        IMPORTANT: "32;1m",  # bold green
        DEPRECATION_WARNING: "33;21m",  # yellow
    }

    def format(self, record):
        if record.levelno == IMPORTANT:
            level_name = logging.getLevelName(logging.INFO)
            level_no = logging.INFO
            if sys.stdout.isatty():
                message = f"{self.COLOR_PREFIX}{self.COLORS[record.levelno]}%(message)s{self.COLOR_SUFFIX}"
            else:
                message = "%(message)s"
        elif record.levelno == DEPRECATION_WARNING:
            level_name = logging.getLevelName(logging.WARNING)
            level_no = logging.WARNING
            if sys.stdout.isatty():
                message = f"{self.COLOR_PREFIX}{self.COLORS[record.levelno]}DEPRECATION WARNING: %(message)s{self.COLOR_SUFFIX}"
            else:
                message = "DEPRECATION WARNING: %(message)s"
        elif record.levelno == COMMAND:
            level_name = logging.getLevelName(logging.DEBUG)
            level_no = logging.DEBUG
            message = "%(message)s"
        else:
            level_name = record.levelname
            level_no = record.levelno
            message = "%(message)s"
        if sys.stdout.isatty():
            level_fmt = f"[{self.COLOR_PREFIX}{self.COLORS[level_no]}{level_name.lower()}{self.COLOR_SUFFIX}]"
        else:
            level_fmt = f"[{level_name.lower()}]"
        formatter = logging.Formatter(f"{level_fmt} {message}")
        return formatter.format(record)


def _create_logger(level=None, logger_path=None, fmt=None, console=True):
    """
    Creates a logging object and returns it
    """
    logger_path = logger_path if logger_path else os.path.join(DEFAULT_LOG_DIR, SDK_CLIENT_LOG_NAME + ".log")
    logger_name, _ = os.path.splitext(os.path.basename(logger_path))
    logger = verboselogs.VerboseLogger(logger_name)
    logger.propagate = False

    console_level = DEFAULT_LEVEL
    file_level = level if level is not None else DEFAULT_LEVEL

    logger.setLevel(file_level)

    if fmt is None:
        fmt = "%(asctime)s - %(levelname)s - %(filename)s:%(lineno)s - %(message)s"
    formatter = logging.Formatter(fmt)

    if console:
        console_fh = logging.StreamHandler(sys.stdout)
        console_fh.setFormatter(ConsoleFormatter())
        console_fh.set_name("stream_handler")
        console_fh.setLevel(console_level)
        logger.addHandler(console_fh)

    enable_log_file = os.environ.get(SDK_CLIENT_LOG_EN_ENV_VAR)

    if not enable_log_file or enable_log_file.lower() != "false":
        # creates directories (if needed) for the log file that will be stored in hailo/dfc folder
        os.makedirs(DFC_FOLDER_PATH, exist_ok=True)
        logger_folder = os.path.dirname(logger_path)
        if logger_folder:
            os.makedirs(logger_folder, exist_ok=True)
        # create the logging file handlers
        for logger_file_path in [logger_path, DFC_LOG_FILE_PATH]:
            if has_append_permission(logger_file_path):
                fh = logging.FileHandler(logger_file_path, delay=True)
                fh.setFormatter(formatter)
                logger.addHandler(fh)
            elif logger_file_path != DFC_LOG_FILE_PATH:
                logger.warning(f"Unable to create log file in {logger_file_path} (Permission denied)")

    return logger


_g_logger = None


def default_logger():
    global _g_logger
    if _g_logger is None:
        sdk_client_log_dir = os.environ.get(SDK_CLIENT_LOG_DIR_ENV_VAR)
        logger_path = os.path.join(sdk_client_log_dir, SDK_CLIENT_LOG_NAME + ".log") if sdk_client_log_dir else None
        log_level = os.environ.get(LOG_LEVEL_ENV_VAR, DEFAULT_LEVEL).upper()
        _g_logger = _create_logger(level=log_level, logger_path=logger_path)
    return _g_logger


def create_custom_logger(log_path, fmt=None, console=False):
    """Returns a logger to the specified path, creating it if needed."""
    log_level = os.environ.get(LOG_LEVEL_ENV_VAR, DEFAULT_LEVEL).upper()
    return _create_logger(level=log_level, logger_path=log_path, fmt=fmt, console=console)


def set_default_logger_level(level):
    """
    Set log level of the default log.
    Should only be called from initialization code to avoid log level collisions.
    """
    if level == "ERROR":
        _g_logger.verbose(f"Logger verbosity has been decreased to {level.lower()}s only ")
    else:
        _g_logger.verbose(f"Logger verbosity is: {level}")
    _g_logger.setLevel(level)
