#!/usr/bin/env python
import copy
import os
import tempfile
from importlib.util import find_spec

from future.utils import with_metaclass

from hailo_sdk_common.logger.logger import default_logger

logger = default_logger()


class ConfigStageNotSetException(Exception):
    pass


class PackagingException(Exception):
    pass


class Singleton(type):
    _instances = {}

    def __call__(cls, *args, **kwargs):
        if cls not in cls._instances:
            cls._instances[cls] = super().__call__(*args, **kwargs)
        return cls._instances[cls]


class SDKPaths(with_metaclass(Singleton, object)):
    HAILO_TEMP_DIR_PREFIX = "hailo"
    DEFAULT_BUILD_DIR = "build"

    def __init__(self):
        self._build_dir = type(self).DEFAULT_BUILD_DIR
        self._custom_build_dir = None
        self._has_graphviz = True

        hailo_sdk_common = find_spec("hailo_sdk_common")
        self._is_release = "site-packages" in os.path.dirname(os.path.dirname(hailo_sdk_common.origin))
        self._is_internal = find_spec("hailo_sdk_internals") is not None

        if self._is_release:
            self._build_dir = tempfile.mkdtemp(prefix=type(self).HAILO_TEMP_DIR_PREFIX)

    @property
    def is_release(self):
        return self._is_release

    @property
    def is_internal(self):
        return self._is_internal

    @property
    def build_dir(self):
        if self._is_internal and self._custom_build_dir:
            return self._custom_build_dir

        return self.join_sdk(self._build_dir)

    def set_client_build_dir_path(self):
        if not self.is_release:
            self._build_dir = os.path.join("../sdk_client", type(self).DEFAULT_BUILD_DIR)

    @property
    def custom_build_dir(self):
        return self._custom_build_dir

    @custom_build_dir.setter
    def custom_build_dir(self, custom_build_dir):
        self._custom_build_dir = custom_build_dir

    @property
    def has_graphviz(self):
        return self._has_graphviz

    @has_graphviz.setter
    def has_graphviz(self, val):
        self._has_graphviz = val

    def join_sdk(self, path):
        if self.is_release:
            return path

        hailo_sdk_server = find_spec("hailo_sdk_server")
        return os.path.join(os.path.dirname(os.path.dirname(hailo_sdk_server.origin)), path)

    @staticmethod
    def join_sdk_common(path):
        hailo_sdk_common = find_spec("hailo_sdk_common")
        return os.path.join(os.path.dirname(hailo_sdk_common.origin), path)

    def join_sdk_repo(self):
        return os.path.join(self.join_sdk("."), "../")

    @staticmethod
    def join_sdk_client(path):
        hailo_sdk_client = find_spec("hailo_sdk_client")
        return os.path.join(os.path.dirname(hailo_sdk_client.origin), path)

    def join_build_sdk(self, path):
        if os.path.exists(self.build_dir):
            return os.path.join(self.build_dir, path)

        logger.debug(f"Creating build dir : {self.build_dir}")
        os.makedirs(self.build_dir)
        return os.path.join(self.build_dir, path)

    def join_hailo_tools_path(self, path):
        if self.is_internal:
            hailo_sdk_internals = find_spec("hailo_sdk_internals")
            package_dir = os.path.dirname(os.path.dirname(os.path.dirname(hailo_sdk_internals.origin)))
            return os.path.join(package_dir, "sdk_server", "hailo_tools", path)

        if self.is_release:
            hailo_sdk_common = find_spec("hailo_sdk_common")
            return os.path.join(os.path.dirname(os.path.dirname(hailo_sdk_common.origin)), "hailo_tools", path)


class BaseConfigDirs:
    DIRS_BUILD_ONLY = {
        "outputs_dir": ["outputs"],
        "bin_dir": ["bin"],
        "weights_dir": ["data", "weights"],
        "inputs_dir": ["data", "inputs"],
    }

    DIRS_SDK_ONLY = {}

    DIRS_BOTH = {}

    def __init__(self, hw_arch):
        if hasattr(hw_arch, "name"):
            self._hw_arch = hw_arch.name
        else:
            self._hw_arch = hw_arch
        self._paths = SDKPaths()
        self._dirs = {}
        for d in [type(self).DIRS_BUILD_ONLY, type(self).DIRS_SDK_ONLY, type(self).DIRS_BOTH]:
            self._dirs.update(self._format_dirs(d))

    def get_dir(self, name, in_build=True):
        return self._join_base(self._dirs[name], in_build)

    def _format_dirs(self, input_dirs):
        result = {}
        for name, dir_path in input_dirs.items():
            result[name] = os.path.join(*dir_path).format(hw_arch=self._hw_arch)
        return result

    def _join_base(self, path, in_build=True):
        base_dir = self._paths.build_dir if in_build else "microcode"
        whole_path = os.path.join(base_dir, path)
        return self._paths.join_sdk(whole_path)

    @property
    def build_dirs_keys(self):
        return list(type(self).DIRS_BUILD_ONLY.keys()) + list(type(self).DIRS_BOTH.keys())


class BaseConfigPaths(BaseConfigDirs):
    PATHS = {
        "bin": ["{bin_dir}", "{network_name}.mem"],
    }

    def __init__(self, hw_arch, model_name):
        super().__init__(hw_arch)
        self._model_name = model_name
        self._stage = None
        self._stage_only = False

    def set_stage(self, stage, stage_only=False):
        self._stage = stage
        self._stage_only = stage_only

    def get_path(self, path_name, in_build=True):
        template = os.path.join(*type(self).PATHS[path_name])
        if self._stage is None and "{network_name}" in template:
            raise ConfigStageNotSetException("Set the stage before trying to get paths")
        format_dict = copy.deepcopy(self._dirs)
        format_dict["model_name"] = self._model_name
        if self._stage_only:
            format_dict["network_name"] = self._stage
        else:
            format_dict["network_name"] = f"{self._model_name}.{self._stage}"
        config_path = template.format(**format_dict)
        return self._join_base(config_path, in_build)
