#!/usr/bin/env python

import io

import numpy as np

from hailo_sdk_common.logger.logger import DeprecationVersion, default_logger


def serialize_numpy(data):
    """
    Serialize multiple numpy arrays

    Args:
        data (dict[str, np.ndarray] | np.ndarray): Either a dict of array name to array object, or a single array

    Returns:
        str: the serialized string

    """
    mem_file = io.BytesIO()
    if isinstance(data, dict):
        hailo_np_savez(mem_file, **data)
    else:
        hailo_np_save(mem_file, data)
    mem_file.seek(0)
    return mem_file.read()


def deserialize_numpy(data):
    mem_file = io.BytesIO()
    mem_file.write(data)
    mem_file.seek(0)
    return hailo_np_load(mem_file)


def is_containing_none(item):
    """
    This method check if there is a None item within a sequence

    Args:
        item: Any object to test for None

    Return:
        True if any item contains None, otherwise False.

    """
    if isinstance(item, np.ndarray):
        if item.dtype == object:
            return is_containing_none(item.tolist())
        else:
            return False

    if isinstance(item, dict):
        # TODO: SDK-10099
        return is_containing_none(list(item.values()))

    if isinstance(item, (set, list, tuple)):
        for i in item:
            inner_check = is_containing_none(i)
            if inner_check:
                return True
        return False

    return item is None


def hailo_np_load(file, mmap_mode=None, allow_pickle=False, fix_imports=True, encoding="bytes"):
    """
    Load numpy arrays from npz file.

    Returns
        dict[str, np.ndarray]: a dictionary of array name -> array object

    """
    logger = default_logger()
    try:
        loaded_npz = np.load(
            file,
            mmap_mode=mmap_mode,
            allow_pickle=allow_pickle,
            fix_imports=fix_imports,
            encoding=encoding,
        )
        return _convert_to_dict(loaded_npz)
    except ValueError:
        # Backwards compatibility for legacy npz files in tests
        # TODO: remove this when all npz files have been updated (SDK-9844)
        if not isinstance(file, str):
            # Reseek the file if it's a stateful file object
            file.seek(0)
        loaded_npz = np.load(file, mmap_mode=mmap_mode, allow_pickle=True, fix_imports=fix_imports, encoding=encoding)
        # TODO: https://hailotech.atlassian.net/browse/SDK-9844
        logger.deprecation_warning(
            "Loaded legacy npz file. Legacy files use pickle, they are deprecated and not secure!",
            DeprecationVersion.FUTURE,
        )
        return _convert_to_dict(loaded_npz)


def _convert_to_dict(loaded_npz):
    return dict(loaded_npz) if isinstance(loaded_npz, np.lib.npyio.NpzFile) else loaded_npz.tolist()


def hailo_np_savez(file, *args, **kwds):
    # TODO: SDK-10099
    if is_containing_none(args) or is_containing_none(list(kwds.values())):
        default_logger().warning("np.ndarray contains Nones, so Numpy saves it using pickle")
    np.savez(file, *args, **kwds)


def hailo_np_savez_compressed(file, *args, **kwds):
    # TODO: SDK-10099
    if is_containing_none(args) or is_containing_none(list(kwds.values())):
        default_logger().warning("np.ndarray contains Nones, so Numpy saves it using pickle")
    np.savez_compressed(file, *args, **kwds)


def hailo_np_save(file, *args, **kwds):
    # TODO: SDK-10099
    if is_containing_none(args) or is_containing_none(list(kwds.values())):
        default_logger().warning("np.ndarray contains Nones, so Numpy saves it using pickle")
    np.save(file, *args, **kwds)
